#!/usr/bin/perl

=encoding UTF-8
=cut

=head1 DESCRIPTION

    При параметре --tier=1
    Скрипт архивирует блоки для логинов из файла PI-13927_tier1_to_archiving.csv
    Пейджи, у которых заархивированы все блоки в файле  tier1_pages.csv

    При параметре --tier=2
    Скрипт архивирует блоки по page_id из файла PI-13927_tier2_to_archiving.csv
    Пейджи, у которых заархивированы все блоки в файле  tier2_pages.csv

=head1 PARAMS

 tier  - тип файла для архивации блоков
 file_path  - путь до файла c параметрами для архивации блоков
 min_blocks - минимальное количество работающих блоков на пейдже для архивации
 no_debug - запуск архивации

=head1 USAGE

> ./PI-13927_archive_blocks.pl --tier=[1, 2] --file_path=./path/to/file --min_blocks=<int> --no_debug

=cut

use Carp;
use Text::CSV;
use Pod::Usage;
use Getopt::Long qw();
use POSIX qw(strftime);
use lib::abs qw(
  ../../lib
  );
use qbit;
use Application;
use Utils::CSV;
use Utils::Logger qw( INFOF );
use Date::Parse qw (str2time);

my @NON_ARCHIVED_LOGINS = qw(maximatelecom maximatelecomspb);
my $YEAR_PERIOD         = '2017-12-27';
my $HALF_YEAR_PERIOD    = '2018-06-27';
my $DEBUG               = TRUE;
my $MAX_TIME_DIFFERENCE = 90;

sub _get_args {

    my $tier             = 0;
    my $file_path        = '';
    my $min_blocks_count = undef;
    my $help             = 0;
    Getopt::Long::GetOptions(
        #--- Obligatory
        'tier=i'       => \$tier,
        'file_path=s'  => \$file_path,
        'min_blocks=i' => \$min_blocks_count,
        #---
        'no_debug' => sub {$DEBUG = FALSE},
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;
    if ($tier != 1 && $tier != 2) {
        print "tier must be 1 or 2\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }
    if (!$file_path) {
        print "specify the file path\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }
    if (!$min_blocks_count or $min_blocks_count < 0) {
        print "min_count must be greater than 0\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }
    return ($tier, $min_blocks_count, $file_path);
}

sub get_data_from_csv {
    my ($tier, $filename) = @_;
    my $file_content = readfile($filename);
    my $data = parse_csv($file_content, sep_char => "\t");

    if ($tier == 1) {
        my $processed_data = {};
        foreach my $line (@$data) {
            $line->{'login'} = fix_login($line->{'login'});
            next if (in_array($line->{'login'}, \@NON_ARCHIVED_LOGINS));
            my $period = '';
            if ($line->{'period'} =~ m/12\s+мес/) {
                $period = $YEAR_PERIOD;
            } elsif ($line->{'period'} =~ m/6\s+мес/) {
                $period = $HALF_YEAR_PERIOD;
            } elsif ($line->{'period'} =~ m/нельзя/) {
                next;
            } else {
                die 'Check periods in file!';
            }
            push @{$processed_data->{$period}}, $line->{'login'};
        }

        return $processed_data;
    } else {
        return [map {$_->{'page_id'}} @$data];
    }
}

sub get_statistics_for_page {
    my ($app, $page_id, $period, $block_accessors) = @_;
    my $stat_for_period = $app->clickhouse_db->query(dont_check_fields => {}, without_table_alias => TRUE)->select(
        table  => $app->clickhouse_db->statistics,
        fields => {
            block_id   => '',
            product_id => '',
            sum_money  => {SUM => ['partner_wo_nds']},
        },
        filter =>
          ['AND', [['page_id', '=', \$page_id], ['dt', '>', \$period], ['product_id', 'IN', \$block_accessors],]],
    )->group_by(qw(block_id product_id))->having(['sum_money', '>', \0])->order_by(qw(block_id))->get_all();
    my $res = {};
    foreach my $block (@$stat_for_period) {
        $res->{$block->{'product_id'}}->{$block->{'block_id'}} = $block->{'sum_money'};
    }
    return $res;
}

sub get_csv_file {
    my ($filename, $columns) = @_;
    my $csv = Text::CSV->new({binary => 1});
    $csv->eol("\r\n");
    open my $fh, ">:encoding(utf8)", $filename or die "$filename: $!";
    $csv->bind_columns(\(@$columns));
    $csv->print($fh, $columns);
    return $fh, $csv;
}

sub check_create_date_and_money {
    my ($create_date, $dt, $period, $money) = @_;
    if ((!$create_date || $create_date eq '0000-00-00 00:00:00') && !$dt) {
        return (FALSE, 'old block with money!') if ($money);
        return (TRUE, '-');
    }
    my $create_date_ts = str2time($create_date);
    if ($dt && $create_date ne '0000-00-00 00:00:00' && abs($create_date_ts - str2time($dt)) > $MAX_TIME_DIFFERENCE) {
        return (FALSE, 'block without create_dates are different!');
    }
    if (!$create_date || $create_date eq '0000-00-00 00:00:00') {
        $create_date = $dt;
    }

    if (str2time($create_date) < str2time($period)) {
        return (TRUE, '-');
    } else {
        return (FALSE, 'recent block');
    }
}

sub main {

    my ($tier, $min_blocks_count, $filename) = _get_args();
    # File for archived blocks log
    my $date_string         = strftime "%b_%d_%Y", localtime;
    my $csv_blocks_filename = "tier$tier" . "_$date_string.csv";
    my @columns_for_blocks  = qw(login page_id block_id);
    if ($tier == 1) {
        push(@columns_for_blocks, qw(money need_to_archive archived message));
    }

    my ($fh, $csv) = get_csv_file($csv_blocks_filename, \@columns_for_blocks);

    # File for pages that can be archived
    my $csv_pages_filename = "tier$tier" . "_pages_$date_string.csv";
    my @columns            = qw(login page_id);
    my ($fh_pg, $csv_pages) = get_csv_file($csv_pages_filename, \@columns);

    my $app = Application->new();
    $app->pre_run();
    my $tmp_rights = $app->add_all_tmp_rights();
    $app->set_cur_user({id => 0});

    my $data = get_data_from_csv($tier, $filename);

    # logins checking
    if ($tier == 1) {
        foreach my $key (keys %$data) {
            my $users = $app->users->get_all(
                fields => [qw(id login)],
                filter => {login => $data->{$key}},
            );
            if (scalar @$users != scalar @{$data->{$key}}) {
                die 'Check logins!';
            }
        }
    }

    my $pages = [];
    if ($tier == 1) {
        foreach my $key (keys %$data) {
            my $pages_chunk = $app->partner_db->query->select(
                table  => $app->partner_db->all_pages,
                fields => [qw(page_id login client_id)],
                filter =>
                  ['AND', [['login', 'IN', \$data->{$key}], ['create_date', '<', \$key], ['is_deleted', '=', \0]]],
            )->order_by('page_id')->get_all();
            map {$_->{'period'} = $key} @$pages_chunk;
            push @$pages, @$pages_chunk;
        }
    } else {
        $pages = $app->partner_db->query->select(
            table  => $app->partner_db->all_pages,
            fields => [qw(page_id login client_id)],
            filter => [
                'AND',
                [
                    ['page_id',     'IN',     \$data],
                    ['create_date', '<',      \$YEAR_PERIOD],
                    ['is_deleted',  '=',      \0],
                    ['login',       'NOT IN', \[@NON_ARCHIVED_LOGINS]]
                ]
            ],
        )->order_by('page_id')->get_all();
    }

    my $heavy_pages = $app->partner_db->query->select(
        table  => $app->partner_db->all_blocks,
        fields => {
            page_id => '',
            blocks  => {COUNT => ['id']},
        },
        filter => ['page_id', 'IN', \[map {$_->{'page_id'}} @$pages]],
    )->group_by('page_id')->having(['blocks', '>', \$min_blocks_count])->order_by('page_id')->get_all();

    $heavy_pages = [map {$_->{'page_id'}} @$heavy_pages];
    $pages = [grep {in_array($_->{'page_id'}, $heavy_pages)} @$pages];

    my $pages_block_models = $app->partner_db->query->select(
        table  => $app->partner_db->all_blocks,
        fields => ['page_id', 'model'],
        filter => ['page_id', 'IN', \$heavy_pages],
    )->distinct(TRUE)->get_all();

    my $page_id_blocks = {};
    foreach my $block (@$pages_block_models) {
        if (!exists($page_id_blocks->{$block->{'page_id'}})) {
            $page_id_blocks->{$block->{'page_id'}} = [];
        }
        push @{$page_id_blocks->{$block->{'page_id'}}}, $block->{'model'} if ($block->{'model'} !~ m/^internal/);

    }

    my $page_count = @$pages;
    INFOF "Pages: %d", $page_count;

    foreach my $i (0 .. $#$pages) {
        my $cur_page = $i + 1;

        INFOF "%d | %d", $cur_page, $page_count;
        my $page            = $pages->[$i];
        my $block_accessors = $page_id_blocks->{$page->{'page_id'}};
        my $period          = $tier == 1 ? $page->{'period'} : $YEAR_PERIOD;
        my $statistics      = get_statistics_for_page($app, $page->{'page_id'}, $period, $block_accessors);

        my $non_archived_blocks = 0;
        foreach my $block (@$block_accessors) {
            my $page_id_name = $app->$block->get_page_id_field_name();
            my $query        = $app->partner_db->query->select(
                table  => $app->partner_db->$block,
                fields => {
                    'id'          => 'id',
                    $page_id_name => $page_id_name,
                    'create_date' => 'create_date'
                },
                filter => [
                    'AND',
                    [
                        [$page_id_name => '=' => \$page->{'page_id'}],
                        ['multistate', 'IN', \$app->$block->get_multistates_by_filter('not deleted')],
                    ]
                ],
            );

            my $block_action_table = $block . '_action_log';
            $query->left_join(
                table   => $app->partner_db->$block_action_table,
                fields  => {'dt' => {MAX => ['dt']}},
                join_on => [
                    'AND',
                    [
                        ["elem_$page_id_name", '=', {$page_id_name => $app->partner_db->$block}],
                        ['elem_id',            '=', {id            => $app->partner_db->$block}],
                        ['action',             '=', \'add']
                    ]
                ]
            );

            my $blocks = $query->group_by($page_id_name, 'id')->get_all();

            foreach my $j (0 .. $#$blocks) {
                $non_archived_blocks += @$blocks;
                my $id              = $blocks->[$j]->{'id'};
                my $need_to_archive = TRUE;
                my $archived        = TRUE;
                my $money           = $statistics->{$block}->{$id};

                my ($create_date_is_ok, $message) =
                  check_create_date_and_money($blocks->[$j]->{'create_date'}, $blocks->[$j]->{'dt'}, $period, $money);

                if (!$create_date_is_ok) {
                    $need_to_archive = FALSE;
                    $archived        = FALSE;
                } elsif (exists($statistics->{$block}->{$id})) {
                    $need_to_archive = FALSE;
                    $archived        = FALSE;
                } elsif ($app->$block->can('can_action_delete')
                    && !$app->$block->can_action_delete({'id' => $id, $page_id_name => $page->{'page_id'}}))
                {
                    $need_to_archive = FALSE;
                    $archived        = FALSE;
                    $message         = 'adfox_block';
                } elsif ($id == 100500) {
                    $need_to_archive = FALSE;
                    $archived        = FALSE;
                    $message         = 'tech_block';
                } else {
                    try {
                        if (!$DEBUG) {
                            $app->$block->do_action({'id' => $id, $page_id_name => $page->{'page_id'}}, 'stop');
                        }
                    }
                    catch {
                        $message = $@->message;
                    };
                    try {
                        if (!$DEBUG) {
                            $app->$block->do_action({'id' => $id, $page_id_name => $page->{'page_id'}}, 'delete');
                        }
                    }
                    catch {
                        if ($message ne '-') {
                            $message = $@->message;
                        }
                        $archived = FALSE;
                    };
                }
                INFOF "BLOCK:\t%d\t%d\t%s", $page->{'page_id'}, $id, $archived ? 'yes' : 'no';
                if ($tier == 1) {
                    $csv->print(
                        $fh,
                        [
                            $page->{'login'}, $page->{'page_id'},
                            $id,              $money,
                            $need_to_archive ? 'yes' : 'no', $archived ? 'yes' : 'no',
                            $archived ? '-' : $message
                        ]
                    );
                } elsif ($tier == 2 && $archived) {
                    $csv->print($fh, [$page->{'login'}, $page->{'page_id'}, $id]);
                }
                $non_archived_blocks-- unless ($need_to_archive && !$archived);
            }
        }
        if ($non_archived_blocks == 0) {
            $csv_pages->print($fh_pg, [$page->{'login'}, $page->{'page_id'}]);
            INFOF "PAGE:\t%d", $page->{'page_id'};
        }
    }
    close $fh    or die "$csv_blocks_filename: $!";
    close $fh_pg or die "$csv_pages_filename: $!";
    $app->post_run();
}
main();
