#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

    Скрипт "копирует" пейджи и блоки с нескольких логинов на другой.

=head1 USAGE

  ./bin/oneshots/PI-14394_copy_rambler_pages.p --page_ids=12345,54321,..  2>&1 | tee resend_to_bk.log

=head1 OPTIONS

  page_ids - Список ID площадок через запятую (необязательный)

=for comment
    Нужно подключить прод базу.
    Прод БК
    Прод Баланс
    Прод api_utils_partner2
    В конфиге выставить stage=production

=cut

# project modules
use lib::abs qw(
  ../../lib
  );
use qbit;
use Application;
use Pod::Usage;
use Getopt::Long qw();
use Utils::Logger qw( INFO INFOF WARNF ERROR );

use Term::ANSIColor qw(colored);

my $STASH_PATH      = './bin/oneshots/PI-14394_copy_rambler_pages.json';
my $RAMBLER_LOGIN   = 'rambler-p';
my $RAMBLER_USER_ID = 120143188;

my $APP;
my $PAGE_MODELS  = [qw(context_on_site_campaign video_an_site)];
my @BLOCK_MODELS = qw(context_on_site_rtb video_an_site_inpage video_an_site_instream video_an_site_fullscreen);
my %DOMAINS;
my %TAGS;

my $STASH = {};

main();

# main
sub main {

    my ($page_ids) = _get_args();
    $APP = _get_app();

    INFO '#START';

    %DOMAINS =
      map {$_->{'domain_id'} => TRUE}
      @{$APP->partner_db->owner_site->get_all(fields => [qw(domain_id)], filter => {user_id => $RAMBLER_USER_ID})};
    INFO 'get DOMAINS from "owner_site" - OK';

    %TAGS =
      map {$_->{'tag_id'} => TRUE}
      @{$APP->partner_db->cookie_match_owners->get_all(fields => [qw(tag_id)], filter => {user_id => $RAMBLER_USER_ID})
      };
    INFO 'get TAGS from "cookie_match_owners" - OK';

    $STASH = from_json(readfile($STASH_PATH));
    my @logins = sort keys(%$STASH);

    if ($page_ids) {
        my $all_pages = $APP->all_pages->get_all(
            fields   => [qw(login)],
            filter   => {page_id => $page_ids},
            distinct => TRUE
        );
        @logins = map {$_->{login}} @$all_pages;
    }

    foreach my $login (@logins) {
        try {
            copy($login, $page_ids);
        }
        catch {
            ERROR shift->message;
        };
    }

    writefile($STASH_PATH, to_json($STASH, pretty => TRUE));

    $APP->post_run();

    INFO '#END';
}

sub copy {
    my ($login, $page_ids) = @_;

    INFO "LOGIN: $login";

    my $user = $APP->users->get_by_login($login, fields => [qw(id excluded_domains excluded_phones)])
      or throw Exception sprintf('User with login "%s" not found', $login);
    my $user_id = $user->{'id'};

    copy_domains_and_phones($user);

    my $all_pages = $APP->all_pages->get_all(
        fields => [qw(id model page_id)],
        filter => {owner_id => $user_id, model => $PAGE_MODELS, (@$page_ids ? (page_id => $page_ids) : ())},
        from_view => (@$page_ids ? FALSE : TRUE)
    );

    INFOF 'pages found - %d', scalar(@$all_pages);

    my $count_pages = 0;
    foreach my $page_info (@$all_pages) {
        my $model = $page_info->{'model'};

        my $stash = $STASH->{$login}{$model}{$page_info->{'page_id'}} //= {page_id => $page_info->{'page_id'},};

        if ($stash->{'new_page'}{'ok'}) {
            WARNF "  %d/%d. %d already copied to page_id=%d", ++$count_pages, scalar(@$all_pages),
              $page_info->{'page_id'}, $stash->{'new_page'}{'page_id'};
        } else {
            my $page_settings = $APP->$model->get($page_info->{'id'}, fields => ['*']);

            if ($page_settings->{'is_deleted'}) {
                INFOF '  %d/%d. Page "%s" deleted', ++$count_pages, scalar(@$all_pages), $page_settings->{'page_id'};
                next;
            } else {
                INFOF '  %d/%d. Start COPY PAGE page_id=%d (%s)', ++$count_pages, scalar(@$all_pages),
                  $page_info->{'page_id'}, $model;

                copy_page($stash->{'new_page'}, $model, $page_settings);
            }
        }

        my $max_block_id = 0;
        foreach my $block_model (@BLOCK_MODELS) {

            my $blocks = $APP->$block_model->get_all(
                fields => ['*'],
                filter => {page_id => $stash->{'page_id'}, multistate => 'not deleted'}
            );

            INFOF '    "%s" blocks found - %d', $block_model, scalar(@$blocks);

            my $count_blocks = 0;
            foreach my $block_settings (@$blocks) {
                my $block_stash = $stash->{'blocks'}{$block_model}{$block_settings->{'public_id'}}{'new_block'} //= {};

                if ($block_stash->{'ok'}) {
                    WARNF "      %d/%d. %s already copied to %s", ++$count_blocks, scalar(@$blocks),
                      $block_settings->{'public_id'}, $block_stash->{'public_id'};
                } else {
                    INFOF "      %d/%d. COPY BLOCK: %s", ++$count_blocks, scalar(@$blocks),
                      $block_settings->{'public_id'};
                    copy_block($block_stash, $block_model, $block_settings, $stash->{'new_page'}{'page_id'});
                }

                if ($block_settings->{'id'} > $max_block_id) {
                    $max_block_id = $block_settings->{'id'};
                }
            }
        }

        my $block_seq_db_table = $APP->$model->block_seq_db_table();
        my $pk                 = $block_seq_db_table->primary_key->[0];
        $block_seq_db_table->replace({$pk => $stash->{'new_page'}{'page_id'}, next_block_id => $max_block_id + 1});

        $APP->$model->do_action($stash->{'new_page'}{'pk'}, 'set_need_update');
    }
}

sub copy_domains_and_phones {
    my ($user) = @_;

    $APP->partner_db->user_global_excluded_phones->add_multi(
        [map {{user_id => $RAMBLER_USER_ID, phone => $_}} @{$user->{'excluded_phones'}}],
        replace => TRUE);
    INFO 'copy "user_global_excluded_phones" - OK';

    $APP->partner_db->user_global_excluded_domains->add_multi(
        [map {{user_id => $RAMBLER_USER_ID, domain => $_}} @{$user->{'excluded_domains'}}],
        replace => TRUE);
    INFO 'copy "user_global_excluded_domains" - OK';
}

sub copy_page {
    my ($stash, $model, $page_settings) = @_;

    my $add_fields = $APP->$model->get_add_fields();

    my $page = {};
    unless ($stash->{'created'}) {
        my %add_settings = hash_transform($page_settings, [keys(%$add_fields)]);

        if (exists($add_settings{'login'})) {
            $add_settings{'login'} = $RAMBLER_LOGIN;
        } elsif (exists($add_settings{'owner_id'})) {
            $add_settings{'owner_id'} = $RAMBLER_USER_ID;
        }

        if (exists($add_settings{'block_title'}) && !defined($add_settings{'block_title'})) {
            delete($add_settings{'block_title'});
        }

        if (exists($add_settings{'domain_id'}) && !$DOMAINS{$add_settings{'domain_id'}}) {
            $APP->partner_db->owner_site->add(
                {
                    domain_id   => $add_settings{'domain_id'},
                    user_id     => $RAMBLER_USER_ID,
                    link_stat   => 'https://metrika.yandex.ru/',
                    create_date => curdate(oformat => 'db_time'),
                }
            );
            INFOF ' add domain_id=%s to "owner_site"', $add_settings{'domain_id'};

            $DOMAINS{$add_settings{'domain_id'}} = TRUE;
        }

        $stash->{'pk'} = $APP->$model->add(%add_settings);

        my $page = $APP->$model->get($stash->{'pk'}, fields => [qw(page_id editable_fields)]);
        INFOF '    page copied to %d - OK', $page->{'page_id'};

        $stash->{'created'} = TRUE;
    }

    $page = $APP->$model->get($stash->{'pk'}, fields => [qw(page_id editable_fields)]) unless %$page;

    $stash->{'page_id'} = $page->{'page_id'};

    unless ($stash->{'edited'}) {
        my @editable_fields = grep {!$add_fields->{$_}} keys(%{$page->{'editable_fields'}});

        if (@editable_fields) {
            my %editable_settings = hash_transform($page_settings, \@editable_fields);

            if (exists($editable_settings{'mirrors'})) {
                $editable_settings{'mirrors'} = [
                    map {ref($_) ? $_->{'domain'} : $_}
                      grep {(ref($_) && $_->{'moderation_status'} eq 'approved') || !ref($_)}
                      @{$editable_settings{'mirrors'}}
                ];
            }

            if (exists($editable_settings{'assistants'})) {
                $editable_settings{'assistants'} =
                  [map {delete($_->{'login'}); $_} @{$editable_settings{'assistants'}}];
            }

            if (defined($editable_settings{'tag_id'}) && !$TAGS{$editable_settings{'tag_id'}}) {
                $APP->partner_db->cookie_match_owners->add(
                    {
                        tag_id  => $editable_settings{'tag_id'},
                        user_id => $RAMBLER_USER_ID,
                    }
                );

                $TAGS{$editable_settings{'tag_id'}} = TRUE;
            }

            $APP->$model->do_action($stash->{'pk'}, 'edit', %editable_settings);
            INFO '    edit page copy - OK';
        }

        $stash->{'edited'} = TRUE;
    }

    if ($model eq 'context_on_site_campaign' && !$stash->{'registered_in_balance'}) {
        $APP->$model->do_action($stash->{'pk'}, 'register_in_balance');
        INFO '    set multistate "register_in_balance" - OK';

        $stash->{'registered_in_balance'} = TRUE;
    }

    unless ($stash->{'updated_multistate'}) {
        $APP->partner_db->$model->edit($stash->{'pk'}, {multistate => $page_settings->{'multistate'}});
        INFO '    set multistate - OK';

        $stash->{'updated_multistate'} = TRUE;
    }

    $stash->{'ok'} = TRUE;
}

sub copy_block {
    my ($stash, $model, $block_settings, $page_id) = @_;

    my $page_id_field_name = $APP->$model->get_page_id_field_name();

    unless ($stash->{'created'}) {
        my $new_block_settings = {
            %$block_settings,
            $page_id_field_name => $page_id,
            page_id             => $page_id,
            create_date => curdate(oformat => 'db_time'),
            multistate  => 0,
        };

        my $public_id = $APP->$model->public_id($new_block_settings);

        $block_settings->{'bk_data'} =~ s/$block_settings->{'public_id'}/$public_id/;

        my $_stash = $APP->$model->hook_stash->init(
            mode => 'add',
            $APP->$model->hook_stash_add_fields,
            data => {opts => $new_block_settings, current => $new_block_settings}
        );

        $APP->$model->hook_preparing_fields_to_save($new_block_settings);
        $APP->$model->hook_writing_to_database($new_block_settings);
        INFOF '        block copied to %s - OK', $public_id;

        $stash->{'pk'} = $APP->$model->hook_stash->get('id');

        $stash->{'public_id'} = $public_id;

        $stash->{'created'} = TRUE;
    }

    unless ($stash->{'updated_multistate'}) {
        $APP->partner_db->$model->edit($stash->{'pk'}, {multistate => $block_settings->{'multistate'}});

        $stash->{'updated_multistate'} = TRUE;
    }

    $stash->{'ok'} = TRUE;
}

sub _get_app {

    my $app = Application->new();
    $app->pre_run();

    $app->set_option('cur_user' => {id => $RAMBLER_USER_ID});

    {
        no warnings 'redefine';
        no strict 'refs';

        *{'QBit::Application::check_rights'} = sub {TRUE};
    }

    Log::Log4perl::init(
        \q[
         log4perl.logger = INFO, Screen
         log4perl.appender.Screen = Log::Log4perl::Appender::ScreenColoredLevels
         log4perl.appender.Screen.color.WARN = cyan
         log4perl.appender.Screen.layout   = Log::Log4perl::Layout::PatternLayout::Multiline
         log4perl.appender.Screen.layout.ConversionPattern = [%d{HH:mm:ss}] %p %m%n
    ]
    );

    return $app;
}

sub _get_args {

    my $page_id_str = '';
    my $help        = 0;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'page_ids:s' => \$page_id_str,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    my @errors = ();

    my @page_ids = ();
    if ($page_id_str) {
        @page_ids = split /,/, $page_id_str;
        foreach my $page_id (@page_ids) {
            unless ($page_id =~ /^\d+$/) {
                push @errors, "Wrong page_id='$page_id'";
            }
        }
    }

    if (@errors) {
        print join("\n", map {colored('ERROR! ' . $_, 'red')} @errors), "\n\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    return \@page_ids;
}

__END__
