#!/usr/bin/perl

use feature 'say';
use lib::abs qw(
  ../../lib
  );

use qbit;
use Application;
use Text::CSV_XS;
use DDP;

use PiConstants qw($PREROLL $MIDROLL);

my $APP;
my $MIDROLL_DEFAULTS;
my $MIDROLL_DEFAULTS_BLOCK_ID = 'R-V-278914-1';

sub main {

    $APP = Application->new();
    $APP->pre_run();

    $APP->set_cur_user({id => 0});

    {
        no warnings 'redefine';
        *QBit::Application::check_rights = sub {1};
    }

    $MIDROLL_DEFAULTS = get_midroll_defaults($MIDROLL_DEFAULTS_BLOCK_ID);

    my $prerolls = $APP->video_an_site_instream->get_all(
        fields => [qw(page_id id type category_id category_path)],
        filter => [AND => [{type => $PREROLL}, {multistate => 'not deleted'}, {category_id => [1011, 1012, 1013]},]],
    );

    my $midrolls = $APP->video_an_site_instream->get_all(
        fields => [qw(page_id id type category_id start_time multistate)],
        filter => [
            AND => [
                {type        => $MIDROLL},
                {category_id => [1011, 1012, 1013]},
                {page_id => array_uniq(map {$_->{page_id}} @$prerolls)},
            ]
        ],
    );

    my %pages;

    for my $block (@$prerolls, @$midrolls) {
        $pages{$block->{page_id}}{$block->{category_id}}{$block->{type}}{$block->{id}} = $block;
    }

    for my $page_id (keys(%pages)) {
        my $page = $pages{$page_id};
        for my $category_id (keys(%$page)) {
            my $category = $page->{$category_id};

            process_category($page_id, $category_id, $category);
        }
    }

    say '#END';
}

sub get_midroll_defaults {
    my ($block_id) = @_;

    my $data = $APP->video_an_site_instream->get($block_id, fields => ['*']);

    my $add_fields = $APP->video_an_site_instream->get_add_fields();

    my %data_for_add = map {$_ => $data->{$_}} keys(%$add_fields);

    delete @data_for_add{qw(id page_id category_id category_path)};
    $data_for_add{start_time}      = 10;
    $data_for_add{max_duration}    = 120;
    $data_for_add{count_positions} = 3;

    return \%data_for_add;
}

sub process_category {
    my ($page_id, $category_id, $category) = @_;

    if ($category->{$PREROLL}) {
        use DDP;
        my @preroll_blocks = values(%{$category->{$PREROLL}});
        die sprintf("Unexpected %s preroll blocks for page %s", scalar(@preroll_blocks), $page_id)
          if @preroll_blocks > 1;
        my $preroll_block = $preroll_blocks[0];
        my ($type, $midroll_block) = find_suitable_midroll($category);
        try {
            $APP->partner_db->transaction(
                sub {
                    $APP->video_an_site_instream->do_action($preroll_block, 'delete');

                    if ($type eq 'already_working') {
                        # this page already has midroll block, so do nothing
                    } elsif ($type eq 'restore') {
                        # may fail due to midroll count limitation
                        $APP->video_an_site_instream->do_action($midroll_block, 'restore');
                    } elsif ($type eq 'create_new') {
                        my %args = (
                            (map {$_ => $preroll_block->{$_},} (qw(page_id category_id category_path))),
                            %$MIDROLL_DEFAULTS,
                        );
                        # may fail due to midroll count limitation
                        $midroll_block = {public_id => $APP->video_an_site_instream->add(%args)};
                    } else {
                        die "Unknown midroll action type: $type";
                    }
                    say to_json({type => $type, preroll_block => $preroll_block, midroll_block => $midroll_block});
                }
            );
        }
        catch {
            my ($exception) = @_;
            say to_json(
                {
                    type          => $type,
                    preroll_block => $preroll_block,
                    midroll_block => $midroll_block,
                    error         => $exception->message
                },
                canonical => 1
            );
        };
    }
}

sub find_suitable_midroll {
    my ($category) = @_;

    my ($type, $block) = ('create_new', undef);

    for my $midroll (values(%{$category->{$MIDROLL}})) {
        if ($midroll->{start_time} >= 10 && $midroll->{start_time} <= 60) {
            if ($APP->video_an_site_instream->check_multistate_flag($midroll->{'multistate'}, 'deleted')) {
                ($type, $block) = ('restore', $midroll);
            } else {
                return ('already_working', $midroll);
            }
        }
    }

    return ($type, $block);
}

main();
