#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use lib::abs qw(../../lib);

use qbit;
use Application;
use Utils::Logger qw( INFOF );

$ENV{'LAZY_LOAD'} = 1;
$| = 1;

my $DATA_DIR = './blocks_thematic_filter_data';

####
_run();
#####

sub _run {
    my $app = _get_app();

    my $res = {
        #  articles => {
        #    <article_id> => <count>,
        #    ...
        #  },
        #  articles_by_model => {
        #    <model> => {
        #      <article_id> => <count>,
        #      ...
        #    },
        #  }
        #  blocks => <count>,
        #  blocks_by_model {
        #    <model> => <count>,
        #    ...
        #  },
    };

    #### make path
    `mkdir -p $DATA_DIR`;

    #### articles_dict
    my $articles_data = {};
    my $path_to_data  = "$DATA_DIR/tns_dict_article\.json";
    if (-f $path_to_data) {
        $articles_data = from_json(readfile($path_to_data));
    } else {
        my $articles = $app->tns_dict_article->get_real_article();
        unshift @$articles,
          {
            depth => 0,
            id    => 0,
            name  => 'ROOT',
          };

        my $articles_dict = {};
        foreach my $row (@$articles) {
            $row->{depth} //= 1;
            $articles_dict->{$row->{id}} = $row;
            if (defined $row->{parent_id}) {
                my $parent = $articles_dict->{$row->{parent_id}} //= {};
                push @{$parent->{children}}, $row->{id};
                $row->{depth} = $parent->{depth} + 1;
            }
        }

        $articles_data = {
            list => $articles,
            dict => $articles_dict
        };

        writefile($path_to_data, to_json($articles_data, pretty => TRUE));
    }

    #### blocks
    my @block_with_articles = sort
      grep {$app->$_->DOES('Application::Model::Role::Has::Articles')}
      map  {@$_} $app->product_manager->get_block_model_accessors();

    my $total_blocks = 0;
    foreach my $model (@block_with_articles) {

        INFOF 'model=%s', $model;
        my $model_res = {};

        my $path_to_data = "$DATA_DIR/$model\.json";
        if (-f $path_to_data) {
            $model_res = from_json(readfile($path_to_data));
        } else {
            my $page_id_field_name = $app->$model->get_page_id_field_name();

            my $rows = $app->$model->get_all(
                fields   => [$page_id_field_name, 'id', 'data_articles'],
                order_by => [$page_id_field_name, 'id'],
            );

            INFOF '  found=%d', scalar(@$rows);

            $model_res->{total_blocks} = scalar(@$rows);
            foreach my $row (@$rows) {
                foreach my $article (@{$row->{data_articles} // []}) {
                    if ($article->{cpm} > 0) {
                        $model_res->{blocks}++;
                        $model_res->{articles}->{$article->{id}}++;
                    }
                }
            }

            writefile($path_to_data, to_json($model_res, pretty => TRUE));
        }

        $res->{total_blocks} += $model_res->{total_blocks} // 0;
        $res->{blocks}       += $model_res->{blocks}       // 0;
        map {$res->{articles}->{$_} += $model_res->{articles}->{$_}} keys %{$model_res->{articles}};

        $res->{total_blocks_by_model}->{$model} = $model_res->{total_blocks};
        $res->{blocks_by_model}->{$model} = $model_res->{blocks} // 0;

        my $total_model_articles = 0;
        map {$total_model_articles += $_} values %{$model_res->{articles}};
        $res->{articles_by_model}->{$model} = $total_model_articles;
    }

    {
        my $path_to_data = "$DATA_DIR/res\.json";
        writefile($path_to_data, to_json($res, pretty => TRUE));
    }

    #### output
    {
        my ($list, $dict) = @$articles_data{qw( list  dict )};
        foreach my $row (@$list) {
            my $article = $dict->{$row->{id}};
            my $blocks_amount = $res->{articles}->{$row->{id}} // 0;

            $article->{blocks} = $blocks_amount;
            while (my $parent = $dict->{$article->{parent_id} // '--'}) {
                $parent->{blocks_with_children} //= $parent->{blocks};
                $parent->{blocks_with_children} += $blocks_amount;
                $article = $parent;
            }
        }

        printf "total blocks=%d\n",               $res->{total_blocks};
        printf "total blocks with articles=%d\n", $res->{blocks};
        printf("%-3s %5s %-65s %-10s %s\n", '#', 'id', 'name', 'blocks', 'blocks with children');
        print '-' x 110, "\n";
        my $count = 0;
        foreach my $row (@$list) {
            my $article = $dict->{$row->{id}};
            if ($article->{blocks} || $article->{blocks_with_children}) {
                printf(
                    "%-3s %5s %-65s %-10s %s\n",
                    ++$count, $article->{id}, ('    ' x $article->{depth}) . $article->{name},
                    $article->{blocks}, $article->{blocks_with_children} // ''
                );
            }
        }
    }

    return 1;
}

sub _get_app {

    my $app;
    {
        $app = Application->new();
        $app->pre_run();

        no strict 'refs';
        no warnings 'redefine';
        *{'QBit::Application::check_rights'} = sub {TRUE};
    }

    return $app;
}
