#!/usr/bin/perl

# common modules
use feature 'say';

# project modules
use lib::abs qw(../../lib);
use qbit;
use Application;
use Utils::LogParser::Checks;
use PiConstants qw($LOGS_DIRECTORY_PATH);
use Utils::MonitoringUtils qw(send_to_graphite);
use Getopt::Long;

main();

sub main {
    my ($log_name) = @_;

    my $opts = get_opts();

    my $app = Application->new();
    $app->pre_run();
    my $tmp_rights = $app->add_all_tmp_rights();

    my $list = get_list_files($opts->{root}, $opts->{stat}, $opts->{from});
    for my $filename (@$list) {
        warn "$filename\n";
        process_logs(
            $Utils::LogParser::Checks::LOGS,
            $Utils::LogParser::Checks::CHECKS,
            $opts->{stat}, $filename, $opts->{log_only}
        );
    }

    $app->post_run();
}

sub get_list_files {
    my ($root, $stat, $days) = @_;

    my $from          = time - $days * 24 * 60 * 60;
    my @path          = split /\//, $Utils::LogParser::Checks::LOGS->{$stat}{path};
    my $file_template = pop @path;
    $root .= '/' . join('/', @path);
    my @list;
    my $dir;
    opendir $dir, $root;
    while (my $file = readdir $dir) {
        next unless $file =~ /^$file_template/;
        my $filename = "$root/$file";
        my @stat     = stat($filename);
        next if $stat[9] < $from;
        push @list, $filename;
    }
    closedir $dir;
    return \@list;
}

sub process_logs {
    my ($logs, $checks, $log_key, $log_filename, $log_only) = @_;

    $logs->{$_}{name}   //= $_ for keys %$logs;
    $checks->{$_}{name} //= $_ for keys %$checks;
    $logs->{$_}{preparse} //= sub {TRUE}
      for keys %$logs;

    my @applicable_checks = map {$checks->{$_}} (grep {check($checks->{$_}{apply_to}, $log_key)} keys %$checks);
    unless (@applicable_checks) {
        die "Skip processing $log_key: no checks are applicable";
    }

    my $log = $logs->{$log_key};
    my $fh  = seek_to_start($log_filename);

    my $checks_temp_data;
    my $timestamp = 0;
    while (my $line = readline($fh)) {
        my @line = split /\t/, $line;
        my $t = datetime_to_timestamp($line[3]);
        if ($timestamp != $t) {
            if ($checks_temp_data) {
                send_result($timestamp, \@applicable_checks, $logs, $checks_temp_data, $log_only);
            }
            $checks_temp_data = prepare_data(\@applicable_checks);
            $timestamp        = $t;
        }
        my $preparse = $log->{preparse}($line, $log);
        next unless $preparse;
        for my $check (@applicable_checks) {
            if ($line =~ $check->{regex}) {
                $check->{check_sub}->($log, $check, $checks_temp_data->{$check->{name}}, $line, $preparse);
            }
        }
    }

    if ($checks_temp_data) {
        send_result($timestamp, \@applicable_checks, $logs, $checks_temp_data, $log_only);
    }
}

sub prepare_data {
    my ($applicable_checks) = @_;
    my %checks_temp_data = map {$_->{name} => {}} @$applicable_checks;
    return \%checks_temp_data;
}

sub send_result {
    my ($timestamp, $applicable_checks, $logs, $checks_temp_data, $log_only) = @_;

    my %result;
    for my $check (@$applicable_checks) {
        $check->{final_sub}->($check, $logs, $checks_temp_data->{$check->{name}}, \%result);
    }

    for my $path (keys %result) {
        if ($log_only) {
            print "$timestamp $path $result{$path}\n";
        } else {
            send_to_graphite(
                interval  => 'five_min',
                path      => $path,
                value     => $result{$path},
                timestamp => $timestamp,
            );
        }
    }
}

sub check {
    my ($condition, $arg) = @_;

    if (ref($condition) eq 'Regexp') {
        $arg =~ $condition;
    } elsif (ref($condition) eq 'CODE') {
        $condition->($arg);
    } else {
        $condition eq $arg;
    }
}

sub seek_to_start {
    my ($filename) = @_;
    my $fh;
    if ($filename =~ /\.gz$/) {
        open $fh, "-|", "zcat $filename" or die "Cannot start 'zcat $filename': $!";
    } else {
        open $fh, $filename or die "Cannot open '$filename': $!";
    }
    return $fh;
}

sub datetime_to_timestamp {
    my ($dt) = @_;
    my $time = trdate(log => sec => $dt);
    return $time - ($time % 300);
}

sub get_opts {
    my %opts = (
        stat => 'access_log',
        from => 30,
        root => $LOGS_DIRECTORY_PATH,
    );
    Getopt::Long::GetOptions(
        'root=s'   => \$opts{root},
        'from=s'   => \$opts{from},
        'stat=s'   => \$opts{stat},
        'log_only' => \$opts{log_only},
    ) or die "bad args";

    return \%opts;
}
