#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт возвращает блоки в исходное состояние

=head1 USAGE

  perl -I./lib ./bin/oneshots/PI-15843_revert_changed_for_midroll.pl --log=./bin/oneshots/PI-15843_log
  perl -I./lib ./bin/oneshots/PI-15843_revert_changed_for_midroll.pl --log=./bin/oneshots/PI-15843_log --page_ids=12312,534142
  perl -I./lib ./bin/oneshots/PI-15843_revert_changed_for_midroll.pl --log=./bin/oneshots/PI-15843_log --pages_list=./bin/oneshots/PI-15843_pages_list

=head1 OPTIONS

  log        - Лог работы предыдущего скрипта
  page_ids   - Список ID площадок через запятую (необязательный)
  pages_list - Файл со списком пейджей

=cut

use qbit;

use Utils::ScriptWrapper;

sub args {
    my ($opts) = @_;

    return ('log:s' => \$opts->{'log'}, 'page_ids:s' => \$opts->{'page_ids'}, 'pages_list:s' => \$opts->{'pages_list'});
}

run(
    sub {
        my ($app, $opts) = @_;

        die 'Expected --log' unless defined($opts->{'log'});

        my $dry_run = $opts->{'dry_run'};

        my %page_ids = ();
        if (defined($opts->{'page_ids'})) {
            map {$page_ids{$_} = TRUE} split(/,/, $opts->{'page_ids'});
        }

        if (defined($opts->{'pages_list'})) {
            my $content = readfile($opts->{'pages_list'});
            map {$page_ids{$_} = TRUE} grep {defined($_)} split(/\n/, $content);
        }

        my $content = readfile($opts->{'log'});

        foreach my $log_line (split(/\n/, $content)) {
            my $first_char = substr($log_line, 0, 1);

            next if $first_char eq '#';

            my $data = from_json(substr($log_line, 1));

            if ($first_char eq '*') {

                my $block_pk = {
                    page_id => $data->{page_id},
                    id      => $data->{id},
                };

                next unless $page_ids{$block_pk->{'page_id'}};

                my $block = $app->video_an_site_instream->get($block_pk, fields => [qw(multistate), keys(%$data)]);

                print logstr('UPDATE', $block);

                if ($data->{'repeat'} == 0) {
                    delete $data->{$_}
                      for (qw(page_id id category_id max_repeat_count repeat_after try_client server_side));
                } elsif ($data->{'repeat'} == 1) {
                    delete $data->{$_} for (qw(page_id id category_id try_client server_side));
                }

                print logstr('UPDATE DATA', $data);

                next if $dry_run;

                try {
                    $app->video_an_site_instream->do_action($block_pk, 'edit', %$data);
                }
                catch {
                    my ($e) = @_;

                    print STDERR logstr('ERROR UPDATE', {block => $block, data => $data, error => $e->message});
                };
            } elsif ($first_char eq '-') {
                my @data_to_restore = grep {$page_ids{$_->{'page_id'}}} @$data;

                next unless @data_to_restore;

                my $blocks = $app->video_an_site_instream->get_all(
                    fields => [qw(multistate), keys(%{$data_to_restore[0]})],
                    filter => {public_id => [map {$app->video_an_site_instream->public_id($_)} @data_to_restore]}
                );

                print logstr('RESTORE', $blocks);

                next if $dry_run;

                $app->partner_db->transaction(
                    sub {
                        foreach my $block (@data_to_restore) {
                            try {
                                $app->video_an_site_instream->do_action($block, 'restore');
                            }
                            catch {
                                my ($e) = @_;

                                print STDERR logstr('ERROR RESTORE', {block => $block, error => $e->message});
                            };
                        }
                    }
                );
            }
        }
    }
   );
