#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт нуждается в подключении к PROD базе

Меняет блоки по условию из тикета PI-15979 на posterHorizontal

=head1 USAGE

  perl -I./lib ./bin/oneshots/PI-15979_poster_horizontal.pl --get_blocks
  perl -I./lib ./bin/oneshots/PI-15979_poster_horizontal.pl --page_ids=12312,534142
  perl -I./lib ./bin/oneshots/PI-15979_poster_horizontal.pl

=head1 OPTIONS

  get_blocks - Получить список блоков на которые будут накатываться изменения
  page_ids   - Список ID площадок через запятую (необязательный)
  level      - Блочный уровень который нужно обработать

=cut

use qbit;

use Utils::ScriptWrapper;

use Application::Model::Product::AN::ContextOnSite::BlockTypes::RTB qw(direct_block_types);

my $POSTER_HORIZONTAL = 'posterHorizontal';

my @IGNORE_LOGINS = qw(
  bufferbay
  herstshkulevmedia
  kommersant-rsy
  rbcpartner
  riandir
  savvameteo
  savvina-ale
  shmelkin-roman
  tass-rtb
  wwwrg
  );

my $POSTER_DEFAULT_SETTINGS = {
    font_family      => '',
    title_font_size  => 3,
    site_bg_color    => 'FFFFFF',
    bg_color         => 'FFFFFF',
    border_color     => 'DDDCDA',
    header_bg_color  => undef,
    title_color      => '0000CC',
    text_color       => '000000',
    url_color        => '006600',
    hover_color      => 'DD0000',
    sitelinks_color  => '0000CC',
    no_sitelinks     => 0,
    favicon          => 1,
    links_underline  => 1,
    horizontal_align => 1,
};

my $LIMIT          = 5_000;
my $MINIMAL_HEIGHT = 250;

my $DIRECT_BLOCKS = {map {$_ => TRUE} qw(motion 300x250 300x300 adaptive0418 horizontal)};

my $USERS_HAS_FEATURE = {};

my $USERS_EMAILS     = {};
my $DESIGN_TEMPLATES = {};

my $SKIP_FIELDS = {};
foreach my $format (@{direct_block_types()}) {
    next unless $format->{'id'} eq $POSTER_HORIZONTAL || $DIRECT_BLOCKS->{$format->{'id'}};

    my $skip_fields = $format->{'skip_fields'} // $format->{'settings'}{'turbo'}{'skip_fields'};

    $SKIP_FIELDS->{$format->{'id'}} = {
        skip_fields      => $skip_fields,
        hash_skip_fields => {map {$_ => TRUE} @$skip_fields},
    };
}

sub args {
    my ($opts) = @_;

    return (
        'get_blocks!' => \$opts->{'get_blocks'},
        'page_ids:s'  => \$opts->{'page_ids'},
        'level:s'     => \$opts->{'level'},
    );
}

run(
    sub {
        my ($app, $opts) = @_;

        my $dry_run  = $opts->{'dry_run'};
        my $page_ids = $opts->{'page_ids'};

        my @page_ids = ();
        if (defined($opts->{'page_ids'})) {
            @page_ids = split(/,/, $opts->{'page_ids'});
        }

        map {$USERS_HAS_FEATURE->{$_->{'user_id'}} = TRUE} @{
            $app->partner_db->user_features->get_all(
                fields => [qw(user_id feature)],
                filter => {feature => 'design_auction'}
            )
          };

        if ($dry_run) {
            print logstr('USERS_HAS_FEATURE:', $USERS_HAS_FEATURE);
        }

        my @levels = $opts->{'level'} ? ($opts->{'level'}) : qw(context_on_site_rtb internal_context_on_site_rtb);

        foreach (@levels) {
            process_model($app, $_, \@page_ids, $opts);
        }
    }
   );

sub process_model {
    my ($app, $model, $page_ids, $opts) = @_;

    print logstr('MODEL:', $model);

    my $dry_run = $opts->{'dry_run'};

    my $filter = [
        'AND',
        [
            ['site_version',      '=', 'turbo'],
            ['is_custom_bk_data', '=', 0],
            ['multistate',        '=', 'working'],
            ['page', 'NOT MATCH', ['multistate', '=', 'protected']],
            ($model eq 'context_on_site_rtb' ? ['is_constructed_format', '=', 0] : ()),
            ($model eq 'context_on_site_rtb' && @IGNORE_LOGINS ? ['login', 'NOT IN', \@IGNORE_LOGINS] : ()),
            (@$page_ids ? ['page_id', 'IN', $page_ids] : ()),
        ]
    ];

    my $filter_obj = $app->$model->get_db_filter($filter);

    my $pk = $app->$model->partner_db_table->primary_key();

    if ($dry_run) {
        print logstr('PRIMARY_KEYS:', $pk);
    }

    my $all_blocks     = 0;
    my $changed_blocks = 0;

    my $count_blocks = 0;
    my $offset       = 0;

    while (TRUE) {
        my $blocks = $app->$model->get_all(
            fields   => ['*'],
            filter   => $filter_obj,
            order_by => $pk,
            offset   => $offset,
            limit    => $LIMIT
        );

        my $count_blocks = scalar(@$blocks);

        last unless $count_blocks;

        $all_blocks += $count_blocks;

        print logstr('PROCESS BLOCKS COUNT:', $model, $count_blocks);

        my @owner_ids  = ();
        my @design_ids = ();
        foreach my $block (@$blocks) {
            push(@owner_ids, $block->{'owner_id'});

            foreach (@{$block->{'design_templates'}}) {
                push(@design_ids, $_->{'id'});
            }
        }

        $USERS_EMAILS =
          {map {$_->{'id'} => $_->{'email'}}
              @{$app->users->get_all(fields => [qw(id email)], filter => {id => \@owner_ids})}};

        if ($dry_run) {
            print logstr('USERS_EMAILS:', $USERS_EMAILS);
        }

        $DESIGN_TEMPLATES =
          {map {$_->{'id'} => $_->{'opts'}}
              @{$app->partner_db->design_templates->get_all(fields => [qw(id opts)], filter => {id => \@design_ids})}};

        if ($dry_run) {
            print logstr('USERS_EMAILS:', $USERS_EMAILS);
        }

        if ($dry_run) {
            print logstr('DESIGN_TEMPLATES:', $DESIGN_TEMPLATES);
        }

        foreach (@$blocks) {
            $changed_blocks += process_block($app, $model, $_, $opts);
        }

        $offset += $count_blocks;
    }

    print logstr('REPORT:', $model, "$changed_blocks / $all_blocks");
}

sub process_block {
    my ($app, $model, $block, $opts) = @_;

    my $dry_run    = $opts->{'dry_run'};
    my $get_blocks = $opts->{'get_blocks'};

    my $must_change = _is_need_mutate($block);

    if ($get_blocks) {
        print_block($app, $model, $block, $opts, $must_change);

        return 0;
    }

    return 0 unless $must_change;

    foreach (@{$block->{'design_templates'}}) {
        $_->{'opts'} = $DESIGN_TEMPLATES->{$_->{'id'}};
    }

    print logstr(
        'SETTINGS:',
        $model,
        {campaign_id => $block->{'campaign_id'}, id => $block->{'id'}},
        {
            map {$_ => $block->{$_}} 'direct_block',
            'design_templates',
            'border_color',
            'adaptive_height',
            'horizontal_align',
            @{$SKIP_FIELDS->{$POSTER_HORIZONTAL}{'skip_fields'}},
            @{$SKIP_FIELDS->{$block->{'direct_block'}}{'skip_fields'}}
        }
    );

    # скидываем adaptive_height так как он не имеет смысла
    # border_color должен быть задан, берем с блока или дефолт
    # horizontal_align всегда выставляем 1
    my %update = (
        direct_block     => $POSTER_HORIZONTAL,
        adaptive_height  => '',
        horizontal_align => 1,
        border_color     => $block->{'border_color'} || $POSTER_DEFAULT_SETTINGS->{'border_color'},
        map {$_ => undef} @{$SKIP_FIELDS->{$POSTER_HORIZONTAL}{'skip_fields'}}
    );

    my @need_set_fields =
      grep {!$SKIP_FIELDS->{$POSTER_HORIZONTAL}{'hash_skip_fields'}{$_}}
      @{$SKIP_FIELDS->{$block->{'direct_block'}}{'skip_fields'}};
    foreach my $field (@need_set_fields) {
        unless (exists($POSTER_DEFAULT_SETTINGS->{$field})) {
            die 'UNKNOWN DEFAULT FOR: ' . $field;
        }

        $update{$field} = $POSTER_DEFAULT_SETTINGS->{$field};
    }

    my $key   = 'UPDATE:';
    my $error = '';
    unless ($dry_run) {
        try {
            $app->$model->do_action($block, 'edit', %update);
        }
        catch {
            $key   = 'ERROR:';
            $error = shift->message;

            print STDERR logstr('VERBOSE_ERROR:', $model, $block, \%update, $error);
        };
    }

    print logstr($key, {public_id => $block->{'public_id'}, %update}, $error);

    return 1;
}

sub print_block {
    my ($app, $model, $block, $opts, $must_change) = @_;

    #логин, домен, id пейджа, название пейджа, id блока, название блока, email.
    print logstr(
        "PRINT_BLOCK" . ($must_change ? '_MUST_CHANGE:' : '_IGNORE:'), $block->{'login'} // 'adinside',
        $block->{'domain'},          $block->{'page_id'},
        $block->{'page'}{'caption'}, $block->{'public_id'},
        $block->{'caption'},         $block->{'owner_id'},
        $USERS_EMAILS->{$block->{'owner_id'}}
    );
}

sub _is_need_mutate {
    my ($block) = @_;

    return FALSE if $USERS_HAS_FEATURE->{$block->{'owner_id'}};

    return FALSE unless $DIRECT_BLOCKS->{$block->{'direct_block'}};

    if (   $block->{'direct_block'} eq '300x250'
        || $block->{'direct_block'} eq '300x300')
    {
        return TRUE;
    } elsif ($block->{'direct_block'} eq 'motion' || $block->{'direct_block'} eq 'horizontal') {
        return TRUE if length($block->{'adaptive_height'}) && $block->{'adaptive_height'} >= $MINIMAL_HEIGHT;

        return TRUE
          if $block->{'adaptive_height'} eq ''
              && !($block->{'media_active'} && $block->{'media_blocked'})
              && grep {$_ =~ /.+?x(\d+)/ && $1 > 50} @{$block->{'dsp_blocks'}};
    } elsif (
        $block->{'direct_block'} eq 'adaptive0418'
        && !($block->{'media_active'} && $block->{'media_blocked'})
        && grep {
            $_ =~ /.+?x(\d+)/ && $1 > 50
        } @{$block->{'dsp_blocks'}}
      )
    {
        return TRUE;
    }

    return FALSE;
}
