#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт применяет фильтры и тематики на площадки и их блоки

=head1 USAGE

  bin/oneshots/PI-16098_mass_set_filter_and_articles.pl
  bin/oneshots/PI-16098_mass_set_filter_and_articles.pl --commit
  bin/oneshots/PI-16098_mass_set_filter_and_articles.pl --skip_logs --resend_cmd

=head1 OPTIONS

  commit - применить изменения в БД
  resend_cmd - только показать команду для посылке изменений в БК
  revert=<filename> - восстановить предыдущее состояние

=cut

use feature 'state';
use lib::abs qw(../../lib);

use qbit;
use Utils::ScriptWrapper;

use PiConstants qw($MAX_CPM);

my $sql_filters_add = 'insert ignore into `filters` (`page_id`, `filter_id`) values %s';
my $sql_article_add =
'insert into `articles` (`page_id`, `block_id`,`article_sid`,`cpm`) values %s on duplicate key update `cpm` = values(`cpm`)';
my $sql_filters_drop = 'delete from `filters` where `page_id` in (%s)';
my $sql_article_drop = 'delete from `articles` where (`page_id`, `block_id`,`article_sid`) in (%s)';

run(\&main);

sub main {
    my ($app, $opts) = @_;

    if ($opts->{resend_cmd}) {
        resend_cmd();
        return;
    }

    if ($opts->{revert}) {
        revert_changes($app, $opts);
        resend_cmd();
        return;
    }

    my $page_blocks = get_current($app, $opts);

    if ($opts->{commit}) {
        make_changes($app, $page_blocks);
        resend_cmd();
    }

}

sub revert_changes {
    my ($app, $opts) = @_;
    open F, $opts->{revert} or die "Cannot open '$opts->{revert}': $!";
    my @filters_drop;
    my @filters_add;
    my @articles_drop;
    my @articles_add;
    my $article_sid = get_article_sid();
    while (my $str = <F>) {
        my ($page, $block, $value);
        if (($page, $value) = $str =~ /^F (\d+):([\d,]+)?/) {
            push @filters_drop, $page;
            if ($value) {
                push @filters_add, "($page, $_)" for split(/,/, $value);
            }
        } elsif (($page, $block, $value) = $str =~ /^A (\d+),(\d+):(\d+|no)/) {
            push @articles_drop, "($page, $block, $article_sid)";
            unless ($value eq 'no') {
                push @articles_add, "($page, $block, $article_sid, $value)";
            }
        }
    }
    close F;

    $app->partner_db->transaction(
        sub {
            $app->partner_db->_do(sprintf $sql_filters_drop, join(',', @filters_drop))  if @filters_drop;
            $app->partner_db->_do(sprintf $sql_article_drop, join(',', @articles_drop)) if @articles_drop;
            $app->partner_db->_do(sprintf $sql_filters_add,  join(',', @filters_add))   if @filters_add;
            $app->partner_db->_do(sprintf $sql_article_add,  join(',', @articles_add))  if @articles_add;
        }
    );
}

sub make_changes {
    my ($app, $page_blocks) = @_;

    my @filters_list = get_filters();
    my $article_sid  = get_article_sid();
    my @filter_data;
    my @article_data;
    for my $page_id (keys %$page_blocks) {
        push @filter_data, "($page_id, $_)" for @filters_list;
        for my $block_id (keys %{$page_blocks->{$page_id}}) {
            push @article_data, "($page_id, $block_id, $article_sid, $MAX_CPM)"
              unless $page_blocks->{$page_id}{$block_id};
        }
    }

    $app->partner_db->transaction(
        sub {
            $app->partner_db->_do(sprintf $sql_filters_add, join(',', @filter_data));
            $app->partner_db->_do(sprintf $sql_article_add, join(',', @article_data));
        }
    );
}

sub get_current {
    my ($app, $opts) = @_;

    my $pages       = get_list();
    my $article_sid = get_article_sid();

    my $filters = $app->partner_db->filters->get_all(filter => ["page_id", "IN", \$pages]);
    my $articles = $app->partner_db->articles->get_all(filter => ["page_id", "IN", \$pages]);
    my $blocks = $app->partner_db->video_an_site_instream->get_all(
        fields => [qw(id page_id is_custom_bk_data)],
        filter => ["page_id", "IN", \$pages]
    );

    my %filter_data;
    for my $filter (@$filters) {
        $filter_data{$filter->{page_id}}{$filter->{filter_id}} = 1;
    }

    my %article_data;
    for my $article (@$articles) {
        $article_data{$article->{page_id}}{$article->{block_id}}{$article->{article_sid}} = $article->{cpm};
    }

    my %page_blocks;
    for my $row (@$blocks) {
        $page_blocks{$row->{page_id}}{$row->{id}} = $row->{is_custom_bk_data};
    }

    my $filter_data = '';
    for my $page_id (@$pages) {
        $filter_data .= "F $page_id:";
        if ($filter_data{$page_id}) {
            $filter_data .= join(",", sort {$a <=> $b} keys %{$filter_data{$page_id}});
        }
        $filter_data .= "\n";
    }
    my $article_data = '';
    for my $page_id (sort {$a <=> $b} keys %page_blocks) {
        for my $block_id (sort {$a <=> $b} keys %{$page_blocks{$page_id}}) {
            next if $page_blocks{$page_id}{$block_id};
            $article_data .= "A $page_id,$block_id:";
            $article_data .= $article_data{$page_id}{$block_id}{$article_sid} // 'no';
            $article_data .= "\n";
        }
    }
    print $filter_data;
    print $article_data;

    return \%page_blocks;
}

sub resend_cmd {
    printf "---\nbin/resend_to_bk.pl --page_ids=%s\n---\n", join(",", @{get_list()});
}

sub args {
    my ($opts) = @_;

    return (
        'resend_cmd!' => \$opts->{'resend_cmd'},
        'revert=s'    => \$opts->{'revert'},
    );
}

sub get_list {
    state $list;
    unless ($list) {
        while (my $id = <DATA>) {
            $id =~ s/\s//g;
            if ($id) {
                push @$list, $id;
            }
        }
    }
    return $list;
}

sub get_article_sid {
    return -40;
}

sub get_filters {
    return qw(
      1
      3
      4
      5
      6
      7
      8
      15
      16
      17
      19
      20
      21
      22
      23
      24
      );
}

__DATA__
260214
260002
259865
259701
259864
259856
260105
260103
260099
259869
191823
263881
263882
269587
275930
276119
276178
284761
293617
293622
269897
293626
293631
293685
293635
293636
337807
337812
337816
337821
337825
337829
337833
337837
337841
337844
337848
337852
337853
337854
337856
337857
337858
337860
337861
337862
337863
337864
337865
337866
337867
337868
337869
337871
337872
337873
337874
337875
337876
337877
337878
337879
337881
337882
337883
337884
328590
287837
326241
330941
330904
330945
330931
