#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

    Скрипт нуждается в подключении к PROD базе

    Добавляет на блоки форматы по условию из тикета PI-16405

=head1 USAGE

    perl -I./lib ./bin/oneshots/PI-16405_add_formats.pl --get_blocks
    perl -I./lib ./bin/oneshots/PI-16405_add_formats.pl --page-ids=12312,534142
    perl -I./lib ./bin/oneshots/PI-16405_add_formats.pl

=head1 OPTIONS

    black-list - проверить блек-лист и успокоиться
    get-blocks - ограничить работу скрипта только выводом информации о блоках,
                 которые будут подвергнуты изменениям
    logins     - ограничить работу скрипта только указанными логинами
    page-ids   - ограничить работу скрипта только указанными пейджами
    page-limit - размер выдаваемых списков пейджей на апдейт (примерно)

=cut

use qbit;

use Application::Model::Product::AN::ContextOnSite::BlockTypes::RTB qw(direct_block_types);
use Utils::ScriptWrapper;

my @MODELS  = qw(context_on_site_rtb);
my @FORMATS = (
    '200x300',        '240x400',    '300x300',          '300x500',
    '300x600',        '336x280',    'posterHorizontal', 'posterVertical',
    'modernAdaptive', 'horizontal', 'motion',           'extensibleMobile',
    'vertical',       'adaptive0418',
);
my %FORMATS;
@FORMATS{@FORMATS} = ();
my @ADD = qw(oldPosterHorizontal posterHorizontal modernAdaptive extensibleMobile);
my %ADD = (
    modernAdaptive => {
        adaptiveHeight       => '',
        adaptiveWidth        => '',
        borderColor          => 'DDDCDA',
        borderRadius         => 0,
        borderType           => 'block',
        caption              => 'Адаптивный',
        favicon              => 1,
        horizontalAlign      => 1,
        hoverColor           => 'DD0000',
        imagesFirst          => 0,
        isCustomFormatDirect => 0,
        limit                => 2,
        linksUnderline       => 1,
        name                 => 'modernAdaptive',
        noSitelinks          => 0,
        siteBgColor          => 'FFFFFF',
        sitelinksColor       => '0000CC',
        textColor            => '000000',
        titleColor           => '0000CC',
        urlBackgroundColor   => '0000CC',
        urlColor             => '006600',
    },
    extensibleMobile => {
        adaptiveHeight       => '',
        adaptiveWidth        => '',
        caption              => 'Motion',
        isCustomFormatDirect => 0,
        name                 => 'extensibleMobile',
    },
    oldPosterHorizontal => {
        adaptiveHeight       => '',
        adaptiveWidth        => '',
        borderColor          => 'DDDCDA',
        borderRadius         => 0,
        borderType           => 'block',
        callouts             => undef,
        caption              => 'Постер классический',
        favicon              => 1,
        fontFamily           => '',
        fontSize             => '1.1',
        horizontalAlign      => 1,
        hoverColor           => 'DD0000',
        imagesFirst          => 0,
        isCustomFormatDirect => 1,
        limit                => 1,
        linksUnderline       => 1,
        name                 => 'oldPosterHorizontal',
        noSitelinks          => 0,
        siteBgColor          => 'FFFFFF',
        sitelinksColor       => '000000',
        textColor            => '000000',
        titleColor           => '000000',
        titleFontSize        => 3,
        urlBackgroundColor   => undef,
        urlColor             => '000000',
    },
    posterHorizontal => {
        adaptiveHeight       => '',
        adaptiveWidth        => '',
        borderColor          => 'DDDCDA',
        caption              => 'Постер новый',
        favicon              => 0,
        fontFamily           => '',
        horizontalAlign      => 1,
        hoverColor           => 'DD0000',
        isCustomFormatDirect => 0,
        linksUnderline       => 0,
        name                 => 'posterHorizontal',
        noSitelinks          => 0,
        siteBgColor          => 'FFFFFF',
        sitelinksColor       => '0000CC',
        textColor            => '000000',
        titleColor           => '0000CC',
        titleFontSize        => 3,
        urlColor             => '006600',
    },
);
my $LIMIT = 100;

my @IGNORE_LOGINS = qw(
  wwwrg
  tass-rtb
  samovarjr
  admesales
  spadeladyimb
  bufferbay
  mkrucom
  kommersant-rsy
  wwwkpru
  everestsales
  echomoscu
  izvestiareader
  rentv-channel
  slimushkabestsonoff
  teleradiocompany
  trkpeterburg-5
  tv78-license
  tv78spb
  riandir
  savvameteo
  savvina-ale
  shmelkin-roman
  herstshkulevmedia
  advert-vgtrk
  vgtrk-advert
  fashionpress-parol
  grazia-pim
  rbcpartner
  avitortb2013
  maximatelecom
  m-ru-text
  m-ru-2015
  wwwodnoklassniki-2015
  vk-rsya
  vk-rsya-apps
  );

my %SKIP_FIELDS = (
    oldPosterHorizontal => {
        skip_fields      => [],
        hash_skip_fields => {},
    }
);

my $opts = {page_limit => 1000,};
my $file_n = 0;

sub args {
    my ($opts) = @_;

    return (
        'black-list!'  => \$opts->{black_list},
        'get-blocks!'  => \$opts->{get_blocks},
        'logins:s@'    => \$opts->{logins},
        'page-ids:s@'  => \$opts->{page_ids},
        'page-limit:s' => \$opts->{page_limit},
    );
}

run(
    sub {
        (my $app, $opts) = @_;

        # Check logins (if provided)
        my @logins = split /\s*,\s*/, join ',', @{$opts->{'logins'} // []};
        if (scalar @logins) {
            my %logins;
            my $users = $app->users->get_all(fields => [qw(id login)], filter => {login => \@logins});
            foreach my $user (@$users) {
                delete $logins{$user->{login}};
            }
            if (scalar keys %logins) {
                print logstr('Login list error:', [sort keys %logins]);
                return;
            }
        }

        my %black_list;
        @black_list{@IGNORE_LOGINS} = ();
        my @incorrect_logins = grep {exists $black_list{$_}} @logins;
        if (scalar @incorrect_logins) {
            print logstr('Incorrect logins:', \@incorrect_logins);
            return;
        }

        # Check black list (and exit if only check is needed)
        my $users = $app->users->get_all(fields => [qw(id login)], filter => {login => \@IGNORE_LOGINS});
        foreach my $user (@$users) {
            delete $black_list{$user->{login}};
        }
        if (scalar keys %black_list) {
            print logstr('Black list error:', [sort keys %black_list]);
            return;
        }
        if ($opts->{black_list}) {
            print logstr 'Check black list option is received. Black list is correct. Done.';
            return;
        }

        my @page_ids = split /\s*,\s*/, join ',', @{$opts->{'page_ids'} // []};
        if ($opts->{'page_ids'} && !scalar @page_ids) {
            print logstr('Incorrect list of pages');
            return;
        }

        my %history;
        my %pages_for_update;
        foreach my $model_name (@MODELS) {
            print logstr('Processing model:', $model_name);
            my $model = $app->$model_name;

            my $map = $model->get_design_fields_map();
            foreach my $format (@{direct_block_types()}) {
                next unless (exists $FORMATS{$format->{id}} || exists $ADD{$format->{id}});
                my @skip_fields =
                  map {$map->{$_} // $_} @{$format->{skip_fields} // $format->{settings}{mobile}{skip_fields} // []};
                $SKIP_FIELDS{$format->{id}} = {
                    skip_fields      => \@skip_fields,
                    hash_skip_fields => {map {$_ => TRUE} @skip_fields},
                };
            }

            my $filter = [
                AND => [
                    {
                        site_version      => 'mobile',
                        is_custom_bk_data => 0,
                        multistate        => 'working',
                    },
                    ['page', 'NOT MATCH', ['multistate', '=', 'protected']],
                    (
                        scalar @logins ? ['login', 'IN', \@logins]
                        : (
                            scalar @IGNORE_LOGINS ? ['login', 'NOT IN', \@IGNORE_LOGINS]
                            : ()
                          )
                    ),
                    (
                        scalar @page_ids ? ['page_id', 'IN', \@page_ids]
                        : ()
                    ),
                ]
            ];
            my $pk     = $model->partner_db_table->primary_key();
            my $offset = 0;
            while (TRUE) {
                my $blocks = $model->get_all(
                    fields   => ['*'],
                    filter   => $filter,
                    order_by => $pk,
                    offset   => $offset,
                    limit    => $LIMIT,
                );
                my $count = scalar(@$blocks);
                print logstr sprintf('Iteration: [%d .. %d], count: %d', $offset, $LIMIT + $offset - 1, $count);
                last unless $count;
                $offset += $count;
                foreach my $block (@$blocks) {
                    my $public_id = $model->public_id($block);
                    unless (
                        eval {
                            my $template = check_block($block);
                            if ($template) {
                                print logstr $public_id;
                                return TRUE if $opts->{get_blocks};
                                $history{$public_id} = [];
                                my %formats = map {$_->{design_settings}{name} => $_} @{$block->{design_templates}};
                                foreach my $add (@ADD) {
                                    # На блоке уже есть такой формат
                                    next if exists $formats{$add};

                                    my %settings = (
                                        %{$template->{design_settings}},
                                        map {$_ => undef,} @{$SKIP_FIELDS{$add}{skip_fields}},
                                    );
                                    my @need_set_fields =
                                      grep {exists($ADD{$add}{$_}) && !$SKIP_FIELDS{$add}{hash_skip_fields}{$_}}
                                      @{$SKIP_FIELDS{$template->{design_settings}{name}}{skip_fields}};
                                    $settings{$_} = $ADD{$add}{$_}
                                      foreach (@need_set_fields, qw(isCustomFormatDirect name));
                                    delete @settings{qw(caption blockId)};
                                    if ($settings{isCustomFormatDirect}) {
                                        $settings{callouts}           = undef;
                                        $settings{urlBackgroundColor} = undef;
                                        $settings{adaptiveHeight}     = '';
                                        $settings{adaptiveWidth}      = '';
                                    }
                                    $settings{sitelinksColor} = undef if ($settings{noSitelinks});
                                    $settings{adaptiveType} = undef if ($add ne 'modernAdaptive');
                                    $settings{adaptiveHeight} = ''
                                      unless (scalar grep {$add eq $_} qw(horizontal adaptive motion extensibleMobile));
                                    $settings{adaptiveWidth} = '' if ($add ne 'adaptive');
                                    $settings{borderColor} //= $ADD{$add}{borderColor}
                                      if ($add eq 'posterHorizontal' && !defined $settings{borderType});

                                    # Дополнительные костыли:

                                    # Denis, [25.06.19 14:32]
                                    #   давай тогда так. Если ты нашел форматы 240×400, 300х300 -
                                    #   то в этом случае в добавляемых форматах надо ставить дефолтное количество объявлений,
                                    #   а для modernAdaptive - всегда 1 в таком случае

                                    # Denis, [25.06.19 14:40]
                                    #   хотя погоди…

                                    # Артём Румянцев, [25.06.19 14:52]
                                    #   давай вернёмся к вопросу о лимитах:
                                    #
                                    #   я правильно понял, что если шаблонным форматом является 240×400, 300х300,
                                    #   то в добавляемых форматах лимит ставить в дефольное значение, но не большше 1?
                                    #
                                    #   или это для всех надо так сделать?

                                    # Denis, [25.06.19 14:53]
                                    #   для всех

                                    # Denis, [25.06.19 14:53]
                                    #   posterHorizontal и motion - всегда и так авто

                                    # Denis, [25.06.19 14:53]
                                    #   oldPosterHorizontal - ставим 1

                                    # Denis, [25.06.19 14:53]
                                    #   modernAdaptive - ставим 1

                                    if ($add eq 'modernAdaptive' || $add eq 'oldPosterHorizontal') {
                                        $settings{limit} = 1;
                                    } else {
                                        $settings{limit} = undef;
                                    }

                                    my %design = (
                                        block_id        => $block->{id},
                                        caption         => $ADD{$add}{caption},
                                        design_settings => \%settings,
                                        form_factor     => undef,
                                        page_id         => $block->{page_id},
                                    );
                                    push @{$history{$public_id}}, \%design;
                                    push @{$block->{design_templates}}, \%design;
                                }
                                if (scalar @{$history{$public_id}}) {
                                    print logstr $block->{design_templates};
                                    my %validator_template = $model->get_template();
                                    $validator_template{template}{fields} =
                                      {design_templates => $validator_template{template}{fields}{design_templates}};
                                    $app->validator->check(
                                        app   => $model,
                                        data  => $block,
                                        throw => TRUE,
                                        %validator_template,
                                    );
                                    for (my $i = 0; $i < scalar @{$history{$public_id}}; $i++) {
                                        my $id = $app->design_templates->add(%{$history{$public_id}[$i]});
                                        $history{$public_id}[$i] = $id;
                                    }
                                    print logstr sprintf('For block %s following design templates were added: %s',
                                        $public_id, join(', ', @{$history{$public_id}}));
                                    $pages_for_update{$block->{page_id}}++;
                                }
                            }
                            return TRUE;
                        }
                      )
                    {
                        print logstr sprintf('Error while processing block %s: %s', $public_id, $@);
                    }
                }
                print_pages(\%pages_for_update) if (scalar keys %pages_for_update > $opts->{page_limit});
            }
        }
        # Для отката надо взять данные из этого лога и подставить в запрос:
        # 1) Если откатывать через мультистатус, то:
        #    UPDATE `design_templates` SET `multistate` = 1 WHERE `id` IN (...);
        # 2) Если через полное удаление, то:
        #    DELETE FROM `design_templates_action_log` WHERE `elem_id` IN (...);
        #    DELETE FROM `design_templates` WHERE `id` IN (...);
        print logstr 'Whole added data:', [grep {!ref $_} map {@$_} values %history];
        print_pages(\%pages_for_update);
    }
   );

# Проверяет, нужно ли на блок добавлять дизайны
# Если да, возвращает первый существующий на блоке дизайн, удовлетворяющий требованиям
sub check_block {
    my ($block) = @_;

    my %designes = map {$_->{design_settings}{name} => $_,} @{$block->{design_templates}};

    # П. 3.1
    # хотя бы один формат  200×300, 240×400, 300×300, 300×500, 300×600, 336x280, posterHorizontal, posterVertical
    foreach ('200x300', '240x400', '300x300', '300x500', '300x600', '336x280', 'posterHorizontal', 'posterVertical') {
        return $designes{$_} if $designes{$_};
    }

    # П. 3.2
    # ИЛИ формат  modernAdaptive
    # И у него параметр "adaptiveType" = "l" (т. е. выбрана высота 300 px)
    foreach ('modernAdaptive') {
        return $designes{$_}
          if $designes{$_}
              && $designes{$_}{design_settings}
              && $designes{$_}{design_settings}{adaptiveType}
              && $designes{$_}{design_settings}{adaptiveType} eq 'l';
    }

    # П. 3.3
    # ИЛИ формат  horizontal, motion, extensibleMobile
    # И у него параметр adaptive_height >= "290" (т. е. авто-высота отключена)
    foreach ('horizontal', 'motion', 'extensibleMobile') {
        return $designes{$_}
          if $designes{$_}
              && $designes{$_}{design_settings}
              && $designes{$_}{design_settings}{adaptiveHeight}
              && $designes{$_}{design_settings}{adaptiveHeight} >= 290;
    }

    # Вспомогательная переменная, используется в двух нижеследующих проверках
    # И в "Sizes" медийки есть хотя бы один размер с высотой >= 290px
    # И медийка не заблокирована в стратегии "Раздельный CPM"
    my $media_check =
      (     !$block->{media_blocked}
          && $block->{dsp_blocks}
          && scalar(grep {m/x(\d+)$/ && $1 >= 290} @{$block->{dsp_blocks}}));

    # П. 3.4
    # ИЛИ формат  horizontal, motion, extensibleMobile
    # И у него параметр adaptive_height НЕ задан (т. е. авто-высота включена)
    # И в "Sizes" медийки есть хотя бы один размер с высотой >= 290px
    # И медийка не заблокирована в стратегии "Раздельный CPM"
    foreach ('horizontal', 'motion', 'extensibleMobile') {
        return $designes{$_}
          if $designes{$_}
              && $designes{$_}{design_settings}
              && !$designes{$_}{design_settings}{adaptiveHeight}
              && $media_check;
    }

    # П. 3.5
    # ИЛИ формат  vertical, adaptive0418
    # И в "Sizes" медийки есть хотя бы один размер с высотой >= 290px
    # И медийка не заблокирована в стратегии "Раздельный CPM"
    foreach ('vertical', 'adaptive0418') {
        return $designes{$_}
          if $designes{$_}
              && $designes{$_}{design_settings}
              && !$designes{$_}{design_settings}{adaptiveHeight}
              && $media_check;
    }
}

sub print_pages {
    my ($pages) = @_;

    my @ids = (keys %$pages);
    print logstr 'Pages for update:', \@ids;
    my $fh;
    if (open($fh, '>', sprintf('%s.%d.page_id', $opts->{file_name}, $file_n++))) {
        print $fh join("\n", @ids, '');
        close $fh;
        %$pages = ();
    } else {
        print logstr 'Can\'t write page ids:', $!;
    }
}
