#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

    Скрипт нуждается в подключении к PROD базе

    Добавляет на блоки с десктопной версией дизайны по условию из тикета PI-16600

=head1 USAGE

    perl -I./lib ./bin/oneshots/PI-16600_add_new_designs_for_desktop_blocks.pl --get_blocks
    perl -I./lib ./bin/oneshots/PI-16600_add_new_designs_for_desktop_blocks.pl --page_ids=12312,534142
    perl -I./lib ./bin/oneshots/PI-16600_add_new_designs_for_desktop_blocks.pl

=head1 OPTIONS

    black_list - проверить блек-лист и успокоиться
    get_blocks - ограничить работу скрипта только выводом информации о блоках,
                 которые будут подвергнуты изменениям
    logins     - ограничить работу скрипта только указанными логинами
    page_ids   - ограничить работу скрипта только указанными пейджами
    page_limit - размер выдаваемых списков пейджей на апдейт (примерно)

=cut

use qbit;

use Utils::ScriptWrapper;

my @MODELS  = qw(context_on_site_rtb);
my $FORMATS = {
    vertical => {
        '300x300' => {
            'condition' => {
                filter  => undef,
                formats => {
                    posterHorizontal    => {limit => 1},
                    oldPosterHorizontal => {limit => 1},
                    extensibleMobile    => {limit => 'skip'},
                    zen                 => {limit => 1},
                }
            }
        },
        '336x280' => {
            'condition' => {
                filter  => undef,
                formats => {
                    adaptive0418     => {limit => 1},
                    extensibleMobile => {limit => 'skip'},
                    zen              => {limit => 1},
                }
            }
        },
        'vertical' => {
            'condition' => {
                filter  => undef,
                formats => {
                    posterVertical    => {limit => 'current'},
                    oldPosterVertical => {limit => 'current'},
                    zen               => {limit => 'current'},
                }
            }
        },
        'modernAdaptive' => {
            condition_1 => {
                'filter'  => 'adaptive_type_gt_290',
                'formats' => {
                    posterVertical    => {limit => 'current'},
                    oldPosterVertical => {limit => 'current'},
                    extensibleMobile  => {limit => 'skip'},
                    zen               => {limit => 'current'},
                }
            },
            condition_2 => {
                'filter'  => 'adaptive_type_less_290',
                'formats' => {
                    extensibleMobile => {limit => 'skip'},
                    zen              => {limit => 'current'},
                }
            }
        },
        'posterVertical' => {
            condition => {
                filter    => undef,
                'formats' => {
                    oldPosterVertical => {limit => 'current'},
                    vertical          => {limit => 'current'},
                    zen               => {limit => 'current'},
                }
            }
        },
        '300x500' => {
            condition => {
                filter    => undef,
                'formats' => {
                    posterVertical    => {limit => 1},
                    oldPosterVertical => {limit => 1},
                    zen               => {limit => 1},
                    extensibleMobile  => {limit => 'skip'},
                }
            }
        },
        '300x600' => {
            condition => {
                filter    => undef,
                'formats' => {
                    posterVertical    => {limit => 'current'},
                    oldPosterVertical => {limit => 'current'},
                    zen               => {limit => 'current', adaptiveWidth => 300, adaptiveHeight => 600},
                    extensibleMobile  => {limit => 'skip'},
                }
            }
        },
        'extensibleMobile' => {
            'condition_1' => {
                'filter'  => 'adaptive_height_gt_290',
                'formats' => {
                    'posterHorizontal'    => {limit => 1},
                    'oldPosterHorizontal' => {limit => 1},
                    'zen'                 => {limit => 1},
                    'adaptive0418'        => {limit => 1},
                },
            },
            'condition_2' => {
                'filter'  => 'adaptive_height_less_290',
                'formats' => {
                    'zen'          => {limit => 1},
                    'adaptive0418' => {limit => 1},
                },
            }
        },
        'motion' => {
            'condition_1' => {
                'filter'  => 'adaptive_height_gt_290',
                'formats' => {
                    'posterHorizontal'    => {limit => 1},
                    'oldPosterHorizontal' => {limit => 1},
                    'zen'                 => {limit => 1},
                    'adaptive0418'        => {limit => 1},
                },
            },
            'condition_2' => {
                'filter'  => 'adaptive_height_less_290',
                'formats' => {
                    'zen'          => {limit => 1},
                    'adaptive0418' => {limit => 1},
                },
            }
        },
    },
    horizontal => {
        'horizontal' => {
            'condition_1' => {
                'filter'  => 'adaptive_height_gt_290',
                'formats' => {
                    'posterHorizontal'    => {limit => 'current'},
                    'oldPosterHorizontal' => {limit => 'current'},
                    'zen'                 => {limit => 'current'},
                    'extensibleMobile'    => {limit => 'skip'},
                },
            },
            'condition_2' => {
                'filter'  => 'adaptive_height_less_290',
                'formats' => {
                    'zen'              => {limit => 'current'},
                    'extensibleMobile' => {limit => 'skip'},
                },
            }
        },
        'posterHorizontal' => {
            'condition' => {
                filter  => undef,
                formats => {
                    'oldPosterHorizontal' => {limit => 'current',},
                    'zen'                 => {limit => 'current',},
                    'extensibleMobile'    => {limit => 'skip',},
                    'adaptive0418'        => {limit => 'current',},
                }
            }
        },
        'modernAdaptive' => {
            'condition_1' => {
                'filter'  => 'adaptive_type_gt_290',
                'formats' => {
                    'posterHorizontal'    => {limit => 'current'},
                    'oldPosterHorizontal' => {limit => 'current'},
                    'zen'                 => {limit => 'current'},
                    'extensibleMobile'    => {limit => 'skip'},
                },
            },
            'condition_2' => {
                'filter'  => 'adaptive_type_less_290',
                'formats' => {
                    'zen'              => {limit => 'current'},
                    'extensibleMobile' => {limit => 'skip'},
                },
            }
        },
        'extensibleMobile' => {
            'condition_1' => {
                'filter'  => 'adaptive_height_gt_290',
                'formats' => {
                    'posterHorizontal'    => {limit => 4},
                    'oldPosterHorizontal' => {limit => 4},
                    'zen'                 => {limit => 4},
                    'adaptive0418'        => {limit => 4},
                },
            },
            'condition_2' => {
                'filter'  => 'adaptive_height_less_290',
                'formats' => {
                    'zen'          => {limit => 4},
                    'adaptive0418' => {limit => 4},
                },
            }
        },
        'motion' => {
            'condition_1' => {
                'filter'  => 'adaptive_height_gt_290',
                'formats' => {
                    'posterHorizontal'    => {limit => 4},
                    'oldPosterHorizontal' => {limit => 4},
                    'zen'                 => {limit => 4},
                    'adaptive0418'        => {limit => 4},
                },
            },
            'condition_2' => {
                'filter'  => 'adaptive_height_less_290',
                'formats' => {
                    'zen'          => {limit => 4},
                    'adaptive0418' => {limit => 4},
                },
            }
        },
        '970x250' => {
            'condition' => {
                filter  => undef,
                formats => {
                    'zen'              => {limit => 4,},
                    'extensibleMobile' => {limit => 'skip',},
                }
            }
        },
        '300x300' => {
            'condition' => {
                filter  => undef,
                formats => {
                    posterHorizontal    => {limit => 1},
                    oldPosterHorizontal => {limit => 1},
                    extensibleMobile    => {limit => 'skip'},
                    zen                 => {limit => 1},
                }
            }
        },
        '336x280' => {
            'condition' => {
                filter  => undef,
                formats => {
                    adaptive0418     => {limit => 1},
                    extensibleMobile => {limit => 'skip'},
                    zen              => {limit => 1},
                }
            },
        },
    },
};

my %ADD = (
    extensibleMobile => {
        caption         => 'Анимированный',
        design_settings => {
            adaptiveHeight => '',
            adaptiveWidth  => '',
            name           => 'extensibleMobile',
        },
        is_custom_format_direct => 0,
    },
    oldPosterHorizontal => {
        caption         => 'Постер классический',
        design_settings => {
            adaptiveHeight     => '',
            adaptiveWidth      => '',
            borderColor        => 'DDDCDA',
            borderRadius       => 0,
            borderType         => 'block',
            callouts           => undef,
            favicon            => 1,
            fontFamily         => '',
            fontSize           => '1.1',
            horizontalAlign    => 1,
            hoverColor         => 'DD0000',
            imagesFirst        => 0,
            linksUnderline     => 1,
            name               => 'oldPosterHorizontal',
            noSitelinks        => 0,
            siteBgColor        => 'FFFFFF',
            sitelinksColor     => '000000',
            textColor          => '000000',
            titleColor         => '000000',
            titleFontSize      => 3,
            urlBackgroundColor => undef,
            urlColor           => '000000',
        },
        form_factor             => 'horizontal',
        is_custom_format_direct => 1,
    },
    posterHorizontal => {
        caption         => 'Постер новый',
        design_settings => {
            adaptiveHeight  => '',
            adaptiveWidth   => '',
            borderColor     => 'DDDCDA',
            borderRadius    => 0,
            borderType      => 'block',
            favicon         => 0,
            fontFamily      => '',
            horizontalAlign => 1,
            hoverColor      => 'DD0000',
            linksUnderline  => 0,
            name            => 'posterHorizontal',
            noSitelinks     => 0,
            siteBgColor     => 'FFFFFF',
            sitelinksColor  => '0000CC',
            textColor       => '000000',
            titleColor      => '0000CC',
            titleFontSize   => 3,
            urlColor        => '006600',
        },
        form_factor             => 'horizontal',
        is_custom_format_direct => 0,
    },
    zen => {
        caption         => 'Дзен',
        design_settings => {
            adaptiveHeight  => '',
            adaptiveWidth   => '',
            borderColor     => 'DDDCDA',
            borderRadius    => 0,
            borderType      => 'block',
            favicon         => 1,
            fontFamily      => '',
            fontSize        => '1.1',
            horizontalAlign => 1,
            hoverColor      => 'DD0000',
            imagesFirst     => 0,
            linksUnderline  => 1,
            name            => 'zen',
            noSitelinks     => 0,
            siteBgColor     => 'FFFFFF',
            sitelinksColor  => '0000CC',
            textColor       => '000000',
            titleColor      => '0000CC',
            titleFontSize   => 3,
            urlColor        => '006600',
        },
        is_custom_format_direct => 1,
    },
    adaptive0418 => {
        caption         => 'Адаптивный',
        design_settings => {
            adaptiveHeight  => '',
            adaptiveWidth   => '',
            borderColor     => 'DDDCDA',
            borderRadius    => 0,
            borderType      => 'block',
            favicon         => 1,
            fontFamily      => '',
            fontSize        => '1.1',
            horizontalAlign => 1,
            hoverColor      => 'DD0000',
            imagesFirst     => 0,
            linksUnderline  => 1,
            name            => 'adaptive0418',
            noSitelinks     => 0,
            siteBgColor     => 'FFFFFF',
            sitelinksColor  => '0000CC',
            textColor       => '000000',
            titleColor      => '0000CC',
            titleFontSize   => 3,
            urlColor        => '006600',
        },
        is_custom_format_direct => 1,
    },
    vertical => {
        caption         => 'Вертикальный',
        design_settings => {
            adaptiveHeight     => '',
            adaptiveWidth      => '',
            borderType         => 'none',
            borderColor        => undef,
            callouts           => 0,
            favicon            => 0,
            fontFamily         => '',
            fontSize           => '1.1',
            horizontalAlign    => 0,
            hoverColor         => 'DD0000',
            imagesFirst        => 0,
            linksUnderline     => 0,
            name               => 'vertical',
            noSitelinks        => 0,
            siteBgColor        => 'FFFFFF',
            sitelinksColor     => '0000CC',
            textColor          => '000000',
            titleColor         => '0000CC',
            titleFontSize      => 3,
            urlBackgroundColor => '0000CC',
            urlColor           => '006600',
        },
        form_factor             => 'vertical',
        is_custom_format_direct => 0,
    },
    oldPosterVertical => {
        caption         => 'Постер классический',
        design_settings => {
            adaptiveHeight  => '',
            adaptiveWidth   => '',
            borderColor     => 'DDDCDA',
            borderRadius    => 0,
            borderType      => 'block',
            favicon         => 1,
            fontFamily      => '',
            fontSize        => '1.1',
            horizontalAlign => 1,
            hoverColor      => 'DD0000',
            imagesFirst     => 0,
            linksUnderline  => 1,
            name            => 'oldPosterVertical',
            noSitelinks     => 0,
            siteBgColor     => 'FFFFFF',
            sitelinksColor  => '0000CC',
            textColor       => '000000',
            titleColor      => '0000CC',
            titleFontSize   => 3,
            urlColor        => '006600',
        },
        form_factor             => 'vertical',
        is_custom_format_direct => 1,
    },
    posterVertical => {
        caption         => 'Постер новый',
        design_settings => {
            adaptiveHeight  => '',
            adaptiveWidth   => '',
            borderColor     => 'DDDCDA',
            favicon         => 1,
            fontFamily      => '',
            horizontalAlign => 1,
            hoverColor      => 'DD0000',
            linksUnderline  => 1,
            name            => 'posterVertical',
            noSitelinks     => 0,
            siteBgColor     => 'FFFFFF',
            sitelinksColor  => '0000CC',
            textColor       => '000000',
            titleColor      => '0000CC',
            titleFontSize   => 3,
            urlColor        => '006600',
        },
        form_factor             => 'vertical',
        is_custom_format_direct => 0,
    }
);
my $LIMIT = 1000;

my @IGNORE_LOGINS = qw(
  msn-p
  rambler-p
  avitortb2013
  m-ru-text
  m-ru-2015
  wwwodnoklassniki-2015
  vk-rsya
  vk-rsya-apps
  esforce-h
  smi2smi2
  fashionpress-parol
  grazia-pim
  herstshkulevmedia
  maximatelecom
  rbcpartner
  riandir
  mediaforce-rtb
  );

my $opts = {page_limit => 1000,};
my $file_n = 0;

sub args {
    my ($opts) = @_;

    return (
        'black_list!'  => \$opts->{black_list},
        'get_blocks!'  => \$opts->{get_blocks},
        'logins:s@'    => \$opts->{logins},
        'page_ids:s@'  => \$opts->{page_ids},
        'page_limit:s' => \$opts->{page_limit},
    );
}

run(
    sub {
        (my $app, $opts) = @_;

        # Check logins (if provided)
        my @logins = split /\s*,\s*/, join ',', @{$opts->{'logins'} // []};
        if (scalar @logins) {
            my %logins;
            my $users = $app->users->get_all(fields => [qw(id login)], filter => {login => \@logins});
            foreach my $user (@$users) {
                delete $logins{$user->{login}};
            }
            if (scalar keys %logins) {
                print logstr('Login list error:', [sort keys %logins]);
                return;
            }
        }

        my %black_list;
        @black_list{@IGNORE_LOGINS} = ();
        my @incorrect_logins = grep {exists $black_list{$_}} @logins;
        if (scalar @incorrect_logins) {
            print logstr('Incorrect logins:', \@incorrect_logins);
            return;
        }

        # Check black list (and exit if only check is needed)
        my $users = $app->users->get_all(fields => [qw(id login)], filter => {login => \@IGNORE_LOGINS});
        foreach my $user (@$users) {
            delete $black_list{$user->{login}};
        }
        if (scalar keys %black_list) {
            print logstr('Black list error:', [sort keys %black_list]);
            return;
        }
        if ($opts->{black_list}) {
            print logstr 'Check black list option is received. Black list is correct. Done.';
            return;
        }

        my @page_ids = split /\s*,\s*/, join ',', @{$opts->{'page_ids'} // []};
        if ($opts->{'page_ids'} && !scalar @page_ids) {
            print logstr('Incorrect list of pages');
            return;
        }

        my %history;
        my %pages_for_update;
        foreach my $model_name (@MODELS) {
            print logstr('Processing model:', $model_name);
            my $model = $app->$model_name;

            my $filter = [
                AND => [
                    {
                        is_custom_bk_data => 0,
                        multistate        => 'working',
                    },
                    ['page', 'NOT MATCH', ['multistate', '=', 'protected']],
                ]
            ];
            my $pk     = $model->partner_db_table->primary_key();
            my $offset = 0;

            my $page_model = $model->get_page_model();
            my $query      = $app->all_pages->query(
                fields => $app->all_pages->_get_fields_obj([qw(login)]),
                filter => $app->all_pages->get_db_filter(
                    [
                        'AND' => [
                            ['model', '=', $page_model->accessor],
                            (
                                scalar @page_ids
                                ? ['page_id', 'IN', \@page_ids]
                                : ()
                            ),
                        ]
                    ]
                )
              )->join(
                table  => $app->partner_db->all_blocks,
                fields => ['id', 'page_id'],
                filter => [
                    'AND' => [
                        ['model',        '=',  \$model_name],
                        ['site_version', '=',  \'desktop'],
                        ['id',           '<>', \100500],
                        ['id',           '<>', \0],
                    ]
                ],
                join_on => [page_id => '=' => {page_id => $app->partner_db->all_pages}]
              )->order_by(qw(page_id id));

            my $block_list = $query->get_all();
            if (scalar @logins) {
                my %logins = map {$_ => 1} @logins;
                @$block_list = grep {$logins{$_->{login}}} @$block_list;
            } elsif (scalar @IGNORE_LOGINS) {
                my %logins = map {$_ => 1} @IGNORE_LOGINS;
                @$block_list = grep {!$logins{$_->{login}}} @$block_list;
            }

            while (my @block_of_id = splice @$block_list, 0, $LIMIT) {
                my @pairs = ();
                for my $keys (@block_of_id) {
                    push @pairs, ['AND' => [['id', '=', $keys->{'id'}], ['page_id', '=', $keys->{'page_id'}]]];
                }

                my $blocks = $model->get_all(
                    fields   => ['*'],
                    filter   => ['AND' => [['OR' => [@pairs]], $filter]],
                    order_by => $pk,
                );
                my $count = scalar(@$blocks);
                print logstr sprintf('Iteration: [%d .. %d], count: %d', $offset, $LIMIT + $offset - 1, $count);
                $offset += $count;

                foreach my $block (@$blocks) {
                    next if (scalar @{$block->{design_templates} // []} > 10);
                    my $public_id = $model->public_id($block);
                    unless (
                        eval {
                            my $template = check_block($block);
                            if ($template) {
                                print logstr $public_id;
                                return TRUE if $opts->{get_blocks};
                                $history{$public_id} = [];
                                my %formats =
                                  map {$_->{design_settings}{name} => $_} @{$block->{design_templates}};
                                foreach my $add (keys %{$template->{formats}}) {
                                    # На блоке уже есть такой формат
                                    next if exists $formats{$add};

                                    my %settings = hash_transform(
                                        $template->{template}->{design_settings},
                                        [keys %{$ADD{$add}->{design_settings}}]
                                    );
                                    delete $settings{name};
                                    foreach (keys %settings) {
                                        delete $settings{$_} unless defined $settings{$_};
                                    }

                                    %settings = (%{$ADD{$add}->{design_settings}}, %settings);

                                    if ($ADD{$add}->{is_custom_format_direct}) {
                                        $settings{callouts}           = undef;
                                        $settings{urlBackgroundColor} = undef;
                                        $settings{adaptiveHeight}     = '';
                                        $settings{adaptiveWidth}      = '';
                                    }
                                    if ($template->{formats}->{$add}->{limit} eq 'skip') {
                                        $settings{limit} = undef;
                                    } elsif ($template->{formats}->{$add}->{limit} eq 'current') {
                                        $settings{limit} = $template->{template}->{design_settings}->{limit};
                                        $settings{limit} = 4 if $add eq 'posterHorizontal' && $settings{limit} > 4;
                                    } else {
                                        $settings{limit} = $template->{formats}->{$add}->{limit};
                                    }

                                    $settings{adaptiveType} = undef if ($add ne 'modernAdaptive');
                                    $settings{adaptiveHeight} = ''
                                      unless (scalar grep {$add eq $_} qw(horizontal adaptive motion extensibleMobile));
                                    if ($settings{adaptiveHeight} ne '' && $settings{adaptiveHeight} < 170) {
                                        $settings{adaptiveHeight} = 170;
                                    }
                                    $settings{adaptiveWidth} = '';

                                    $settings{adaptiveWidth} = $template->{formats}->{$add}->{adaptiveWidth}
                                      if defined $template->{formats}->{$add}->{adaptiveWidth};
                                    $settings{adaptiveHeight} = $template->{formats}->{$add}->{adaptiveHeight}
                                      if defined $template->{formats}->{$add}->{adaptiveHeight};

                                    my %design = (
                                        block_id                => $block->{id},
                                        caption                 => $ADD{$add}->{caption},
                                        is_custom_format_direct => $ADD{$add}->{is_custom_format_direct},
                                        design_settings         => \%settings,
                                        form_factor             => $ADD{$add}->{form_factor}
                                          // $template->{template}->{form_factor},
                                        page_id => $block->{page_id},
                                    );
                                    push @{$history{$public_id}}, \%design;
                                    push @{$block->{design_templates}}, \%design;
                                }
                                if (scalar @{$history{$public_id}}) {
                                    print logstr $block->{design_templates};
                                    my %validator_template = $model->get_template();
                                    $validator_template{template}{fields} =
                                      {design_templates => $validator_template{template}{fields}{design_templates}};
                                    $app->validator->check(
                                        app   => $model,
                                        data  => $block,
                                        throw => TRUE,
                                        %validator_template,
                                    );
                                    for (my $i = 0; $i < scalar @{$history{$public_id}}; $i++) {
                                        my $id = $app->design_templates->add(%{$history{$public_id}[$i]});
                                        $history{$public_id}[$i] = $id;
                                    }
                                    print logstr sprintf('For block %s following design templates were added: %s',
                                        $public_id, join(', ', @{$history{$public_id}}));
                                    $pages_for_update{$block->{page_id}}++;
                                }
                            }
                            return TRUE;
                        }
                      )
                    {
                        print logstr sprintf('Error while processing block %s: %s', $public_id, $@);
                    }
                }
                print_pages(\%pages_for_update) if (scalar keys %pages_for_update // 0 > $opts->{page_limit});
            }
        }
        # Для отката надо взять данные из этого лога и подставить в запрос:
        # 1) Если откатывать через мультистатус, то:
        #    UPDATE `design_templates` SET `multistate` = 1 WHERE `id` IN (...);
        # 2) Если через полное удаление, то:
        #    DELETE FROM `design_templates_action_log` WHERE `elem_id` IN (...);
        #    DELETE FROM `design_templates` WHERE `id` IN (...);
        print logstr 'Whole added data:', [grep {!ref $_} map {@$_} values %history];
        print_pages(\%pages_for_update);
    }
   );

# Проверяет, нужно ли на блок добавлять дизайны
# Если да, возвращает хэш с первым существующим на блоке дизайном, удовлетворяющим требованиям и список добавляемых форматов
sub check_block {
    my ($block) = @_;

    foreach my $template (@{$block->{design_templates}}) {
        next
          unless my $formats = $FORMATS->{$template->{form_factor}}->{$template->{design_settings}->{name}};
        foreach my $condition (keys %$formats) {
            return {template => $template, formats => $formats->{$condition}->{formats}}
              unless defined $formats->{$condition}->{filter};

            $template->{design_settings}->{adaptiveHeight} //= '';
            if ($formats->{$condition}->{filter} eq 'adaptive_height_gt_290') {

                if (
                    (
                           $template->{design_settings}->{adaptiveHeight} ne ''
                        && $template->{design_settings}->{adaptiveHeight} >= 290
                    )
                    || (   $template->{design_settings}->{adaptiveHeight} eq ''
                        && !$block->{media_blocked}
                        && $block->{dsp_blocks}
                        && scalar(grep {m/x(\d+)$/ && $1 >= 290} @{$block->{dsp_blocks}}))
                   )
                {
                    return {template => $template, formats => $formats->{$condition}->{formats}};
                }
            } elsif ($formats->{$condition}->{filter} eq 'adaptive_height_less_290') {

                if (
                    (
                           $template->{design_settings}->{adaptiveHeight} ne ''
                        && $template->{design_settings}->{adaptiveHeight} < 290
                    )
                    || (   $template->{design_settings}->{adaptiveHeight} eq ''
                        && !$block->{media_blocked}
                        && $block->{dsp_blocks}
                        && !scalar(grep {m/x(\d+)$/ && $1 >= 290} @{$block->{dsp_blocks}}))
                   )
                {
                    return {template => $template, formats => $formats->{$condition}->{formats}};
                }
            } elsif ($formats->{$condition}->{filter} eq 'adaptive_type_gt_290') {
                if (
                    (
                        defined $template->{design_settings}->{adaptiveType}
                        && $template->{design_settings}->{adaptiveType} eq 'l'
                    )
                    || (   !defined $template->{design_settings}->{adaptiveType}
                        && !$block->{media_blocked}
                        && $block->{dsp_blocks}
                        && scalar(grep {m/x(\d+)$/ && $1 >= 290} @{$block->{dsp_blocks}}))
                   )
                {
                    return {template => $template, formats => $formats->{$condition}->{formats}};
                }
            } elsif ($formats->{$condition}->{filter} eq 'adaptive_type_less_290') {
                if (   !defined $template->{design_settings}->{adaptiveType}
                    && !$block->{media_blocked}
                    && $block->{dsp_blocks}
                    && scalar(grep {m/x(\d+)$/ && $1 >= 250} @{$block->{dsp_blocks}}))
                {
                    return {template => $template, formats => $formats->{$condition}->{formats}};
                }

            }
        }
    }
    return undef;
}

sub print_pages {
    my ($pages) = @_;

    my @ids = (keys %$pages);
    print logstr 'Pages for update:', \@ids;
    my $fh;
    if (open($fh, '>', sprintf('%s.%d.page_id', $opts->{file_name}, $file_n++))) {
        print $fh join("\n", @ids, '');
        close $fh;
        %$pages = ();
    } else {
        print logstr 'Can\'t write page ids:', $!;
    }
}
