#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт переносит старые нативные блоки из контентых в новые нативные

=head1 USAGE

  perl -Ilib bin/oneshots/PI-17469_move_old_natural_blocks_from_content_to_new_natural.pl --page_ids=12312,534142 --dry_run
  perl -Ilib bin/oneshots/PI-17469_move_old_natural_blocks_from_content_to_new_natural.pl --dry_run
  perl -Ilib bin/oneshots/PI-17469_move_old_natural_blocks_from_content_to_new_natural.pl \
        --rollback=PI-17469_move_old_natural_blocks_from_content_to_new_natural.out

=head1 OPTIONS

  page_ids   - Список ID площадок через запятую (необязательный)
  rollback   - имя предыдущего лог-файла, откат действий

=cut

use qbit;

use Utils::PublicID 'split_block_public_id';
use Utils::ScriptWrapper;

my %MODEL_MAP = (
    'C-A-' => {
        src_table_name => 'context_on_site_natural',
        dst_table_name => 'context_on_site_content',
    },
    'C-I-' => {
        src_table_name => 'internal_context_on_site_natural',
        dst_table_name => 'internal_context_on_site_content',
    },
);

my @EXTRA_FIELDS = qw(
  adv_positions
  candidates
  grid_columns
  grid_position
  grid_rows
  need_pictures
  sources
  );

sub args {
    my ($opts) = @_;

    return (
        'page_ids:s@' => \$opts->{page_ids},
        'rollback:s'  => \$opts->{rollback},
    );
}
my ($app, $opts);
run(
    sub {
        ($app, $opts) = @_;

        my %page_ids;
        if (defined($opts->{page_ids})) {
            map {$page_ids{$_} = TRUE} split(/,/, join(',', @{$opts->{page_ids}}));
        }

        my %blocks;
        my %pages;
        if ($opts->{rollback}) {
            my $fh;
            if (open($fh, '<', $opts->{rollback})) {
                my %old_blocks;
                while (<$fh>) {
                    if (/Remove block:\s+(\S+)\s+(.+)$/) {
                        my $block_id = $1;
                        my $block    = from_json($2);
                        $old_blocks{$block_id} = $block;
                    }
                }
                close $fh;
                print logstr 'Found blocks:', scalar keys %old_blocks;
                my $modifier = sub {
                    my ($block, $old_block) = @_;

                    return $old_block;
                };
                foreach my $public_id (sort keys %old_blocks) {
                    my ($model, $page_id, $block_id) = split_block_public_id($public_id);
                    my $src_table_name = $MODEL_MAP{$model}{src_table_name};
                    my $dst_table_name = $MODEL_MAP{$model}{dst_table_name};
                    my $src_table      = $app->partner_db->$src_table_name;
                    my $pk             = $src_table->primary_key();
                    my $block_pk       = {
                        $pk->[0] => $page_id,
                        $pk->[1] => $block_id,
                    };
                    move_block($src_table_name, $dst_table_name, $block_pk, $modifier, $old_blocks{$public_id});
                    $blocks{$public_id}++;
                    $pages{$page_id}++;
                }
            } else {
                print logstr 'Can\'t open log-file for rollback: ', $!;
                return;
            }
        } else {
            foreach my $tables ([qw(context_on_site_content context_on_site_natural)],
                [qw(internal_context_on_site_content internal_context_on_site_natural)])
            {
                my ($src_table_name, $dst_table_name) = @$tables;
                my $src_table = $app->partner_db->$src_table_name;

                my $pk               = $src_table->primary_key();
                my $block_list_query = $app->partner_db->query->select(
                    table  => $src_table,
                    fields => $pk,
                    filter => [
                        OR => [
                            [
                                AND => [
                                    [is_custom_bk_data => '=' => \0],
                                    [
                                        {JSON_EXTRACT => [opts => \'$.sources']} => '=' =>
                                          {JSON_ARRAY => [['+' => [\0, \0]]]}
                                    ],
                                    [
                                        {JSON_EXTRACT => [opts => \'$.grid_columns']} => '=' =>
                                          {JSON_EXTRACT => [{JSON_ARRAY => [['+' => [\0, \1]]]}, \'$[0]']}
                                    ],
                                    [
                                        {JSON_EXTRACT => [opts => \'$.grid_rows']} => '=' =>
                                          {JSON_EXTRACT => [{JSON_ARRAY => [['+' => [\0, \1]]]}, \'$[0]']}
                                    ],
                                    (
                                        scalar keys %page_ids ? [$pk->[0] => 'IN' => \[keys %page_ids]]
                                        : ()
                                    ),
                                ]
                            ],
                            [
                                AND => [
                                    [is_custom_bk_data => '=' => \1],
                                    [
                                        {JSON_EXTRACT => [bk_data => \'$.CustomBlockData.WidgetInfo.Sources']} => '=' =>
                                          {JSON_ARRAY => [['+' => [\0, \0]]]}
                                    ],
                                    (
                                        scalar keys %page_ids ? [$pk->[0] => 'IN' => \[keys %page_ids]]
                                        : ()
                                    ),
                                ]
                            ],
                        ]
                    ]
                );
                print $block_list_query->get_sql_with_data(), "\n";
                my $block_list = $block_list_query->get_all();
                print logstr 'Found blocks:', scalar @$block_list;

                my $modifier = sub {
                    my ($block) = @_;

                    my $opts = from_json $block->{opts};
                    delete @$opts{@EXTRA_FIELDS};
                    $block->{opts} = to_json $opts;

                    if ($block->{bk_data}) {
                        my $bk_data = from_json $block->{bk_data};
                        $bk_data->{BlockModel} = $dst_table_name;
                        $block->{bk_data}      = to_json $bk_data;
                    }

                    return $block;
                };

                foreach my $block_pk (@$block_list) {
                    move_block($src_table_name, $dst_table_name, $block_pk, $modifier);
                    $blocks{$app->$src_table_name->public_id($block_pk)}++;
                    $pages{$block_pk->{$pk->[0]}}++;
                }
            }
        }
        print logstr 'Following blocks are moved:',   join ',', sort keys %blocks;
        print logstr 'Following pages are affected:', join ',', sort keys %pages;
    }
   );

sub move_block {
    my ($src_name, $dst_name, $block_pk, $modifier, $param) = @_;

    my $src_table     = $app->partner_db->$src_name;
    my $dst_table     = $app->partner_db->$dst_name;
    my $src_log_name  = $src_name . '_action_log';
    my $dst_log_name  = $dst_name . '_action_log';
    my $src_log_table = $app->partner_db->$src_log_name;
    my $dst_log_table = $app->partner_db->$dst_log_name;
    $app->partner_db->transaction(
        sub {
            # Copy block
            my $block = $src_table->get($block_pk, fields => [$src_table->field_names()]);
            print logstr 'Remove block:', $app->$src_name->public_id($block_pk), $block;
            $block = $modifier->($block, $param);
            print logstr 'Add block:', $app->$dst_name->public_id($block_pk), $block;
            $dst_table->add($block) unless $opts->{dry_run};

            # Copy logs
            my $action_logs_filter =
              $app->partner_db->filter([AND => [map {['elem_' . $_ => '=' => \$block_pk->{$_}]} keys %$block_pk]]);
            my $action_logs_query = $app->partner_db->query->select(
                table  => $src_log_table,
                fields => [grep {$_ ne 'id'} $src_log_table->field_names()],
                filter => $action_logs_filter,
            )->order_by('id');
            my $action_logs = $action_logs_query->get_all();
            print logstr 'Copy logs:', scalar @$action_logs;
            $dst_log_table->add_multi($action_logs) if (!$opts->{dry_run} && scalar @$action_logs);

            # Remove source logs
            $src_log_table->delete($action_logs_filter) unless $opts->{dry_run};

            # Remove source block
            $src_table->delete($block_pk) unless $opts->{dry_run};
        }
    );
}
