#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт создает RTB блоки из настроек плоского Директа, которые представлены в виде файла формата paje_id domain json_with_settings

=head1 USAGE

  perl -Ilib bin/oneshots/PI-17516_create_rtb_from_flat_direct.pl --block_settings=file.txt
  perl -Ilib bin/oneshots/PI-17516_create_rtb_from_flat_direct.pl --rollback=previous_log.txt

=head1 OPTIONS

  block_settings - файл с настройками блоков
  rollback       - log-файл для архивирования ранее добавленных блоков

=cut

use qbit;

use Utils::ScriptWrapper;
use Math::Round;

sub args {
    my ($opts) = @_;

    return (
        'block_settings:s' => \$opts->{block_settings},
        'rollback:s'       => \$opts->{rollback},
    );
}

# Конвертация форматов, неподдерживаемых МСФ в Адаптивный
# https://st.yandex-team.ru/PI-17516?#5d80edd38c7003001ec6c1dc
my $format_mapping = {
    "160x450"        => "modernAdaptive",
    "180x150"        => "modernAdaptive",
    "234x60"         => "modernAdaptive",
    "468x60"         => "modernAdaptive",
    "600x60"         => "modernAdaptive",
    "auto"           => "modernAdaptive",
    "flat"           => "modernAdaptive",
    "flat, 300x300," => "300x300",
    "Horizontal"     => "horizontal",
    "mediaTouch"     => "modernAdaptive",
    "mobile"         => "modernAdaptive",
    #PI-17738
    "newPosterVertical"   => "posterVertical",
    "newPosterHorizontal" => "posterHorizontal",
    "oldHorizontal"       => "horizontal",
    "horizontal0318"      => "horizontal",
    "vertical0318"        => "vertical",
};

my $PAGES_LIST = {
    3664   => 1,
    7024   => 1,
    9042   => 1,
    10518  => 1,
    49105  => 1,
    55283  => 1,
    60918  => 1,
    80073  => 1,
    121365 => 1,
    135620 => 1,
    136546 => 1,
    139798 => 1,
    149851 => 1,
    9524   => 1,
    35333  => 1,
    53141  => 1,
    57314  => 1,
    60635  => 1,
    62092  => 1,
    62815  => 1,
    64104  => 1,
    65327  => 1,
    68252  => 1,
    84814  => 1,
    89439  => 1,
    93605  => 1,
    96883  => 1,
    98894  => 1,
    110927 => 1,
    113401 => 1,
    113496 => 1,
    118466 => 1,
    119826 => 1,
    122619 => 1,
    124890 => 1,
    126298 => 1,
    127330 => 1,
    127860 => 1,
    130886 => 1,
    131153 => 1,
    134720 => 1,
    137031 => 1,
    137232 => 1,
    139370 => 1,
    142542 => 1,
    144103 => 1,
    147793 => 1,
    148519 => 1,
    153757 => 1,
    166986 => 1,
    184208 => 1,
    189184 => 1,
    201195 => 1,
    225360 => 1,
    331648 => 1,
    1056   => 1,
    1800   => 1,
    2239   => 1,
    4248   => 1,
    6561   => 1,
    11119  => 1,
    14860  => 1,
    19075  => 1,
    24230  => 1,
    30013  => 1,
    32149  => 1,
    40633  => 1,
    41418  => 1,
    45404  => 1,
    47928  => 1,
    48243  => 1,
    49087  => 1,
    53293  => 1,
    54075  => 1,
    54863  => 1,
    56527  => 1,
    57984  => 1,
    60569  => 1,
    62247  => 1,
    64532  => 1,
    64914  => 1,
    68707  => 1,
    70313  => 1,
    78298  => 1,
    80011  => 1,
    82767  => 1,
    82840  => 1,
    86696  => 1,
    89299  => 1,
    90653  => 1,
    91347  => 1,
    91765  => 1,
    92651  => 1,
    93496  => 1,
    94916  => 1,
    98331  => 1,
    105923 => 1,
    113982 => 1,
    115434 => 1,
    115731 => 1,
    117618 => 1,
    117950 => 1,
    118588 => 1,
    120515 => 1,
    122612 => 1,
    124761 => 1,
    125987 => 1,
    126485 => 1,
    126557 => 1,
    127707 => 1,
    132703 => 1,
    134872 => 1,
    135358 => 1,
    135361 => 1,
    135553 => 1,
    135846 => 1,
    140360 => 1,
    141193 => 1,
    141277 => 1,
    141462 => 1,
    142172 => 1,
    146574 => 1,
    147152 => 1,
    150192 => 1,
    154032 => 1,
    154155 => 1,
    155721 => 1,
    160949 => 1,
    163057 => 1,
    164861 => 1,
    166989 => 1,
    175133 => 1,
    180326 => 1,
    182841 => 1,
    185005 => 1,
    185913 => 1,
    187630 => 1,
    188255 => 1,
    190721 => 1,
    221153 => 1,
    320875 => 1,
    1451   => 1,
    1823   => 1,
    2281   => 1,
    5580   => 1,
    5828   => 1,
    7327   => 1,
    7498   => 1,
    8539   => 1,
    10578  => 1,
    11422  => 1,
    14815  => 1,
    15049  => 1,
    15715  => 1,
    17376  => 1,
    17646  => 1,
    18144  => 1,
    19384  => 1,
    19393  => 1,
    21256  => 1,
    26792  => 1,
    28344  => 1,
    28898  => 1,
    29789  => 1,
    33812  => 1,
    37065  => 1,
    38762  => 1,
    45272  => 1,
    46321  => 1,
    47290  => 1,
    47476  => 1,
    48523  => 1,
    50025  => 1,
    52012  => 1,
    52080  => 1,
    53185  => 1,
    54855  => 1,
    55145  => 1,
    55267  => 1,
    55602  => 1,
    57336  => 1,
    57440  => 1,
    58151  => 1,
    58332  => 1,
    58842  => 1,
    60312  => 1,
    62372  => 1,
    62899  => 1,
    63696  => 1,
    63954  => 1,
    64074  => 1,
    64652  => 1,
    64805  => 1,
    65888  => 1,
    66032  => 1,
    67378  => 1,
    67812  => 1,
    68949  => 1,
    69058  => 1,
    69887  => 1,
    69983  => 1,
    70002  => 1,
    70831  => 1,
    70907  => 1,
    71179  => 1,
    72734  => 1,
    73880  => 1,
    75263  => 1,
    75532  => 1,
    79253  => 1,
    79481  => 1,
    81249  => 1,
    82663  => 1,
    82783  => 1,
    83544  => 1,
    83653  => 1,
    84758  => 1,
    85287  => 1,
    87075  => 1,
    88233  => 1,
    89897  => 1,
    91813  => 1,
    92720  => 1,
    93177  => 1,
    94008  => 1,
    97552  => 1,
    97839  => 1,
    98006  => 1,
    98473  => 1,
    101211 => 1,
    102413 => 1,
    103424 => 1,
    103815 => 1,
    103850 => 1,
    104644 => 1,
    107279 => 1,
    107430 => 1,
    108468 => 1,
    108601 => 1,
    109074 => 1,
    110915 => 1,
    112093 => 1,
    112930 => 1,
    113114 => 1,
    113132 => 1,
    113785 => 1,
    114665 => 1,
    114738 => 1,
    115473 => 1,
    116192 => 1,
    116281 => 1,
    117295 => 1,
    117311 => 1,
    117637 => 1,
    120546 => 1,
    121198 => 1,
    122865 => 1,
    123164 => 1,
    123498 => 1,
    123901 => 1,
    125340 => 1,
    126587 => 1,
    127364 => 1,
    128976 => 1,
    129433 => 1,
    129538 => 1,
    130121 => 1,
    130609 => 1,
    130663 => 1,
    131213 => 1,
    131581 => 1,
    132240 => 1,
    134579 => 1,
    134806 => 1,
    134884 => 1,
    134901 => 1,
    134917 => 1,
    135041 => 1,
    136040 => 1,
    136388 => 1,
    136580 => 1,
    137722 => 1,
    137784 => 1,
    137919 => 1,
    137930 => 1,
    138137 => 1,
    139660 => 1,
    139817 => 1,
    140292 => 1,
    140794 => 1,
    141213 => 1,
    142800 => 1,
    142870 => 1,
    142901 => 1,
    143120 => 1,
    143578 => 1,
    144033 => 1,
    144554 => 1,
    144618 => 1,
    144634 => 1,
    145368 => 1,
    145627 => 1,
    147330 => 1,
    149006 => 1,
    149660 => 1,
    150231 => 1,
    150250 => 1,
    150283 => 1,
    150474 => 1,
    150597 => 1,
    150743 => 1,
    152589 => 1,
    153760 => 1,
    153866 => 1,
    154136 => 1,
    154168 => 1,
    154828 => 1,
    154876 => 1,
    155073 => 1,
    155550 => 1,
    155860 => 1,
    159024 => 1,
    159995 => 1,
    162205 => 1,
    162520 => 1,
    163671 => 1,
    165787 => 1,
    167871 => 1,
    168569 => 1,
    171772 => 1,
    174756 => 1,
    174821 => 1,
    175921 => 1,
    176167 => 1,
    184753 => 1,
    185112 => 1,
    185255 => 1,
    185601 => 1,
    185805 => 1,
    185940 => 1,
    186014 => 1,
    186116 => 1,
    186356 => 1,
    186368 => 1,
    186502 => 1,
    186552 => 1,
    186780 => 1,
    186935 => 1,
    187858 => 1,
    188569 => 1,
    188842 => 1,
    215097 => 1,
    230518 => 1,
    231639 => 1,
    264228 => 1,
    273594 => 1,
    321768 => 1,
    353528 => 1,
    399456 => 1,
    401886 => 1,
    1310   => 1,
    1435   => 1,
    1564   => 1,
    1588   => 1,
    1643   => 1,
    1826   => 1,
    1834   => 1,
    1946   => 1,
    2662   => 1,
    2743   => 1,
    3006   => 1,
    3104   => 1,
    3132   => 1,
    3480   => 1,
    3731   => 1,
    4416   => 1,
    4907   => 1,
    5789   => 1,
    6506   => 1,
    7505   => 1,
    7677   => 1,
    7709   => 1,
    7843   => 1,
    8003   => 1,
    9008   => 1,
    9065   => 1,
    9589   => 1,
    9599   => 1,
    9790   => 1,
    9841   => 1,
    10668  => 1,
    11379  => 1,
    11704  => 1,
    12306  => 1,
    12454  => 1,
    12785  => 1,
    12859  => 1,
    13449  => 1,
    14091  => 1,
    14119  => 1,
    15026  => 1,
    15411  => 1,
    15714  => 1,
    16059  => 1,
    18397  => 1,
    19510  => 1,
    19571  => 1,
    19677  => 1,
    20986  => 1,
    21792  => 1,
    21921  => 1,
    23518  => 1,
    23934  => 1,
    24065  => 1,
    25001  => 1,
    25089  => 1,
    26782  => 1,
    27794  => 1,
    27952  => 1,
    28014  => 1,
    28172  => 1,
    28471  => 1,
    29026  => 1,
    29749  => 1,
    30306  => 1,
    30417  => 1,
    30591  => 1,
    31188  => 1,
    31681  => 1,
    31827  => 1,
    32444  => 1,
    32874  => 1,
    34442  => 1,
    35642  => 1,
    36346  => 1,
    36384  => 1,
    36615  => 1,
    37019  => 1,
    37240  => 1,
    39141  => 1,
    39614  => 1,
    39685  => 1,
    40419  => 1,
    40855  => 1,
    40870  => 1,
    42319  => 1,
    42368  => 1,
    44040  => 1,
    44042  => 1,
    44077  => 1,
    44080  => 1,
    44212  => 1,
    44667  => 1,
    45493  => 1,
    45623  => 1,
    45766  => 1,
    46902  => 1,
    46965  => 1,
    47448  => 1,
    48042  => 1,
    48121  => 1,
    48364  => 1,
    49528  => 1,
    49533  => 1,
    49703  => 1,
    49707  => 1,
    50097  => 1,
    50196  => 1,
    50268  => 1,
    51427  => 1,
    51495  => 1,
    51797  => 1,
    52001  => 1,
    52182  => 1,
    52962  => 1,
    52974  => 1,
    53060  => 1,
    53148  => 1,
    53167  => 1,
    53475  => 1,
    53556  => 1,
    53727  => 1,
    53765  => 1,
    53847  => 1,
    53889  => 1,
    54009  => 1,
    54047  => 1,
    54444  => 1,
    54581  => 1,
    54870  => 1,
    54878  => 1,
    56058  => 1,
    56251  => 1,
    56335  => 1,
    56559  => 1,
    56699  => 1,
    56816  => 1,
    56863  => 1,
    56889  => 1,
    56926  => 1,
    57108  => 1,
    57313  => 1,
    57881  => 1,
    58199  => 1,
    58417  => 1,
    58670  => 1,
    58678  => 1,
    58915  => 1,
    59119  => 1,
    59189  => 1,
    59278  => 1,
    59310  => 1,
    59337  => 1,
    59512  => 1,
    59584  => 1,
    59907  => 1,
    60376  => 1,
    60481  => 1,
    60500  => 1,
    60505  => 1,
    61204  => 1,
    61423  => 1,
    61488  => 1,
    61566  => 1,
    61680  => 1,
    61805  => 1,
    61910  => 1,
    62513  => 1,
    62550  => 1,
    62679  => 1,
    62883  => 1,
    62896  => 1,
    62950  => 1,
    63214  => 1,
    63426  => 1,
    63960  => 1,
    63979  => 1,
    64065  => 1,
    64146  => 1,
    64151  => 1,
    64230  => 1,
    64352  => 1,
    64613  => 1,
    64631  => 1,
    64644  => 1,
    64749  => 1,
    64757  => 1,
    64959  => 1,
    65076  => 1,
    65168  => 1,
    65926  => 1,
    65928  => 1,
    66006  => 1,
    66073  => 1,
    66188  => 1,
    66275  => 1,
    67611  => 1,
    67806  => 1,
    68128  => 1,
    68237  => 1,
    68491  => 1,
    68750  => 1,
    68884  => 1,
    69043  => 1,
    69238  => 1,
    70718  => 1,
    70913  => 1,
    71028  => 1,
    71188  => 1,
    71259  => 1,
    71555  => 1,
    71740  => 1,
    72026  => 1,
    72456  => 1,
    72895  => 1,
    72957  => 1,
    73013  => 1,
    73539  => 1,
    73867  => 1,
    74156  => 1,
    74182  => 1,
    74674  => 1,
    74744  => 1,
    75163  => 1,
    75467  => 1,
    75504  => 1,
    75624  => 1,
    75990  => 1,
    76526  => 1,
    76674  => 1,
    76747  => 1,
    77071  => 1,
    77109  => 1,
    77145  => 1,
    77899  => 1,
    77929  => 1,
    78290  => 1,
    78608  => 1,
    78675  => 1,
    79104  => 1,
    79334  => 1,
    79500  => 1,
    79699  => 1,
    79871  => 1,
    80053  => 1,
    80074  => 1,
    80108  => 1,
    80167  => 1,
    80413  => 1,
    80438  => 1,
    80642  => 1,
    80777  => 1,
    80786  => 1,
    81055  => 1,
    81157  => 1,
    81425  => 1,
    81444  => 1,
    81546  => 1,
    81595  => 1,
    81645  => 1,
    81688  => 1,
    81809  => 1,
    81836  => 1,
    82285  => 1,
    82367  => 1,
    82427  => 1,
    82684  => 1,
    82691  => 1,
    82733  => 1,
    83028  => 1,
    83079  => 1,
    83194  => 1,
    83414  => 1,
    83494  => 1,
    83503  => 1,
    83658  => 1,
    83929  => 1,
    84278  => 1,
    84322  => 1,
    84470  => 1,
    84525  => 1,
    84723  => 1,
    84854  => 1,
    85115  => 1,
    85704  => 1,
    85715  => 1,
    85727  => 1,
    85775  => 1,
    85789  => 1,
    86016  => 1,
    87025  => 1,
    87123  => 1,
    87124  => 1,
    87369  => 1,
    87397  => 1,
    88351  => 1,
    88379  => 1,
    88716  => 1,
    88757  => 1,
    89201  => 1,
    89330  => 1,
    89898  => 1,
    89925  => 1,
    90065  => 1,
    90199  => 1,
    90384  => 1,
    90684  => 1,
    91534  => 1,
    91829  => 1,
    92082  => 1,
    92106  => 1,
    92489  => 1,
    92537  => 1,
    92839  => 1,
    92861  => 1,
    93092  => 1,
    93653  => 1,
    93660  => 1,
    93686  => 1,
    93697  => 1,
    93901  => 1,
    93950  => 1,
    93977  => 1,
    94180  => 1,
    94362  => 1,
    94365  => 1,
    94568  => 1,
    94769  => 1,
    94780  => 1,
    94912  => 1,
    95357  => 1,
    95380  => 1,
    95930  => 1,
    96301  => 1,
    96475  => 1,
    96484  => 1,
    96895  => 1,
    97468  => 1,
    97627  => 1,
    97643  => 1,
    97662  => 1,
    97673  => 1,
    97739  => 1,
    97802  => 1,
    97954  => 1,
    98038  => 1,
    98055  => 1,
    98109  => 1,
    98168  => 1,
    98174  => 1,
    98716  => 1,
    99027  => 1,
    99612  => 1,
    99758  => 1,
    100333 => 1,
    100443 => 1,
    100986 => 1,
    101065 => 1,
    101068 => 1,
    101197 => 1,
    102231 => 1,
    102340 => 1,
    102454 => 1,
    102456 => 1,
    102500 => 1,
    103289 => 1,
    103447 => 1,
    103616 => 1,
    103668 => 1,
    103725 => 1,
    103995 => 1,
    104057 => 1,
    104931 => 1,
    105296 => 1,
    105412 => 1,
    105486 => 1,
    105696 => 1,
    105839 => 1,
    105903 => 1,
    106588 => 1,
    106721 => 1,
    107236 => 1,
    107261 => 1,
    107271 => 1,
    107294 => 1,
    107348 => 1,
    107386 => 1,
    107502 => 1,
    107503 => 1,
    107647 => 1,
    107836 => 1,
    107857 => 1,
    107859 => 1,
    107919 => 1,
    107957 => 1,
    108302 => 1,
    108364 => 1,
    108558 => 1,
    109067 => 1,
    109083 => 1,
    109663 => 1,
    109723 => 1,
    109728 => 1,
    110073 => 1,
    110450 => 1,
    110619 => 1,
    111351 => 1,
    111906 => 1,
    111967 => 1,
    111979 => 1,
    112094 => 1,
    112337 => 1,
    112818 => 1,
    112860 => 1,
    112954 => 1,
    113120 => 1,
    113508 => 1,
    113554 => 1,
    113647 => 1,
    113717 => 1,
    113719 => 1,
    113828 => 1,
    113907 => 1,
    114302 => 1,
    114405 => 1,
    114629 => 1,
    114650 => 1,
    114704 => 1,
    114903 => 1,
    115445 => 1,
    115664 => 1,
    115759 => 1,
    115804 => 1,
    115852 => 1,
    116580 => 1,
    117023 => 1,
    117143 => 1,
    117254 => 1,
    117628 => 1,
    117942 => 1,
    117954 => 1,
    118292 => 1,
    118293 => 1,
    118549 => 1,
    118842 => 1,
    118891 => 1,
    119182 => 1,
    119493 => 1,
    119515 => 1,
    119536 => 1,
    119640 => 1,
    119768 => 1,
    119810 => 1,
    120726 => 1,
    121045 => 1,
    122039 => 1,
    122170 => 1,
    122531 => 1,
    122607 => 1,
    122622 => 1,
    122678 => 1,
    123055 => 1,
    123551 => 1,
    124285 => 1,
    124313 => 1,
    124383 => 1,
    124794 => 1,
    124812 => 1,
    124919 => 1,
    125348 => 1,
    125596 => 1,
    125679 => 1,
    125875 => 1,
    126262 => 1,
    126379 => 1,
    126662 => 1,
    126888 => 1,
    126944 => 1,
    126952 => 1,
    127044 => 1,
    127265 => 1,
    127408 => 1,
    127525 => 1,
    127582 => 1,
    127721 => 1,
    127760 => 1,
    127886 => 1,
    128084 => 1,
    128337 => 1,
    128401 => 1,
    128421 => 1,
    128896 => 1,
    129023 => 1,
    129084 => 1,
    129085 => 1,
    129364 => 1,
    129687 => 1,
    129693 => 1,
    129725 => 1,
    129806 => 1,
    129934 => 1,
    130115 => 1,
    130178 => 1,
    130213 => 1,
    130471 => 1,
    130530 => 1,
    130576 => 1,
    130622 => 1,
    130735 => 1,
    130739 => 1,
    130740 => 1,
    130781 => 1,
    130850 => 1,
    131317 => 1,
    131318 => 1,
    131331 => 1,
    131345 => 1,
    131832 => 1,
    131942 => 1,
    132093 => 1,
    132257 => 1,
    132297 => 1,
    132730 => 1,
    132761 => 1,
    132999 => 1,
    133045 => 1,
    134312 => 1,
    134569 => 1,
    134666 => 1,
    134775 => 1,
    134828 => 1,
    134936 => 1,
    135158 => 1,
    135503 => 1,
    135575 => 1,
    135628 => 1,
    135642 => 1,
    135679 => 1,
    135681 => 1,
    135771 => 1,
    135828 => 1,
    135874 => 1,
    136102 => 1,
    136126 => 1,
    136211 => 1,
    136279 => 1,
    136369 => 1,
    136387 => 1,
    136490 => 1,
    136665 => 1,
    136830 => 1,
    137122 => 1,
    137417 => 1,
    137420 => 1,
    137619 => 1,
    137716 => 1,
    138018 => 1,
    138019 => 1,
    138066 => 1,
    138237 => 1,
    138260 => 1,
    138409 => 1,
    138464 => 1,
    138481 => 1,
    138585 => 1,
    138587 => 1,
    138626 => 1,
    138649 => 1,
    138708 => 1,
    138736 => 1,
    138740 => 1,
    138742 => 1,
    138901 => 1,
    138906 => 1,
    138952 => 1,
    139075 => 1,
    139146 => 1,
    139175 => 1,
    139192 => 1,
    139239 => 1,
    139242 => 1,
    139253 => 1,
    139283 => 1,
    139285 => 1,
    139299 => 1,
    139323 => 1,
    139347 => 1,
    139389 => 1,
    139433 => 1,
    139530 => 1,
    139695 => 1,
    139706 => 1,
    139751 => 1,
    139806 => 1,
    139807 => 1,
    139903 => 1,
    139985 => 1,
    140031 => 1,
    140351 => 1,
    141054 => 1,
    141057 => 1,
    141215 => 1,
    141243 => 1,
    141358 => 1,
    141569 => 1,
    141692 => 1,
    141698 => 1,
    141722 => 1,
    141741 => 1,
    141752 => 1,
    141894 => 1,
    142018 => 1,
    142073 => 1,
    142126 => 1,
    142167 => 1,
    142168 => 1,
    142211 => 1,
    142289 => 1,
    142376 => 1,
    142601 => 1,
    142629 => 1,
    142733 => 1,
    142759 => 1,
    142764 => 1,
    142785 => 1,
    142824 => 1,
    142833 => 1,
    142849 => 1,
    142920 => 1,
    142992 => 1,
    143095 => 1,
    143115 => 1,
    143140 => 1,
    143141 => 1,
    143146 => 1,
    143260 => 1,
    143273 => 1,
    143320 => 1,
    143588 => 1,
    143599 => 1,
    143668 => 1,
    143753 => 1,
    143794 => 1,
    143821 => 1,
    143857 => 1,
    143934 => 1,
    144053 => 1,
    144496 => 1,
    144571 => 1,
    144713 => 1,
    144732 => 1,
    144767 => 1,
    144776 => 1,
    144780 => 1,
    144848 => 1,
    145242 => 1,
    145364 => 1,
    145378 => 1,
    145409 => 1,
    145680 => 1,
    145780 => 1,
    146253 => 1,
    146254 => 1,
    146286 => 1,
    146745 => 1,
    146927 => 1,
    147034 => 1,
    147129 => 1,
    147148 => 1,
    147277 => 1,
    147567 => 1,
    147593 => 1,
    147814 => 1,
    147815 => 1,
    147899 => 1,
    148002 => 1,
    148090 => 1,
    148265 => 1,
    148372 => 1,
    148528 => 1,
    148851 => 1,
    148852 => 1,
    148871 => 1,
    149001 => 1,
    149279 => 1,
    149551 => 1,
    149698 => 1,
    149896 => 1,
    150366 => 1,
    150594 => 1,
    150683 => 1,
    150763 => 1,
    151523 => 1,
    151836 => 1,
    152117 => 1,
    152123 => 1,
    152139 => 1,
    152157 => 1,
    152422 => 1,
    152423 => 1,
    152429 => 1,
    152466 => 1,
    152565 => 1,
    152600 => 1,
    152620 => 1,
    153758 => 1,
    153794 => 1,
    153802 => 1,
    153805 => 1,
    153808 => 1,
    153962 => 1,
    154029 => 1,
    154061 => 1,
    154157 => 1,
    154330 => 1,
    154340 => 1,
    154427 => 1,
    154814 => 1,
    154987 => 1,
    155058 => 1,
    155158 => 1,
    155192 => 1,
    155227 => 1,
    155367 => 1,
    155584 => 1,
    155657 => 1,
    155690 => 1,
    155714 => 1,
    158834 => 1,
    158842 => 1,
    158880 => 1,
    158941 => 1,
    158951 => 1,
    159329 => 1,
    159654 => 1,
    160394 => 1,
    160623 => 1,
    160954 => 1,
    160957 => 1,
    161542 => 1,
    162261 => 1,
    162372 => 1,
    162382 => 1,
    162610 => 1,
    162862 => 1,
    163042 => 1,
    163055 => 1,
    163355 => 1,
    163638 => 1,
    164870 => 1,
    164902 => 1,
    164991 => 1,
    165256 => 1,
    165414 => 1,
    166693 => 1,
    167518 => 1,
    167596 => 1,
    168204 => 1,
    168344 => 1,
    168437 => 1,
    170610 => 1,
    171718 => 1,
    171972 => 1,
    172268 => 1,
    172855 => 1,
    173674 => 1,
    175006 => 1,
    175010 => 1,
    175011 => 1,
    175720 => 1,
    175920 => 1,
    176294 => 1,
    176760 => 1,
    177672 => 1,
    177775 => 1,
    178081 => 1,
    178607 => 1,
    178612 => 1,
    179700 => 1,
    180239 => 1,
    180306 => 1,
    180537 => 1,
    182458 => 1,
    182954 => 1,
    183509 => 1,
    183622 => 1,
    183993 => 1,
    184202 => 1,
    184415 => 1,
    184590 => 1,
    184708 => 1,
    184741 => 1,
    184924 => 1,
    184997 => 1,
    185067 => 1,
    185105 => 1,
    185152 => 1,
    185238 => 1,
    185252 => 1,
    185267 => 1,
    185269 => 1,
    185413 => 1,
    185433 => 1,
    185503 => 1,
    185515 => 1,
    185535 => 1,
    185537 => 1,
    185600 => 1,
    185602 => 1,
    185637 => 1,
    185726 => 1,
    185793 => 1,
    185888 => 1,
    185998 => 1,
    186009 => 1,
    186035 => 1,
    186038 => 1,
    186046 => 1,
    186233 => 1,
    186291 => 1,
    186469 => 1,
    186495 => 1,
    186529 => 1,
    186563 => 1,
    186714 => 1,
    186818 => 1,
    186872 => 1,
    186882 => 1,
    186897 => 1,
    187096 => 1,
    187120 => 1,
    187194 => 1,
    187633 => 1,
    187683 => 1,
    187715 => 1,
    187837 => 1,
    187903 => 1,
    187904 => 1,
    187932 => 1,
    187978 => 1,
    188195 => 1,
    188257 => 1,
    188333 => 1,
    188349 => 1,
    188353 => 1,
    188464 => 1,
    188476 => 1,
    188478 => 1,
    188533 => 1,
    188776 => 1,
    188835 => 1,
    188853 => 1,
    188941 => 1,
    189011 => 1,
    189035 => 1,
    189285 => 1,
    189290 => 1,
    189302 => 1,
    189565 => 1,
    190009 => 1,
    190502 => 1,
    190756 => 1,
    192736 => 1,
    201717 => 1,
    203092 => 1,
    203490 => 1,
    208747 => 1,
    213812 => 1,
    214592 => 1,
    214801 => 1,
    216469 => 1,
    216817 => 1,
    220284 => 1,
    224379 => 1,
    224910 => 1,
    227235 => 1,
    228198 => 1,
    230473 => 1,
    230516 => 1,
    233710 => 1,
    243633 => 1,
    247111 => 1,
    251570 => 1,
    251782 => 1,
    251785 => 1,
    251789 => 1,
    251790 => 1,
    251996 => 1,
    257090 => 1,
    264092 => 1,
    264093 => 1,
    265178 => 1,
    265180 => 1,
    265182 => 1,
    268090 => 1,
    272123 => 1,
    273503 => 1,
    274434 => 1,
    276372 => 1,
    278332 => 1,
    280562 => 1,
    281637 => 1,
    284716 => 1,
    285148 => 1,
    285152 => 1,
    295054 => 1,
    325289 => 1,
    331650 => 1,
    331852 => 1,
    343240 => 1,
    346073 => 1,
    353177 => 1,
    353516 => 1,
    353525 => 1,
    353526 => 1,
    383831 => 1,
    385786 => 1,
    386734 => 1,
    397547 => 1,
};

my $block_template = {
    "page_id"          => 0,
    "caption"          => "Для замены устаревшего блока Директа",
    "design_templates" => [
        {
            "design_settings"         => {},
            "form_factor"             => "vertical",
            "caption"                 => gettext("default template"),
            "is_custom_format_direct" => 0
        }
    ],
    "site_version" => "desktop",
    "strategy"     => 1,
    "media_block"  => "300x600",
    "blind"        => 0,
    "dsp_blocks"   =>,
    ["160x600", "240x400", "240x600", "300x300", "300x500", "300x600"],
    "media_active"    => undef,
    "media_blocked"   => undef,
    "media_cpm"       => undef,
    "percent_traffic" => undef,
    "text_active"     => undef,
    "text_blocked"    => undef,
    "text_cpm"        => undef,
    "mincpm"          => undef
};

my $added_blocks = {};

my ($app, $opts);
run(
    sub {
        ($app, $opts) = @_;

        my %blocks;
        my %pages;
        if ($opts->{block_settings}) {
            my $fh;
            if (open($fh, '<', $opts->{block_settings})) {

                my $fields_map =
                  {%{$app->context_on_site_rtb->get_design_fields_map()}, width => 'width', height => 'height'};
                while (<$fh>) {
                    my ($page_id, $domain, $settings) = split "\t", $_;
                    $page_id =~ s/"//g;
                    next unless $PAGES_LIST->{$page_id};
                    $settings =~ s/":"true"/":true/g;
                    $settings =~ s/":"false"/":false/g;
                    $settings = from_json($settings);
                    $settings->{direct_block} = $format_mapping->{$settings->{type}} // $settings->{type};
                    my $block = clone $block_template;
                    $block->{page_id} = $page_id;
                    my $page_nfo = $app->context_on_site_campaign->get_all(
                        filter => {page_id => $page_id},
                        fields => [qw(owner_id page_id multistate_name)]
                    );

                    unless (@$page_nfo) {
                        print logstr "NOT FOUND page_id=$page_id";
                        next;
                    }
                    if ($page_nfo->[0]->{owner_id} == 28166288) {
                        print logstr "Login m-ru-text blocked for adding RTB blocks. page_id=$page_id";
                        next;
                    }
                    if ($page_nfo->[0]->{multistate_name} =~ /Заблокирована/) {
                        print logstr "BLOCKED page_id=$page_id";
                        next;
                    }
                    if ($page_nfo->[0]->{multistate_name} =~ /Заархивирована/) {
                        print logstr "DELETED page_id=$page_id";
                        next;
                    }
                    #Удаляем неподдерживаемые параметры
                    $settings = {hash_transform($settings, [keys $fields_map])};
                    foreach (keys %$settings) {
                        $settings->{$fields_map->{$_}} = delete $settings->{$_};

                        if (exists $settings->{borderType} && $settings->{borderType} eq 'inline') {
                            $settings->{borderType} = 'block';
                        }
                        if (exists $settings->{borderType} && $settings->{borderType} eq 'none') {
                            delete $settings->{$_} foreach (qw(borderColor borderRadius));
                        }
                        #Приводим цвет к верхнему регистру, вырезаем #, добавляем недостающие символы, отрезаем лишние
                        if ($_ =~ /color/) {
                            $settings->{$fields_map->{$_}} =~ s/(.*)/\U$1/g;
                            $settings->{$fields_map->{$_}} =~ s/#//g;
                            $settings->{$fields_map->{$_}} =~ s/^(.)(.)(.)\z/$1$1$2$2$3$3/g;
                            $settings->{$fields_map->{$_}} =~ s/^(.){1,5}\z/$1$1$1$1$1$1/g;
                            $settings->{$fields_map->{$_}} = substr $settings->{$fields_map->{$_}}, 0, 6
                              if length $settings->{$fields_map->{$_}} > 6;
                            delete $settings->{$fields_map->{$_}} if $settings->{$fields_map->{$_}} eq 'NONE';
                        }
                    }
                    #Исправляем названия шрифтов
                    $settings->{fontFamily} = 'verdana'
                      if exists $settings->{fontFamily} && $settings->{fontFamily} eq 'Verdana,Helvetica';
                    #Получаем настройки формата из МСФ
                    my $format;
                    try {
                        $format = $app->api_format_system->formats(
                            format => $settings->{name},
                            role   => 'partner',
                            site   => 'desktop',
                        );
                    }
                    catch {
                        $format = $app->api_format_system->formats(
                            format => $settings->{name},
                            role   => 'manager',
                            site   => 'desktop',
                        );
                    };
                    #Если параметр для данного формат может принимать только одно значение - принудительно выставляем его
                    my @only_one_settings = grep {scalar @{$_->{type}->{values} // []} == 1} @{$format->{settings}};
                    foreach my $one_settings (@only_one_settings) {
                        my $name = $one_settings->{name};
                        $settings->{$name} = $one_settings->{type}->{values}->[0]->{value};
                    }
                    #Выставляем значения для обязательных параметров по-умолчанию
                    my @hidden_settings = grep {$_->{type}->{name} eq 'hidden'} @{$format->{settings}};
                    foreach my $hidden_settings (@hidden_settings) {
                        my $name = $hidden_settings->{name};
                        $settings->{$name} = $hidden_settings->{type}->{value};
                    }

                    #Проверяем количество объявлений и при необходимости выставляем значение is_custom_format_direct
                    my ($limit_settings) = grep {$_->{name} eq 'limit'} @{$format->{settings}};
                    if (defined $limit_settings->{type}->{value}) {
                        $settings->{limit} //= $limit_settings->{type}->{value};
                        $settings->{limit} = 9 if $settings->{limit} > 9;
                        $block->{design_templates}->[0]->{is_custom_format_direct} = 1
                          if $settings->{limit} > $limit_settings->{type}->{value};
                    }

                    #Приводим значение fontSize к ближайщему допустимому
                    if (exists $settings->{fontSize}) {
                        $settings->{fontSize} = round($settings->{fontSize}, 1);
                        $settings->{fontSize} = 1.2 if $settings->{fontSize} > 1.2;
                        $settings->{fontSize} = 0.8 if $settings->{fontSize} < 0.8;
                    }
                    if ($page_id == 123194 || $page_id == 67181 || $page_id == 126916) {
                        $block->{design_templates}->[0]->{form_factor} = 'horizontal'
                          if $settings->{name} eq 'horizontal';
                        $settings->{urlBackgroundColor} = 'FFFFFF' if $page_id == 126916;
                        $settings->{linksUnderline} //= JSON::XS::false;
                        $settings->{imagesFirst}    //= JSON::XS::false;
                        $settings->{callouts}       //= JSON::XS::false;
                        $settings->{borderRadius}   //= JSON::XS::false;
                        $settings->{borderType}     //= 'none';
                    }

                    $block->{design_templates}->[0]->{design_settings} = $settings;
                    try {
                        my $res = $app->context_on_site_rtb->add(%$block);
                        print logstr "Added_block " . $res;
                        push @{$added_blocks->{$page_id}}, $res;
                    }
                    catch {
                        print logstr "Error adding block for page ", $page_id, " Error:", $@->{text},
                          " Design settings:", $settings;
                    }
                }
                close $fh;
            } else {
                print logstr 'Can\'t open log-file for block_settings: ', $!;
                return;
            }
        } elsif ($opts->{rollback}) {
            my $fh;
            if (open($fh, '<', $opts->{rollback})) {

                while (<$fh>) {
                    if (/Added_block\s+(\S+)$/) {
                        my $block_id = $1;
                        try {
                            $app->context_on_site_rtb->do_action($block_id, "delete");
                            print logstr "Delete block ", $block_id;
                        }
                        catch {
                            print logstr "Can't delete block ", $block_id, " Error:", $@->{text};
                        }
                    }
                }
                close $fh;

            } else {
                print logstr 'Can\'t open log-file for rollback: ', $!;
                return;
            }
        } else {
            print logstr "rollback or block_settings paremetr required.";
        }
    }
   );
