#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

    Конвертирует дизайн на блоке в adaptive

=head1 USAGE

    perl -I./lib ./bin/oneshots/PI-18374_convert_to_adaptive.pl

=head1 OPTIONS

    rollback_file - передать лог-файл предыдущего выполнения скрипта для отката

=cut

use qbit;

use Utils::ScriptWrapper;

my %FORMAT_LIST2CONVERT = map {$_ => 1}
  qw(160x600 200x300 240x400 250x250 300x250 300x300 300x500 300x600 320x100 336x280 420x240 728x90 970x250
  extensibleMobile grid motion 400x240 970x90 480x320 1000x120);
my $NAME_WH_MAP = {
    '728x90'         => {width => 320, height => 40},
    '970x250'        => {width => 320, height => 80},
    extensibleMobile => {width => 300, height => 250},
    grid             => {width => 300, height => 250},
    #motion -- это extentionalMobile.
    #Остальные -- как 320х40, но 320х(что-то) в соответствии с пропорциями.
    #https://st.yandex-team.ru/PI-18374#5e6b9b3989e8fa77f7555e27
    motion     => {width => 300, height => 250},
    '400x240'  => {width => 320, height => 192},
    '970x90'   => {width => 320, height => 50},
    '480x320'  => {width => 320, height => 213},
    '1000x120' => {width => 320, height => 50},
    # минимальная высота = 50, см. https://st.yandex-team.ru/PI-18374#5e9dceb32c29ea5ac451bd44
};

sub args {
    my ($opts) = @_;

    return ('rollback_file:s' => \$opts->{rollback_file},);
}

run(
    sub {
        my ($app, $opts) = @_;

        if ($opts->{rollback_file}) {
            my $fh;
            if (open($fh, '<', $opts->{rollback_file})) {
                my %design_templates_by_id;
                while (<$fh>) {
                    if (/-\s(\{.+\})$/) {
                        my $design_template_before = from_json($1);
                        $design_templates_by_id{$design_template_before->{id}} = {}
                          unless exists $design_templates_by_id{$design_template_before->{id}};
                        $design_templates_by_id{$design_template_before->{id}}->{before} = $design_template_before;
                    } elsif (/\+\s(\{.+\})$/) {
                        my $design_template_after = from_json($1);
                        $design_templates_by_id{$design_template_after->{id}} = {}
                          unless exists $design_templates_by_id{$design_template_after->{id}};
                        $design_templates_by_id{$design_template_after->{id}}->{after} = $design_template_after;
                    }
                }
                close $fh;

                unless (keys %design_templates_by_id) {
                    print logstr 'SKIP: not found rollback data in file';
                    return;
                }

                my $id_list_str = join(',', keys %design_templates_by_id);
                my $design_template_list_db = $app->partner_db->_get_all(
                    qq(
                    select design_templates.id, design_templates.opts
                    from design_templates
                    where
                    design_templates.id in ($id_list_str)
                )
                );
                my %hh = map {$_->{id} => $_} @$design_template_list_db;

                my @design_templats_list_to_update;
                foreach my $design_template_id (keys %design_templates_by_id) {
                    my $opts_before = from_json($design_templates_by_id{$design_template_id}->{before}{opts});
                    my $opts_after  = from_json($design_templates_by_id{$design_template_id}->{after}{opts});

                    my $opts_now = from_json($hh{$design_template_id}->{opts});
                    if (to_json($opts_now, canonical => TRUE) eq to_json($opts_after, canonical => TRUE)) {
                        push @design_templats_list_to_update,
                          {
                            id   => $design_template_id,
                            opts => to_json($opts_before)
                          };
                        print logstr($design_templats_list_to_update[-1]);
                    } else {
                        print logstr("SKIP: design was changed; id[$design_template_id]");
                    }
                }
                unless ($opts->{'dry_run'}) {
                    $app->partner_db->design_templates->add_multi(\@design_templats_list_to_update,
                        duplicate_update => 1);
                }
            } else {
                print logstr 'Can\'t open log-file for rollback: ', $!;
                return;
            }
        } else {
            my $design_template_list = $app->partner_db->_get_all(
                q(
    select design_templates.id, design_templates.opts, design_templates.page_id
    from context_on_site_rtb, design_templates
    where
    (context_on_site_rtb.campaign_id, context_on_site_rtb.id) in (select page_id, id from all_blocks where model = 'context_on_site_rtb' and site_version = 'amp') and
    context_on_site_rtb.campaign_id = design_templates.page_id and
    context_on_site_rtb.id = design_templates.block_id and
    context_on_site_rtb.is_custom_bk_data = 0
            )
            );

            my %pages_to_send;
            my @design_templats_list_to_update;
            foreach my $design_template (@$design_template_list) {
                my $db_opts = from_json($design_template->{opts});
                if (!_exists_in_msf($db_opts->{'design_settings'}{'name'})) {
                    my $patch = _get_format_width_height($db_opts->{'design_settings'}{'name'});
                    if (keys %$patch) {
                        print logstr('-', $design_template);
                        $db_opts->{'design_settings'} = {%{$db_opts->{'design_settings'}}, %$patch};
                        $design_template->{opts} = to_json($db_opts, canonical => TRUE);
                        push @design_templats_list_to_update, $design_template;
                        $pages_to_send{$design_template->{page_id}}++;
                        print logstr('+', $design_template);
                    }
                }
            }

            unless ($opts->{'dry_run'}) {
                $app->partner_db->design_templates->add_multi(\@design_templats_list_to_update, duplicate_update => 1);
            }

            print logstr('pages to send:', join(",", sort {$a <=> $b} keys %pages_to_send));
        }
    }
   );

sub _exists_in_msf {
    my $format = shift // '';
    my $msf_format4amp = {'adaptive' => 1,};
    return exists $msf_format4amp->{$format};
}

sub _get_format_width_height {
    my ($old_name) = @_;

    my $patch = {};
    if (exists $FORMAT_LIST2CONVERT{$old_name}) {
        if (exists $NAME_WH_MAP->{$old_name}) {
            $patch = {
                name => 'adaptive',
                (%{$NAME_WH_MAP->{$old_name}})
            };
        } else {
            my ($w, $h) = $old_name =~ /(\d+)x(\d+)/ig;
            if ($w && $h) {
                $patch = {
                    name   => 'adaptive',
                    width  => $w,
                    height => $h,
                };
            } else {
                printf "!can not convert format: name=[%s]\n", $old_name;
            }
        }
    }

    return $patch;
}
