package CustomBkData;
# use strict;
# use warnings FATAL => 'all';

# use lib::abs qw(../../../lib);
use qbit;
use Application;

use Getopt::Long;
use File::Path qw(make_path remove_tree);

use FilterValues;
use TransformKey;

use base qw(Exporter);
our @EXPORT = qw(
  get_app
  get_models
  to_csv
  get_opts
  prepare_path
  fix_old_design
  fix_design_wo_template
  normalize_data
  normalize_result
  append_params
  merge_result
  hash_to_array
  split_by_condition
  has_design_template
  );

sub get_models {
    my ($app) = @_;
    my $block_models = $app->product_manager->get_block_model_names;
    my @model_with_godmode;
    for my $accessor (grep {!/^ssp/} @$block_models) {
        my $model_fields = $app->$accessor->get_model_fields;
        push @model_with_godmode, $accessor if $model_fields->{is_custom_bk_data};
    }
    return \@model_with_godmode;
}

my $app;

sub get_app {

    unless ($app) {
        $app = Application->new();

        $app->pre_run();
        $app->set_cur_user({id => 0});
        {
            no warnings 'redefine';
            *QBit::Application::check_rights = sub {1};
        }
    }
    return $app;
}

sub to_csv {
    my ($header, $data) = @_;

    my $separator = "\t";
    my $out = join($separator, @$header) . "\n";
    for my $row (@$data) {
        if (@$header < @$row) {
            splice @$row, @$header;
        }
        $out .= join($separator, @$row) . "\n";
    }
    return $out;
}

sub get_opts {
    my (@strict) = @_;

    my %opts = (bkdata => 1);
    Getopt::Long::GetOptions(
        "limit=n"           => \$opts{limit},
        "date=s"            => \$opts{date},
        "root=s"            => \$opts{root},
        "overwrite!"        => \$opts{overwrite},
        "update!"           => \$opts{update},
        "custom=s"          => \$opts{custom},
        "strict=s"          => \$opts{strict},
        "source|src=s"      => \$opts{source},
        "destination|dst=s" => \$opts{destination},
        "input=s"           => \$opts{input},
        "input_list|il=s"   => \$opts{input_list},
        "input_json|ij=s"   => \$opts{input_json},
        "list=s"            => \$opts{list},
        "bkdata!"           => \$opts{bkdata},
    );

    unless ($opts{root}) {
        die "need root";
    }

    if ($opts{input}) {
        $opts{input} = readfile($opts{input});
    }
    if ($opts{input_list}) {
        my $list = readfile($opts{input_list});
        $opts{input_list} = [split /\n/, $list];
    }
    if ($opts{input_json}) {
        my $json = readfile($opts{input_json});
        $opts{input_json} = from_json($json);
    }

    if ($opts{list}) {
        $opts{list} = [split /,/, $opts{list}];
    }

    for my $key (@strict) {
        unless (defined $opts{$key}) {
            die "need $key";
        }
    }

    return \%opts;
}

sub prepare_path {
    my (@path) = @_;

    my $path = join('/', @path);
    (my $root = $path) =~ s|/[^/]+$||;
    unless (-e $root) {
        make_path($root);
    }
    if (-e $path) {
        my @time = localtime;
        my $ts   = sprintf "%04d%02d%02d-%02d%02d%02d",
          $time[5] + 1900,
          $time[4] + 1,
          $time[3],
          $time[2],
          $time[1],
          $time[0],
          ;
        rename $path, "$path.$ts.backup";
    }

    return $path;
}

sub fix_old_design {
    my ($data) = @_;

    for my $field (qw(RtbDesign Design)) {
        if (exists $data->{$field} and not ref $data->{$field}) {
            $data->{$field} = from_json('{' . $data->{$field} . '}');
        }
    }
    return $data;
}

sub fix_design_wo_template {
    my ($data) = @_;

    for my $field (qw(RtbDesign Design)) {
        if (exists $data->{$field}) {
            my $no_template;
            for my $k (keys %{$data->{$field}}) {
                if ($k =~ /^\d+$/) {
                    unless (exists $data->{$field}{$k}{design}) {
                        $no_template = 1;
                        last;
                    }
                } else {
                    $no_template = 1;
                    last;
                }
            }
            if ($no_template) {
                $data->{$field} = {
                    0 => {
                        name   => '',
                        design => $data->{$field},
                    },
                };
            }
        }
    }
    return $data;
}

sub normalize_result {
    my ($result) = @_;
    my %result;
    while (my ($key, $value) = each %$result) {
        my @keys = keys %$value;
        if (@keys == 1) {
            if ($keys[0] eq '<scalar>') {
                $result{$key} = 'scalar';
            } elsif ($keys[0] eq '<array>') {
                $result{$key} = [];
            } else {
                $result{$key} = \@keys;
            }
        } else {
            $result{$key} = [map {$_ eq '<array>' ? () : $_} sort @keys];
        }
    }
    return \%result;
}

sub append_params {
    my ($result, $path, $data) = @_;

    $data = transform_key($path, $data);

    unless (ref $data) {
        $data = filter_value($path, $data);
        $result->{$path}{$data} = undef if defined $data;
    } elsif (ref $data eq 'ARRAY') {
        unless (@$data) {
            $result->{$path}{'<array>'} = undef;
        } elsif (ref $data->[0]) {
            for my $e (@$data) {
                append_params($result, $path, $e);
            }
        } else {
            for my $e (@$data) {
                $result->{$path}{$e} = undef;
            }
        }
    } elsif (ref $data eq 'HASH') {
        for my $key (keys %$data) {
            my $tkey = $key;
            if ($tkey =~ /-?\d+$/) {
                $tkey = 'NNNNNN';
            }
            append_params($result, "$path.$tkey", $data->{$key});
        }
    }
}

sub hash_to_array {
    my ($data) = @_;
    my @result;
    for my $key (sort {$data->{$b} <=> $data->{$a}} keys %$data) {
        push @result, [$key, $data->{$key}];
    }
    return \@result;
}

my @split_by_condition = (
    ['is_readonly',  'readonly',  'not_readonly'],
    ['is_protected', 'protected', 'not_protected'],
    ['is_internal',  'internal',  'external'],
    ['money',        'has_money', 'no_money'],
    ['is_working',   'working',   'deleted'],
    ['bk_data_diff', 'has_diff',  'no_diff'],
);

sub split_by_condition {
    my ($result, $row) = @_;

    my @path = ('.');
    for my $r (@split_by_condition) {
        push @path, ($row->{$r->[0]} ? $r->[1] : $r->[2]);
    }
    while (@path) {
        my $path = join '/', @path;
        push @{$result->{$path}}, $row;
        pop @path;
    }
}

sub merge_result {
    my ($result, $data) = @_;
    for my $path (keys %$data) {
        my $values = [map {back_transform_key($path, $_)} keys %{$data->{$path}}];
        @{$result->{$path}}{@$values} = () if @$values;
    }
}

sub has_design_template {
    my ($data) = @_;

    my $no_template;
    for my $field (qw(RtbDesign Design)) {
        if (exists $data->{$field}) {
            for my $k (keys %{$data->{$field}}) {
                if ($k =~ /^\d+$/) {
                    unless (exists $data->{$field}{$k}{design}) {
                        $no_template = 1;
                        last;
                    }
                } else {
                    $no_template = 1;
                    last;
                }
            }
        }
    }
    return !$no_template;
}

1;
