#!/usr/bin/perl

=head1 пример вызова

    bin/oneshots/PI-18431/PI-18431_get_custom_bk_data.pl
        --root=/home/ie2018/tmp/new_bk_data
        --destination=custom_bk_data.json
        --limit=10

bin/oneshots/PI-18431/PI-18431_get_custom_bk_data.pl --root=/home/ie2018/tmp/new_bk_data --destination=custom_bk_data.json --date=2019-07-01 --limit=10
bin/oneshots/PI-18431/PI-18431_get_custom_bk_data.pl --root=/home/ie2018/tmp/new_bk_data --destination=custom_bk_data.json --date=2019-07-01 --list=C-A-268663-6 --update

bin/oneshots/PI-18431/PI-18431_get_custom_bk_data.pl --root=/home/ie2018/tmp/bk_data3 --destination=custom_bk_data.json --date=2019-07-01

=cut

use lib::abs qw(../../../lib .);

use qbit;
use CustomBkData;
use Time::HiRes qw(gettimeofday);

main();

sub main {

    my $opts = get_opts('destination', 'date', 'bkdata');
    my $bk_data = get_custom_bk_data($opts);
    writefile(prepare_path($opts->{root}, $opts->{destination}), to_json($bk_data, pretty => 1));
}

sub get_custom_bk_data {
    my ($opts) = @_;

    my $result;
    if ($opts->{update}) {
        $result = from_json(readfile("$opts->{root}/$opts->{destination}"));
    }
    my $app = get_app();
    my %filter_by_model;
    if (my $block_list = $opts->{list} || $opts->{input_list}) {
        require Utils::PublicID;
        my $r = Utils::PublicID::group_public_ids_by_model($app, $block_list);
        for my $model (keys %$r) {
            if (@{$r->{$model}}) {
                $filter_by_model{$model} = [{public_id => $r->{$model}}];
            }
        }
    } else {
        %filter_by_model = map {$_ => []} @{get_models($app)};
    }

    my $start = gettimeofday;
    for my $accessor (sort keys %filter_by_model) {
        my $model = $app->$accessor;
        my $is_internal = $model->get_campaign_model->is_internal_page ? 1 : 0;

        my $fields = [keys %{$model->get_model_fields}];
        my $mw     = $model->get_multistate_by_name('working');
        my $md     = $model->get_multistate_by_name('deleted');
        my $mr     = $model->get_campaign_model->get_multistate_by_name('read_only') // 0;

        my $r0   = gettimeofday;
        my $list = $model->get_all(
            fields => ['public_id', 'id', 'page_id', 'bk_data', 'multistate', @$fields],
            filter => [AND => [{is_custom_bk_data => 1}, @{$filter_by_model{$accessor}},]],
            ($opts->{limit} ? (limit => $opts->{limit}) : ())
        );

        my $r1    = gettimeofday;
        my $count = 0;
        my $total = @$list;
        my $last  = $r1;
        for my $row (@$list) {
            if (my $bk_data_custom = delete $row->{bk_data}) {
                my $create_date = $row->{create_date} // '1900-01-01';
                $create_date =~ s/\s.+$//;
                my %data = (
                    model          => $accessor,
                    public_id      => $row->{public_id},
                    id             => $row->{id},
                    page_id        => $row->{page_id},
                    is_working     => ($row->{multistate} & $mw ? 1 : 0),
                    is_deleted     => ($row->{multistate} & $md ? 1 : 0),
                    is_protected   => ($row->{is_protected} ? 1 : 0),
                    is_readonly    => ($row->{page}{multistate} & $mr ? 1 : 0),
                    is_internal    => $is_internal,
                    create_date    => $create_date,
                    bk_data_custom => fix_old_design(from_json($bk_data_custom)),
                );
                if ($opts->{bkdata}) {
                    $data{bk_data_code} = make_bk_block_data($model, $row);
                    make_bk_data_files("$opts->{root}/bk_data", \%data);
                }
                $result->{$row->{public_id}} = \%data;
            }
            unless (++$count % 100) {
                my $now = gettimeofday;
                warn sprintf("\t%s: %d/%d %0.3f\n", $accessor, $count, $total, $now - $last);
                $last = $now;
            }
        }
        my $r2 = gettimeofday;
        warn sprintf("%s: %d %0.3f %0.3f\n", $accessor, scalar(@$list), $r1 - $r0, $r2 - $r1);
    }
    my $get_data = gettimeofday();
    warn sprintf("get data: %0.3f\n", $get_data - $start);

    my %pages;
    my @filter_data;
    for my $row (values %$result) {
        push @filter_data, sprintf("(%d,%d)", $row->{page_id}, $row->{id});
        $pages{$row->{page_id}} = undef;
    }

    my $money = get_money(\@filter_data, $opts->{date});
    my $get_money = gettimeofday();
    warn sprintf("get money: %0.3f\n", $get_money - $get_data);

    my $sl = get_site_and_login($app, [keys %pages]);
    my $get_logins = gettimeofday();
    warn sprintf("get logins: %0.3f\n", $get_logins - $get_money);

    for my $row (values %$result) {
        $row->{money}  = $money->{$row->{page_id}}{$row->{id}} // 0;
        $row->{login}  = $sl->{$row->{page_id}}{login};
        $row->{domain} = $sl->{$row->{page_id}}{domain};
    }
    my $update_data = gettimeofday();
    warn sprintf("update_data: %0.3f\n", $update_data - $get_logins);

    return $result;
}

sub make_bk_block_data {
    my ($model, $block) = @_;

    $block->{is_custom_bk_data} = 0;
    my $bk_data = $model->get_bk_block_data($block);
    if ($bk_data) {
        $bk_data->{BlockModel} = $model->accessor;
        if (exists($bk_data->{'PageImpOptions'})) {
            map {$bk_data->{'PageImpOptions'}{$_} = [] if !exists($bk_data->{'PageImpOptions'}{$_})} qw(Enable Disable);
        }
    }

    return $bk_data;
}

sub get_site_and_login {
    my ($app, $pages) = @_;

    my %by_page =
      map {$_->{page_id} => $_}
      @{$app->all_pages->get_all(filter => {page_id => $pages}, fields => [qw(page_id login domain)])};

    return \%by_page;
}

sub get_money {
    my ($filter_data, $date) = @_;

    my %money;
    while (my @part = splice @$filter_data, 0, 1000) {
        get_money_data(\%money, \@part, $date);
    }

    return \%money;
}

sub get_money_data {
    my ($money, $filter_data, $date) = @_;

    my $filter     = join ",", @$filter_data;
    my $scale      = 10_000_000_000;
    my $ch_request = "
select
    page_id,
    block_id,
    round(sum(all_wo_nds)/$scale) as money
from
    statistics
where
    dt >= '$date'
    and (page_id,block_id) in ($filter)
group by
    page_id, block_id;
";
    $ch_request =~ s/\s+/ /g;
    my $ch_db = "clickhouse_mdb --server=prod";
    my $ch    = `echo "$ch_request" | $ch_db`;

    for my $row (split /\n/, $ch) {
        $row =~ s/\s+$//;
        my @row = split /\t/, $row;
        next unless $row[0] and $row[1];
        $money->{$row[0]}{$row[1]} = $row[2];
    }

    return $money;
}

sub make_bk_data_files {
    my ($root, $row) = @_;

    writefile(prepare_path("$root/$row->{public_id}.code.json"),    to_json($row->{bk_data_code},   pretty => 1));
    writefile(prepare_path("$root/$row->{public_id}.godmode.json"), to_json($row->{bk_data_custom}, pretty => 1));
}
