#!/usr/bin/perl

=head1 пример вызова

    bin/oneshots/PI-18431/PI-18431_make_diff.pl
        --root=/home/ie2018/tmp/new_bk_data
        --destination=diff_bk_data.json
        --custom=custom_bk_data.json
        --strict=strict_bk_data.json

bin/oneshots/PI-18431/PI-18431_make_diff.pl --root=/home/ie2018/tmp/bk_data2 --destination=diff_bk_data.json --custom=custom_bk_data.json --strict=strict_bk_data.json
bin/oneshots/PI-18431/PI-18431_make_diff.pl --root=/home/ie2018/tmp/bk_data3 --destination=diff_bk_data.json --custom=custom_bk_data.json --strict=strict_bk_data.json


=cut

use lib::abs qw(../../../lib .);

use qbit;
use CustomBkData;
use TransformKey;
use FilterValues;

main();

sub main {
    my $opts   = get_opts('custom', 'strict', 'destination');
    my $strict = from_json(readfile("$opts->{root}/$opts->{strict}"));
    my $custom = from_json(readfile("$opts->{root}/$opts->{custom}"));

    switch_on_ignore();
    my $parted = param_to_path_by_block($opts, $custom, $strict);

    my @list = sort keys %$parted;
    my $data = '';
    for (my $i = $#list; $i > 0; $i--) {
        my @s = split /\//, $list[$i];
        my @r;
        for my $p (split /\//, $list[$i - 1]) {
            my $s = shift @s;
            if ($p eq $s) {
                push @r, " " x 3;
            } else {
                push @r, $s;
                last;
            }
        }
        $data = join(' ', @r, @s) . ": " . $parted->{$list[$i]} . "\n" . $data;
    }
    $data = '*' . ": " . $parted->{$list[0]} . "\n" . $data;
    writefile(prepare_path($opts->{root}, $opts->{destination}), $data);
    # writefile(prepare_path($opts->{root}, $opts->{destination}), to_json($parted, pretty => 1));
}

sub param_to_path_by_block {
    my ($opts, $custom_data, $strict_data) = @_;

    my %parted;
    for my $row (values %$custom_data) {
        my %result;
        append_params(\%result, '', fix_design_wo_template($row->{bk_data_custom}));
        $row->{bk_data_diff} = compare_for_strict($strict_data, normalize_result(\%result));
        split_by_condition(\%parted, $row);
    }

    my $root = "$opts->{root}/god_mode";
    my %counts;
    for my $path (keys %parted) {
        $counts{$path} = make_bk_data_report("$root/$path", $parted{$path});
    }
    return \%counts;
}

sub compare_for_strict {
    my ($strict, $data) = @_;
    my %result;
    for my $key (keys %$data) {
        unless (exists $strict->{$key}) {
            $result{$key} = $data->{$key};
        } else {
            if (my $diff = compare_data($strict->{$key}, $data->{$key})) {
                $result{$key} = $diff;
            }
        }
    }
    if (%result) {
        return \%result;
    } else {
        return undef;
    }
}

sub compare_data {
    my ($strict, $real) = @_;
    if (@$real) {
        my %hk = map {$_ => undef} @$strict;
        my @no;
        for my $v (@$real) {
            unless (exists $hk{$v}) {
                push @no, $v;
            }
        }
        return \@no if @no;
    }
    return undef;
}

sub make_bk_data_report {
    my ($root, $list) = @_;

    my @blocks;
    my @block_path;
    my @block_diff_count;
    my %summary;
    my %by_model;
    my %by_login;
    my %by_key;
    my %by_login_and_key;

    for my $row (@$list) {
        $by_model{$row->{model}}++;
        $by_login{$row->{login}}++;
        push @blocks,
          [
            $row->{public_id}, $row->{model}, $row->{login}, $row->{domain},
            $row->{money} // 0, $row->{is_working} // 0, $row->{create_date}
          ];
        my $block_diff_count = [$row->{public_id}, $row->{money} // 0, $row->{login}, 0, [],];
        for my $key (keys %{$row->{bk_data_diff}}) {
            $by_login_and_key{$row->{login}}{$key}++;
            $by_key{$key}++;
            my $diff = [map {back_transform_key($key, $_)} @{$row->{bk_data_diff}{$key}}];
            my $values = join(",", @$diff);

            if (@$diff) {
                push @block_path, [$row->{public_id}, $key, $values];
                push @{$block_diff_count->[4]}, $key;
                $block_diff_count->[3]++;
            }
            $summary{$key}{count}++;
            push @{$summary{$key}{blocks}}, [$row->{public_id}, $row->{money} // 0];
            $summary{$key}{models}{$row->{model}} = undef;

            for my $v (@$diff) {
                $summary{$key}{values}{$v}++;
            }
        }
        if ($block_diff_count->[3] > 0) {
            my $max_keys = 5;
            my @tmp      = sort @{$block_diff_count->[4]};
            if ($block_diff_count->[3] > $max_keys) {
                splice @tmp, $max_keys, $#tmp, '...';
            }
            $block_diff_count->[4] = join(', ', @tmp);
            push @block_diff_count, $block_diff_count;
        }
    }

    my @summary;
    my @values_count;
    for my $key (sort keys %summary) {
        my @models = keys %{$summary{$key}{models}};
        my @blocks = map {$_->[0]} sort {$b->[1] <=> $a->[1] or $a->[0] cmp $b->[0]} @{$summary{$key}{blocks}};
        splice @blocks, 5 if @blocks > 5;
        my $values = $summary{$key}{values};
        my @values = keys %$values;
        for my $v (sort {$values->{$b} <=> $values->{$a}} @values) {
            push @values_count, [$key, $values->{$v}, $v];
        }
        push @summary, [$key, $summary{$key}{count}, join(", ", @blocks), join(", ", @values), join(", ", @models),];
    }

    my @by_login_and_key;
    for my $login (keys %by_login_and_key) {
        for my $key (keys %{$by_login_and_key{$login}}) {
            push @by_login_and_key, [$login, $key, $by_login_and_key{$login}{$key},];
        }
    }

    @summary          = sort {$b->[1] <=> $a->[1] or $a->[0] cmp $b->[0]} @summary;
    @blocks           = sort {$b->[4] <=> $a->[4] or $a->[0] cmp $b->[0]} @blocks;
    @block_path       = sort {$a->[0] cmp $b->[0] or $a->[1] cmp $b->[1]} @block_path;
    @by_login_and_key = sort {$a->[0] cmp $b->[0] or $a->[1] cmp $b->[1]} @by_login_and_key;
    @block_diff_count = sort {$a->[3] <=> $b->[3] or $a->[4] cmp $b->[4] or $b->[1] <=> $a->[1]} @block_diff_count;

    writefile(prepare_path("$root/blocks.tsv"),
        to_csv([qw(block_id model login domain money is_working create_date)], \@blocks));
    writefile(prepare_path("$root/block_path.tsv"), to_csv([qw(block_id key values)],             \@block_path));
    writefile(prepare_path("$root/summary.tsv"),    to_csv([qw(path count blocks values models)], \@summary));

    writefile(prepare_path("$root/stat_by_login_and_path.tsv"), to_csv([qw(login path count)], \@by_login_and_key));

    writefile(prepare_path("$root/stat_by_values_path.tsv"), to_csv([qw(path count value)], \@values_count));

    writefile(prepare_path("$root/stat_by_model.tsv"), to_csv([qw(model count)], hash_to_array(\%by_model)));
    writefile(prepare_path("$root/stat_by_login.tsv"), to_csv([qw(login count)], hash_to_array(\%by_login)));
    writefile(prepare_path("$root/stat_by_path.tsv"),  to_csv([qw(path count)],  hash_to_array(\%by_key)));

    writefile(prepare_path("$root/stat_total_block.txt"), scalar(@$list) . "\n");

    writefile(prepare_path("$root/block_diff_count.tsv"),
        to_csv([qw(block_id money login count list)], \@block_diff_count));

    return scalar(@$list);
}

sub stat_by_template {
    my ($opts, $custom_data) = @_;

    my %by_model;
    for my $row (values %$custom_data) {
        unless (has_design_template($row->{bk_data_custom})) {
            $by_model{$row->{model}}++;
        }
    }
    writefile(
        prepare_path($opts->{root}, "no_template_by_model.tsv"),
        to_csv([qw(model count)], hash_to_array(\%by_model))
    );
}
