#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

    изменяет bk_data блоков в годмоде исправляет RtbDesign/Design с учетом айкциона дизайнов

=head1 USAGE

    perl -Ilib bin/oneshots/PI-18991_fix_design_in_godmode.pl
    perl -Ilib bin/oneshots/PI-18991_fix_design_in_godmode.pl --remap --output_pages=page_ids.txt --output_blocks=blocks.json
    perl -Ilib bin/oneshots/PI-18991_fix_design_in_godmode.pl --list_blocks=R-A-249322-23 --output_pages=page_ids.txt

=head1 OPTIONS

    output_blocks   - имя файла для данных по всем полученным блокам
    output_pages    - имя файла для списка затронутых площадок
    create_template - создавать шаблоны для блоков у которых их нет
    list_blocks     - список блоков через запятую
    input_blocks    - имя файла со списком блоков
    remap           - заменять дизайн 0 на реальный

=cut

use qbit;
use Utils::ScriptWrapper;

my $NO_TEMPLATE  = 1;
my $PLAIN_DESIGN = 2;
my $NO_FIELD     = 3;
my $ZERO_DESIGN  = 4;

my $DEFAULT_TEMPLATE = {
    context_on_site_rtb => {
        desktop => {
            "design_settings" => {
                "name"            => "adaptive",
                "limit"           => 9,
                "width"           => 940,
                "height"          => 90,
                "bgColor"         => "FFFFFF",
                "urlColor"        => "006600",
                "textColor"       => "333333",
                "borderType"      => "block",
                "hoverColor"      => "F00000",
                "titleColor"      => "0069B6",
                "borderColor"     => "DDDDDD",
                "imagesFirst"     => \0,
                "noSitelinks"     => \1,
                "siteBgColor"     => "FFFFFF",
                "borderRadius"    => \0,
                "headerBgColor"   => "DDDDDD",
                "linksUnderline"  => \0,
                "horizontalAlign" => \0
            },
            "form_factor"             => "horizontal",
            "is_custom_format_direct" => "0"
        }
    }
};

sub args {
    my ($opts) = @_;

    return (
        'output_blocks:s'  => \$opts->{'output_blocks'},
        'output_pages:s'   => \$opts->{'output_pages'},
        'create_template!' => \$opts->{'create_template'},
        'list_blocks:s'    => \$opts->{'list_blocks'},
        'input_blocks:s'   => \$opts->{'input_blocks'},
        'remap!'           => \$opts->{'remap'},
    );
}

run(\&main);

sub prepare_opts {
    my ($opts) = @_;

    if ($opts->{list_blocks}) {
        $opts->{filter_blocks} = [split /,/, $opts->{list_blocks}];
    }

    if ($opts->{input_blocks}) {
        for my $str (split /\n/, readfile($opts->{input_blocks})) {
            if ($str and $str =~ /([A-Z]+-[A-Z]+-\d+-\d+)/) {
                push @{$opts->{filter_blocks}}, $1;
            }
        }
    }

    return $opts;
}

sub main {
    my ($app, $opts) = @_;
    $opts = prepare_opts($opts);

    my $data = get_data($app, $opts);
    my $pages = make_data($app, $data, $opts);

    my @pages = keys %$pages;
    $app->all_pages->mark_pages_for_async_update(page_ids => \@pages);

    if ($opts->{output_pages}) {
        writefile($opts->{output_pages}, join("\n", @pages) . "\n");
    }
}

sub make_data {
    my ($app, $data, $opts) = @_;

    my %pages;
    for my $accessor (keys %$data) {
        my $page_id_fn = $app->$accessor->get_page_id_field_name();
        for my $row (@{$data->{$accessor}}) {
            print logstr("PREPARE_BLOCK", $accessor, $row->{public_id}, $row->{row});
            my $bk_data = $row->{bk_data_obj};
            my $design  = $bk_data->{$row->{design_field}};
            try {
                if ($row->{need_update} == $PLAIN_DESIGN) {
                    $design = from_json('{' . $design . '}');
                } elsif ($row->{need_update} == $ZERO_DESIGN) {
                    $design = $design->{0}{design};
                }
                if ($row->{zero_design}) {
                    if ($row->{need_update} == $ZERO_DESIGN) {
                        print logstr("SKIP_BLOCK_ZERO", $row->{public_id});
                        $design = undef;
                    } else {
                        print logstr("MAKE_BLOCK_ZERO", $row->{public_id});
                        $design = {
                            0 => {
                                design => $design,
                                name   => '',
                            },
                        };
                    }
                } elsif (my $fd = $row->{found_design}) {
                    print logstr("MAKE_BLOCK_FOUND", $row->{public_id}, $fd);
                    $design = {
                        0 => {
                            design => {"horizontalAlign" => \1},
                            name   => 'Ненастраиваемые форматы',
                            type   => 'media',
                        },
                        $fd->{id} => {
                            design => $design,
                            name   => $fd->{caption},
                        },
                    };
                } elsif ($opts->{create_template} and my $dd = $row->{default_design}) {
                    my $new_design = {
                        page_id   => $row->{page_id},
                        block_id  => $row->{block_id},
                        type      => 'tgo',
                        caption   => 'Исходный дизайн',
                        order_num => 0,
                        opts      => to_json($dd),
                    };
                    my $id = $app->partner_db->design_templates->add($new_design);
                    print logstr("MAKE_BLOCK_NEW", $row->{public_id}, $id, $new_design);
                    $design = {
                        0 => {
                            design => {"horizontalAlign" => \1},
                            name   => 'Ненастраиваемые форматы',
                            type   => 'media',
                        },
                        $id => {
                            design => $design,
                            name   => 'Исходный дизайн',
                        },
                    };
                } else {
                    print logstr("MAKE_BLOCK_SKIP", $row->{public_id});
                    $design = undef;
                }
            }
            catch {
                my ($e) = @_;
                print logstr("MAKE_BLOCK_ERROR", $row->{public_id}, $e->message);
                $design = undef;
            };

            next unless $design;

            $bk_data->{$row->{design_field}} = $design;
            try {
                $app->partner_db->$accessor->edit(
                    {
                        $page_id_fn => $row->{page_id},
                        id          => $row->{block_id},
                    },
                    {bk_data => to_json($bk_data, pretty => TRUE)}
                );
                print logstr("STORE_BLOCK_OK", $row->{public_id}, $bk_data);
                $pages{$row->{page_id}}++;
            }
            catch {
                my ($e) = @_;
                print logstr("STORE_BLOCK_FAIL", $row->{public_id}, $bk_data, $e->message);
            };
        }
    }

    return \%pages;
}

sub get_data {
    my ($app, $opts) = @_;

    my $models = get_models_with_filter($app, $opts);

    my $to_process = {};
    for my $accessor (keys %$models) {
        my ($list, $skip) = get_block_list($app, $accessor, $models->{$accessor}, $opts);
        print logstr("FOUND", $accessor, scalar(@$list), scalar(@$skip));
        for my $id (@$skip) {
            print logstr("SKIP_READONLY_OR_PROTECTED", $accessor, $id);
        }
        if (@$list) {
            $to_process->{$accessor} = $list;
        }
    }

    my ($count_nt, $need_template) = get_design_templates($app, $to_process);
    if ($count_nt) {
        print logstr("WITHOUT_TEMPLATE", $count_nt);

        for my $accessor (keys %$need_template) {
            my $model = $app->$accessor;
            my $ddt   = $model->can("_get_default_direct_template");
            print logstr("NO_TEMPLATE", $accessor, scalar(@{$need_template->{$accessor}}));
            for my $row (@{$need_template->{$accessor}}) {
                if ($ddt) {
                    try {
                        $row->{default_design} = $model->_get_default_direct_template($row->{row});
                        print logstr("MAKE_TEMPLATE", $row->{public_id});
                    }
                    catch {
                        my ($e) = @_;
                        $row->{error_design} = $e->message;
                        print logstr("ERROR_TEMPLATE", $row->{public_id});
                        my $sv = $row->{row}{site_version};
                        if (my $t = $DEFAULT_TEMPLATE->{$accessor}{$sv}) {
                            $row->{default_design} = $t;
                            print logstr("HAS_DEFAULT_TEMPLATE", $row->{public_id}, $accessor, $sv);
                        } else {
                            print logstr("NO_DEFAULT_TEMPLATE", $row->{public_id}, $accessor, $sv);
                        }
                    };
                } else {
                    $row->{zero_design} = TRUE;
                    print logstr("ZERO_TEMPLATE", $row->{public_id});
                }
            }
        }
    }

    if ($opts->{output_blocks}) {
        writefile($opts->{output_blocks}, to_json($to_process, pretty => TRUE));
    }

    return $to_process;
}

sub get_models_with_filter {
    my ($app, $opts) = @_;

    my %filter_by_model;
    if ($opts->{filter_blocks}) {
        require Utils::PublicID;
        my $r = Utils::PublicID::group_public_ids_by_model($app, $opts->{filter_blocks});
        for my $model (keys %$r) {
            if (@{$r->{$model}}) {
                $filter_by_model{$model} = [{public_id => $r->{$model}}];
            }
        }
    } else {
        %filter_by_model = map {$_ => []} @{get_models($app)};
    }

    return \%filter_by_model;
}

sub get_models {
    my ($app) = @_;

    my $block_models = $app->product_manager->get_block_model_names;
    my @model_with_godmode;
    for my $accessor (grep {!/^ssp/} @$block_models) {
        my $model_fields = $app->$accessor->get_model_fields;
        push @model_with_godmode, $accessor if $model_fields->{is_custom_bk_data};
    }

    return \@model_with_godmode;
}

sub get_block_list {
    my ($app, $accessor, $filter, $opts) = @_;

    my $model = $app->$accessor;

    my @fields = keys %{$model->get_model_fields};

    my $mw = $model->get_multistate_by_name('working');
    my $md = $model->get_multistate_by_name('deleted');
    my $mr = $model->get_campaign_model->get_multistate_by_name('read_only') // 0;
    my $df = $model->design_field;

    my $list = $model->get_all(
        fields => \@fields,
        filter => [AND => [{is_custom_bk_data => 1}, @$filter]],
    );

    my @to_process;
    my @skip;
    for my $row (@$list) {
        if (my $bk_data_custom = $row->{bk_data}) {
            my $bk_data = from_json($bk_data_custom);
            my $need_update = is_need_update($df, $bk_data, $opts);
            next unless $need_update;
            if ($row->{is_protected} or $row->{page}{multistate} & $mr) {
                push @skip, $row->{public_id};
            } else {
                my %data = (
                    model        => $accessor,
                    design_field => $df,
                    need_update  => $need_update,
                    public_id    => $row->{public_id},
                    block_id     => $row->{id},
                    page_id      => $row->{page_id},
                    is_working   => ($row->{multistate} & $mw ? 1 : 0),
                    is_deleted   => ($row->{multistate} & $md ? 1 : 0),
                    bk_data_obj  => $bk_data,
                    row          => $row,
                );
                push @to_process, \%data;
            }
        }
    }

    return \(@to_process, @skip);
}

sub is_need_update {
    my ($df, $data, $opts) = @_;

    my $result;
    if (exists $data->{$df}) {
        unless (ref $data->{$df}) {
            $result = $PLAIN_DESIGN;
        } else {
            for my $k (keys %{$data->{$df}}) {
                if ($k =~ /^\d+$/) {
                    unless (exists $data->{$df}{$k}{design}) {
                        $result = $NO_TEMPLATE;
                        last;
                    }
                } else {
                    $result = $NO_TEMPLATE;
                    last;
                }
            }
            if ($opts->{remap} and !$result and scalar(keys %{$data->{$df}}) == 1 and $data->{$df}{0}) {
                $result = $ZERO_DESIGN;
            }
        }
    }

    return $result;
}

sub get_design_templates {
    my ($app, $data) = @_;

    my @filter;
    my %need;
    for my $list (values %$data) {
        for my $row (@$list) {
            push @filter, [AND => [[page_id => '=' => \$row->{page_id}], [block_id => '=' => \$row->{block_id}]]];
            $need{$row->{page_id}}{$row->{block_id}} = $row;
        }
    }

    my $list = $app->partner_db->design_templates->get_all(
        filter => [AND => [[type => '=' => \'tga'], [multistate => '=' => \0], [OR => \@filter],]],
        fields => [qw(id page_id block_id order_num type caption)],
    );

    for my $row (@$list) {
        if (my $src = delete $need{$row->{page_id}}{$row->{block_id}}) {
            $src->{found_design} = $row;
        }
    }

    my %need_template;
    my $count = 0;
    for my $page_id (keys %need) {
        for my $block_id (keys %{$need{$page_id}}) {
            my $row = $need{$page_id}{$block_id};
            push @{$need_template{$row->{model}}}, $row;
            $count++;
        }
    }

    return ($count, \%need_template);
}
