#!/usr/bin/perl
use qbit;
use Utils::ScriptWrapper;
use PiConstants qw($STAT_MONEY_SCALE);

sub args {
    my ($opts) = @_;

    return (
        'money_after:s'   => \$opts->{'money_after'},
        'create_before:s' => \$opts->{'create_before'},
        'input:s'         => \$opts->{'input'},
        'output:s'        => \$opts->{'output'},
    );
}

run(\&main);

sub main {
    my ($app, $opts) = @_;
    my $models = get_models($app);

    my $create_before = $opts->{create_before} // '2019-10-01';
    my $money_after   = $opts->{money_after}   // '2019-10-01';
    if ($opts->{input}) {
        $opts->{filter_list} = read_file_to_hash($opts->{input});
    }

    my @result;
    my @filter;
    for my $accessor (@$models) {
        my $prefix = $app->$accessor->public_id_prefix;
        my $list = get_list($app, $accessor, $create_before, $money_after);
        for my $row (@$list) {
            $row->{public_id} = "$prefix$row->{page_id}-$row->{block_id}";
            $row->{accessor}  = $accessor;
            push @result, $row;
            push @filter, "($row->{page_id},$row->{block_id})";
        }
        print STDERR logstr("FOUND", $accessor, scalar(@$list));
    }

    my $money = get_money(\@filter, $money_after);
    print STDERR logstr("MONEY");

    my %by_accessor;
    for my $row (@result) {
        my $in_list = $opts->{input} && delete $opts->{filter_list}{$row->{public_id}};
        my $login = $row->{login} // 'adinside';
        my $has_money = $money->{$row->{page_id}}{$row->{block_id}} // 0;
        my $is_working = $row->{is_working} ? 'working' : 'not_working';
        my @row = ($row->{public_id}, $login, $in_list ? 'in_list' : 'not_listed', $has_money, $is_working,);
        unless ($has_money or ($opts->{input} and !$in_list)) {
            push @{$by_accessor{$row->{accessor}}}, $row;
            print logstr("BLOCK_TO_PROCESS", @row);
        } else {
            print logstr("BLOCK_SKIP", @row);
        }
    }

    if ($opts->{input} and ref $opts->{filter_list}) {
        for my $id (keys %{$opts->{filter_list}}) {
            print logstr("NOT_IN_LIST", $id);
        }
    }

    if ($opts->{output}) {
        open F, ">", $opts->{output};
        for my $accessor (keys %by_accessor) {
            my $data = get_block_data($app, $accessor, $by_accessor{$accessor});
            for my $row (@{$by_accessor{$accessor}}) {
                my $d = $data->{$row->{page_id}}{$row->{block_id}};
                printf F "%s\t%s\t%s\t%s\n",
                  $row->{public_id},
                  $row->{is_working} ? 'is_working' : 'not_working',
                  $d->{create_date},
                  $d->{edit_date} // '';
            }
        }
        close F;
    }
}

sub get_models {
    my ($app) = @_;

    my $block_models = $app->product_manager->get_block_model_names;
    my @model_with_godmode;
    for my $accessor (grep {!/^ssp/} @$block_models) {
        my $model_fields = $app->$accessor->get_model_fields;
        push @model_with_godmode, $accessor if $model_fields->{is_custom_bk_data};
    }

    return \@model_with_godmode;
}

sub get_list {
    my ($app, $accessor, $date_create, $date_update) = @_;

    my $model   = $app->$accessor;
    my $pn      = $model->get_page_id_field_name;
    my $working = $model->get_multistate_by_name('working');
    my $table   = $app->partner_db->$accessor;

    my $alog     = $accessor . "_action_log";
    my $table_al = $app->partner_db->$alog;

    my $campaign  = $model->get_campaign_model_name;
    my $table_cmp = $app->partner_db->$campaign;
    my $protected = $app->$campaign->get_multistate_by_name('protected');
    my $cpn       = $app->$campaign->get_page_id_field_name;
    if (defined(my $readonly = $app->$campaign->get_multistate_by_name('read_only'))) {
        $protected |= $readonly;
    }

    my $query = $app->partner_db->query->select(
        table  => $table,
        fields => {
            block_id   => 'id',
            page_id    => $pn,
            is_working => {if => [[multistate => '&' => \$working], \1, \0]},
        },
        filter => [is_custom_bk_data => '=' => \1],
      )->join(
        table   => $table_cmp,
        fields  => [],
        join_on => [$cpn => '=' => {$pn => $table}],
        filter  => [[multistate => "&" => \$protected] => '=' => \0],
      )->join(
        table   => $table_al,
        fields  => [],
        join_on => [
            AND =>
              [["elem_$pn" => '=' => {$pn => $table}], [elem_id => '=' => {id => $table}], [action => "=" => \"add"],]
        ],
        filter => [dt => "<" => \$date_create],
      )->left_join(
        table   => $table_al,
        alias   => 'edited',
        fields  => [],
        join_on => [
            AND => [
                ["elem_$pn" => '=' => {$pn => $table}],
                [elem_id    => '=' => {id  => $table}],
                [action     => "=" => \"edit"],
                [dt         => ">" => \$date_update],
            ]
        ],
        filter => [id => "=" => \undef],
      );

    unless ($campaign =~ /^internal/) {
        $query = $query->join(
            table   => $app->partner_db->users,
            fields  => ['login'],
            join_on => [id => '=' => {owner_id => $table_cmp}],
        );
    }

    return $query->get_all;
}

sub get_money {
    my ($filter, $date) = @_;

    my $query =
        "select page_id, block_id, round(sum(all_wo_nds)/$STAT_MONEY_SCALE) as money"
      . " from statistics"
      . " where (page_id,block_id) in (%s)"
      . " and dt > '$date'"
      . " group by page_id,block_id";

    my $ch_db = "clickhouse_mdb --server=prod -- --format=JSONEachRow | tail -n +2";

    my %result;
    while (my @part = splice @$filter, 0, 5000) {
        my $ch_request = sprintf $query, join(',', @part);
        my $ch = `echo "$ch_request" | $ch_db`;
        for my $row (split /\n/, $ch) {
            $row = from_json($row);
            $result{$row->{page_id}}{$row->{block_id}} = $row->{money};
        }
    }

    return \%result;
}

sub read_file_to_hash {
    my ($filename) = @_;

    my $fh;
    open $fh, $filename;
    my %result;
    while (my $str = <$fh>) {
        $str =~ s/\s+//g;
        next unless $str and $str !~ /id/;
        $result{$str}++;
    }

    return \%result;
}

sub get_block_data {
    my ($app, $accessor, $list) = @_;

    my $model = $app->$accessor;
    my $pn    = $model->get_page_id_field_name;

    my $alog     = $accessor . "_action_log";
    my $table_al = $app->partner_db->$alog;
    my @filter   = map [AND => [["elem_$pn" => '=' => \$_->{page_id}], ["elem_id" => '=' => \$_->{block_id}],]], @$list;

    my $query = $app->partner_db->query->select(
        table  => $table_al,
        fields => {
            page_id     => "elem_$pn",
            block_id    => "elem_id",
            create_date => {max => [{if => [['action' => '=' => \'add'], 'dt', \undef]}]},
            edit_date   => {max => [{if => [['action' => '=' => \'edit'], 'dt', \undef]}]}
        },
        filter => [AND => [[action => 'IN' => \[qw(edit add)]], [OR => \@filter],],],
    )->group_by('page_id', 'block_id');

    # warn("".($query->get_sql_with_data)[0]."\n");

    my %result;
    for my $row (@{$query->get_all}) {
        $result{$row->{page_id}}{$row->{block_id}} = $row;
    }

    return \%result;
}
