#!/usr/bin/perl
package main;
use qbit;
use Utils::ScriptWrapper;
use Utils::PublicID;
use PiConstants qw($STAT_MONEY_SCALE $MAX_REVENUE_STRATEGY_ID);

# замена удаленной константы
my $YAN_TRAFFIC_STRATEGY_ID = 2;

my %filter_by_sm = (
    db => {
        f => "`block`.`strategy` = $YAN_TRAFFIC_STRATEGY_ID",
        c => "`block`.`percent_traffic`",
    },
    opts => {
        f => "`block`.`opts`->>'\$.strategy' = $YAN_TRAFFIC_STRATEGY_ID",
        c => "`block`.`opts`->>'\$.percent_traffic'",
    },
);

my $convertors = {
    db   => \&convert_block_db,
    opts => \&convert_block_opts,
};

run(\&main);

sub args {
    my ($opts) = @_;

    return ('report:s' => \$opts->{'report'});
}

sub main {
    my ($app, $opts) = @_;
    if ($opts->{report}) {
        make_report($app, $opts->{report});
    } else {
        convert_blocks($app, $opts);
    }
}

sub convert_block_db {
    my ($app, $row, $rec) = @_;

    my $fix_bk_data = update_bk_data($row);

    $app->partner_db->_do("
    update $rec->{table} set $fix_bk_data
    strategy = $MAX_REVENUE_STRATEGY_ID,
    percent_traffic = null
    where $rec->{page} = $row->{page_id} and id = $row->{block_id}
");
}

sub convert_block_opts {
    my ($app, $row, $rec) = @_;

    my $fix_bk_data = update_bk_data($row);

    $app->partner_db->_do("
    update $rec->{table} set $fix_bk_data
    opts = json_set(opts,'\$.strategy',$MAX_REVENUE_STRATEGY_ID,'\$.percent_traffic',null)
    where $rec->{page} = $row->{page_id} and id = $row->{block_id}
");
}

sub update_bk_data {
    my ($row) = @_;
    if ($row->{gm}) {
        return
            "bk_data = json_remove("
          . "json_set(bk_data,'\$.OptimizeType',$MAX_REVENUE_STRATEGY_ID),"
          . "'\$.AlternativeTraffPart'),";
    } else {
        return '';
    }
}

sub convert_blocks {
    my ($app, $opts) = @_;

    my $count = 0;
    my %pages_to_resend;
    for my $rec (@{list_model($app)}) {
        my $list = get_db_data($app, $rec);
        my $convertor = $convertors->{$rec->{storage}};
        for my $row (@$list) {
            if ($rec->{skip}) {
                print STDERR logstr('SKIP_BLOCK', $row->{public_id}, $row);
            }
            try {
                $convertor->($app, $row, $rec);
                unless ($row->{deleted}) {
                    $pages_to_resend{$row->{page_id}} = 1;
                }
                print logstr('DONE_BLOCK', $row->{public_id}, $row);
                $count++;
            }
            catch {
                my ($e) = @_;
                print STDERR logstr('ERROR_BLOCK', $row->{public_id}, $row, $e->message);
            };
        }
    }
    my @pages = keys %pages_to_resend;
    print logstr('TOTAL', $count);
    print logstr('PAGES', \@pages);

    $app->all_pages->mark_pages_for_async_update(page_ids => \@pages);
}

sub make_report {
    my ($app, $fn) = @_;

    my @report;
    for my $rec (@{list_model($app)}) {
        push @report, @{get_db_data_with_money($app, $rec)};
    }
    writefile($fn, to_csv(['public_id', 'login', 'domain', 'money', 'gm', 'deleted'], \@report));
}

sub get_money {
    my ($filter, $date) = @_;

    my $query =
        "select page_id, block_id, round(sum(all_wo_nds)/$STAT_MONEY_SCALE) as money"
      . " from statistics"
      . " where (page_id,block_id) in (%s)"
      . " and dt > '$date'"
      . " group by page_id,block_id";

    my $ch_db = "clickhouse_mdb --server=prod -- --format=JSONEachRow | tail -n +2";

    my %result;
    while (my @part = splice @$filter, 0, 5000) {
        my $ch_request = sprintf $query, join(',', @part);
        my $ch = `echo "$ch_request" | $ch_db`;
        for my $row (split /\n/, $ch) {
            $row = from_json($row);
            $result{$row->{page_id}}{$row->{block_id}} = $row->{money};
        }
    }

    return \%result;
}

sub list_model {
    my ($app) = @_;

    my @list;
    for my $accessor (@{$app->product_manager->get_block_model_accessors}) {
        my $model = $app->$accessor;
        if ($model->DOES('Application::Model::Role::Has::Strategies')) {
            my $method = $model->storage_method_of_strategies;
            my $s      = $model->get_strategies;
            unless ($convertors->{$method}) {
                print STDERR logstr("bad method '$method' for '$accessor'");
                next;
            }
            my $table = $model->db_table_name;
            my $no_pt;
            if ($method eq 'db') {
                eval {$app->partner_db->_get_all("select percent_traffic from $table limit 0");};
                if ($@) {
                    $no_pt = 1;
                }
            }
            push @list,
              {
                accessor => $accessor,
                table    => $table,
                page     => $model->get_page_id_field_name,
                storage  => $method,
                prefix   => $model->public_id_prefix,
                deleted  => $model->get_multistate_by_name('deleted'),
                skip     => ($s->{$MAX_REVENUE_STRATEGY_ID} ? 0 : 1),
                no_pt    => $no_pt,
              };
        }
    }

    return \@list;
}

sub get_db_data {
    my ($app, $row) = @_;

    my $pt = $row->{no_pt} ? '' : "$filter_by_sm{$row->{storage}}{c} as `percent_traffic`,";
    my $sql = "select
    `block`.`$row->{page}` as page_id,
    `block`.`id` as block_id,
    `block`.`is_custom_bk_data` as `gm`,
    `block`.`bk_data`->>'\$.AlternativeTraffPart' as `AlternativeTraffPart`,
    $pt
    if(`block`.`multistate` & $row->{deleted}, 1, 0) as `deleted`,
    all_pages.login,
    all_pages.domain
    from `$row->{table}` as `block`
    join all_pages on all_pages.page_id = `block`.`$row->{page}`
    where
        (`block`.`is_custom_bk_data` = 0 and $filter_by_sm{$row->{storage}}{f})
        or
        (`block`.`is_custom_bk_data` = 1 and `block`.`bk_data`->>'\$.OptimizeType' = 2)
";

    my $list = $app->partner_db->_get_all($sql);
    for my $r (@$list) {
        $r->{public_id} = "$row->{prefix}$r->{page_id}-$r->{block_id}";
    }

    return $list;
}

sub get_db_data_with_money {
    my ($app, $row) = @_;

    my $list = get_db_data($app, $row);

    my @mf;
    for my $r (@$list) {
        push @mf, "($r->{page_id},$r->{block_id})";
    }
    my $money = get_money(\@mf, '2020-05-01');
    for my $r (@$list) {
        $r->{money} = $money->{$r->{page_id}}{$r->{block_id}} // 0;
    }

    return $list;
}

sub to_csv {
    my ($header, $data) = @_;

    my $separator = "\t";
    my $out = join($separator, @$header) . "\n";
    for my $rec (@$data) {
        my @row = @{$rec}{@$header};
        $out .= join($separator, @row) . "\n";
    }
    return $out;
}
1;
