#!/usr/bin/perl

use lib::abs qw(
  ../../lib
  ../../local/lib/perl5/
  );

use qbit;
use Utils::Oneshots::AddBlockBySample;
use Utils::ScriptWrapper 'oneshot';
use PiConstants qw(
  $SELFEMPLOYED_STATUS_READY
  );

run(\&main);

sub args {
    my ($opts) = @_;

    return (
        'logins=s@' => \$opts->{logins},
        'input=s'   => \$opts->{input},
    );
}

sub prepare_args {
    my ($opts) = @_;

    my @list;
    if ($opts->{logins}) {
        @list = grep {$_} split /\s*,\s*/, join ',', @{$opts->{logins}};
    } elsif ($opts->{input}) {
        unless (-r $opts->{input}) {
            die "Wrong file path '$opts->{input}'\n";
        } else {
            open(my $FH, "<", $opts->{input});
            while (my $str = <$FH>) {
                $str =~ s/\s+//g;
                push @list, grep {$_} split /,/, $str;
            }
            close $FH;
        }
    }

    $opts->{login_list} = \@list;
}

sub main {
    my ($app, $opts) = @_;
    for my $login (@{$opts->{login_list}}) {
        my $user = $app->users->get_by_login($login, fields => ['id', 'client_id', 'login', 'self_employed', 'inn']);
        complete_self_employed($app, $user, $opts);
    }
}

sub complete_self_employed {
    my ($app, $user, $opts) = @_;

    throw Exception::Validation::BadArguments gettext('Missed required parameter "%s"', 'client_id')
      unless $user->{client_id};

    my $offer = $app->documents->get_contracts(client_id => $user->{client_id}, need_signed_offer_ru => 1);
    unless ($offer) {
        print logstr("NO_OFFER", $user);
        return;
    }

    my @list =
      sort {$b->{update_dt} cmp $a->{update_dt}} grep {$_->{collateral_type_id} eq '2160'} @{$offer->{Collaterals}};

    if ($list[0]{selfemployed}) {
        print logstr("HAS_SE", $user);
    } else {
        print logstr("NO_SE", $user);

        my $request_id = find_request_id($app, $user->{id});
        my $ok = check_request_id($app, $user->{id}, $request_id);

        print logstr 'NOT_SE_IN_FNS', $user unless $ok;

        if ($ok && $opts->{commit}) {
            try {
                $app->api_balance->create_collateral_selfemployed(
                    operator_uid => $user->{id},
                    contract_id  => $offer->{Contract}->{contract2_id},
                );

                $app->users->do_action(
                    $user, 'edit',
                    self_employed            => $SELFEMPLOYED_STATUS_READY,
                    self_employed_request_id => $request_id,
                );
            }
            catch {
                my ($e) = @_;
                print logstr("FAIL", $user, $e->message);
            };
        }
    }
}

sub find_request_id {
    my ($app, $user_id) = @_;

    return $app->partner_db->query->select(
        table => $app->users->_action_log_db_table(),
        fields =>
          {request_id => {JSON_UNQUOTE => [{MAX => [{JSON_EXTRACT => ['opts', \'$.self_employed_request_id']}]}]},},
        filter => [elem_id => '=' => \$user_id],
    )->get_all()->[0]{request_id};
}

sub check_request_id {
    my ($app, $user_id, $request_id) = @_;

    my $result = FALSE;
    try {
        my $request_status = $app->api_selfemployed_oebs->bind_status($request_id);
        print logstr $user_id, $request_id, $request_status;
        $result = TRUE if ('COMPLETED' eq $request_status->{resultCode});
    }
    catch {
        print logstr $user_id, "$@";
    };

    return $result;
}
