#!/usr/bin/perl

use File::Slurp qw(read_file);

use lib::abs qw(
  ../../lib
  );

use qbit;
use Utils::ScriptWrapper;

my %MOL_DIMENSION_FIELDS = ();

sub args {
    my ($opts) = @_;

    return (
        'user_ids=s'       => \$opts->{user_ids},
        'report_ids=s'     => \$opts->{report_ids},
        'report_id_file=s' => \$opts->{report_id_file}
    );
}

sub prepare_args {
    my ($opts) = @_;

    if ($opts->{user_ids}) {
        $opts->{user_ids} = [split /,/, $opts->{user_ids}];
    }
    if ($opts->{report_ids}) {
        $opts->{report_ids} = [split /,/, $opts->{report_ids}];
    }
}

run(
    sub {
        my ($app, $opts) = @_;

        my @user_id_list = $opts->{user_ids}   ? @{$opts->{user_ids}}   : ();
        my @report_ids   = $opts->{report_ids} ? @{$opts->{report_ids}} : ();
        my @reports_to_keep = ();

        if (defined $opts->{report_id_file} && -f $opts->{report_id_file}) {
            @reports_to_keep =
              map {0 + $_}
              grep {$_ =~ /^\d+$/} @{from_json(read_file($opts->{report_id_file}))};
        } else {
            print STDERR logstr('!cant read report_id_file: ' . ($opts->{report_id_file} // 'UNDEF'));
        }

        %MOL_DIMENSION_FIELDS = map {$_ => 1} (
            (map {'date|' . $_} keys %{$app->bk_statistics->_get_date_groups()}),
            (map {'geo|' . $_} keys %{$app->bk_statistics->_get_region_levels()})
        );

        my $mol_fields        = _get_mol_fields($app);
        my $mol_entity_fields = _get_mol_entity_fields($app);

        my %block_model = map {$_ => 1} @{$app->product_manager->get_block_model_accessors};
        my %page_model  = map {$_ => 1} @{$app->product_manager->get_page_model_accessors};

        #select json_widgets_settings->'$[*].settings.report_id' as aa from widgets limit 100
        my %widget_report_id_list;
        my $use_as_widget_list = $app->partner_db->widgets->get_all(
            (scalar @user_id_list ? (filter => ['AND', [['user_id', 'IN', \\@user_id_list]]],) : ()),
            fields => [qw(json_widgets_settings)]);
        for my $widget (@$use_as_widget_list) {
            if (defined $widget->{json_widgets_settings}) {
                for my $settings (grep {defined $_->{settings}{report_id} && $_->{settings}{report_id} =~ /^\d+$/}
                    @{from_json($widget->{json_widgets_settings})})
                {

                    $widget_report_id_list{$settings->{settings}{report_id}} = $settings->{settings}{show_type}
                      // 'chart';
                }
            }
        }

        my @already_converted_report_list = map {$_->{report_id}} @{
            $app->partner_db->statistics_reports->get_all(
                fields => [qw(report_id)],
                filter => [
                    'AND',
                    [['report_type', '=', \'mol'], ['owner_id', 'IS NOT', \undef], ['report_id', 'IS NOT', \undef]]
                ],
            )
          };

        my $filter = [
            ['id', 'NOT IN', \\@already_converted_report_list],
            # not deleted
            {'not' => [['multistate' => '&' => \4]]},
            ['report_type', '=',      \'pi'],
            ['owner_id',    'IS NOT', \undef],
            [
                'OR' => [
                    # reports with more than 1 visit
                    ['id', 'IN', \\@reports_to_keep],
                    # reports with widgets on dashboard
                    ['id', 'IN', \[keys %widget_report_id_list]],
                ]
            ],
        ];
        push @$filter, ['owner_id', '=', \\@user_id_list] if scalar @user_id_list;
        push @$filter, ['id',       '=', \\@report_ids]   if scalar @report_ids;

        my $user_report_list = $app->partner_db->statistics_reports->get_all(
            fields => [qw(id caption description level query owner_id not_show multistate)],
            filter => ['AND' => $filter],
        );

        my %unsupported_level = ();

        for my $report (@$user_report_list) {
            my %converted_report = %$report;
            $converted_report{report_id}   = delete $converted_report{id};
            $converted_report{report_type} = 'mol';

            my $can_convert = 1;
            print logstr('~REPORT_ID=' . $report->{id});

            my $level          = $report->{level};
            my $level_accessor = ('payment' eq $level || 'charging' eq $level) ? undef : "statistics_$level";
            my $product_level  = $level_accessor ? $app->$level_accessor->product->accessor : $level;

            if (exists $unsupported_level{$level}) {
                $can_convert = 0;
                next;
            } else {
                my $query = from_json($report->{query});
                $converted_report{query} = $query;
                delete $converted_report{query}{isLevelsDifferent};
                for my $chart_table (($query->{tableLevel}[0], $query->{chartLevel}[0])) {
                    next unless $chart_table;
                    $chart_table->{stat_type} = 'dsp' if $level eq 'dsp';
                    $chart_table->{stat_type} = 'mm'  if $level eq 'mobile_mediation_block';

                    my $level_filter = _get_level_filter($product_level, \%block_model, \%page_model);
                    unless ($level_filter) {
                        print logstr('WARN: NO_LEVEL_FILTER: id='
                              . $report->{id}
                              . ' level='
                              . $level
                              . ' product='
                              . $product_level);
                    }

                    if ($chart_table->{levels}[0]{filter}[0]) {
                        my @filter_mol;
                        for my $filter (@{$chart_table->{levels}[0]{filter}[1]}) {
                            if ('MATCH' eq $filter->[1]) {
                                $filter = $filter->[2];
                            }
                            my $converted_field =
                              _guess_mol_field($mol_entity_fields, $filter->[0], $product_level, \%block_model,
                                \%page_model);
                            if ($converted_field && 'SKIP' ne $converted_field) {
                                $filter->[0] = $converted_field;
                                push @filter_mol, $filter;
                            } else {
                                print logstr('WARN: SKIP_FILTER', $filter);
                            }
                        }
                        push @filter_mol, $level_filter if $level_filter;
                        $chart_table->{levels}[0]{filter}[1] = scalar @filter_mol ? \@filter_mol : undef;
                        $chart_table->{levels}[0]{id} = 'payment';
                    } else {
                        if ($level_filter) {
                            $chart_table->{levels}[0]{filter} = ["AND", [$level_filter]];
                        } else {
                            $chart_table->{levels}[0]{filter} = undef;
                        }
                        $chart_table->{levels}[0]{id} = 'payment';
                    }

                    if ($can_convert) {

                        my @from_dimension_fields = grep {$_ !~ /^date\|/} @{$chart_table->{dimension_fields}};

                        my %fields_converted;
                        for my $field (@{$chart_table->{fields}}) {
                            my $converted_field =
                              _guess_mol_field($mol_fields, $field, $product_level, \%block_model, \%page_model);
                            if ($converted_field) {
                                if ('SKIP' ne $converted_field) {
                                    $fields_converted{$converted_field}++;
                                } else {
                                    print logstr('SKIPPED: ' . $field . '->' . $converted_field . ' lEVEL: ' . $level);
                                }
                            } else {
                                $can_convert = 0;
                            }
                        }
                        unless (keys %fields_converted) {
                            print logstr('ERROR: NO_COUNTABLE_FIELDS');
                            $can_convert = 0;
                        } else {
                            $chart_table->{fields} = [keys %fields_converted];
                        }

                        my %entity_fields_converted;
                        for my $entity_field ((@{$chart_table->{entity_fields}}, @from_dimension_fields)) {
                            my $converted_field =
                              _guess_mol_field($mol_entity_fields, $entity_field, $product_level, \%block_model,
                                \%page_model);
                            if ($converted_field) {
                                if ('SKIP' ne $converted_field) {
                                    $entity_fields_converted{$converted_field}++;
                                } else {
                                    print logstr(
                                        'SKIPPED: ' . $entity_field . '->' . $converted_field . ' lEVEL: ' . $level);
                                }
                            } else {
                                $can_convert = 0;
                            }
                        }
                        $chart_table->{entity_fields} = [keys %entity_fields_converted];
                        unless (keys %entity_fields_converted) {
                            print logstr('WARN: NO_FIELDS');
                        }

                        # GROUP BY
                        $chart_table->{dimension_fields} =
                          [grep {exists $MOL_DIMENSION_FIELDS{$_}} @{$chart_table->{dimension_fields}}];
                        delete $chart_table->{dimension_filter};

                        # ORDER BY
                        my %order_by;
                        for my $entry (@{$chart_table->{order_by}}) {
                            my $converted_field =
                              _guess_mol_field($mol_fields, $entry->{field}, $product_level, \%block_model,
                                \%page_model);
                            if ($converted_field && 'SKIP' ne $converted_field) {
                                $order_by{$converted_field} = {
                                    dir   => $entry->{dir},
                                    field => $converted_field,
                                };
                            }
                        }
                        $chart_table->{order_by} = [values %order_by];

                        # LIMITS
                        unless (exists $chart_table->{limits}) {
                            $chart_table->{limits} = {
                                limit  => 50,
                                offset => 0,
                            };
                        }

                        # PERIOD
                        if (exists $chart_table->{period} && 'ARRAY' eq ref $chart_table->{period}) {
                            if ('ARRAY' eq ref $chart_table->{period}[1] || 'ARRAY' eq ref $chart_table->{period}[0]) {
                                $can_convert = 0;    # period compare is not ready yet
                            } else {
                                if (
                                    dates_delta_days(
                                        $chart_table->{period}[0],
                                        $chart_table->{period}[1],
                                        iformat => 'db'
                                    ) > 365 * 2
                                   )
                                {
                                    print logstr('WARN: PERIOD_CHANGED');
                                    $chart_table->{period}[0] =
                                      trdate('norm', 'db',
                                        date_sub(trdate('db', 'norm', $chart_table->{period}[1]), day => 365 * 2 - 1));
                                }
                            }
                        }
                    }
                }
            }

            if ($can_convert) {
                my $is_report_ok = _is_report_ok_in_mol($app, %converted_report);
                if ($is_report_ok) {
                    $converted_report{query}{$_} = $converted_report{query}{$_}[0] foreach (qw(tableLevel chartLevel));

                    if (exists $widget_report_id_list{$report->{id}}) {
                        if ('chart' eq $widget_report_id_list{$report->{id}}) {
                            $converted_report{query}{tableLevel} = $converted_report{query}{chartLevel};
                        } elsif ('table' eq $widget_report_id_list{$report->{id}}) {
                            delete $converted_report{query}{chartLevel};
                        }
                    } else {
                        delete $converted_report{query}{chartLevel};
                    }

                    $converted_report{query}      = to_json($converted_report{query});
                    $converted_report{multistate} = 0;
                    $converted_report{level}      = 'payment';

                    my $new_report_id = $app->statistics_reports->add(%converted_report);
                    $app->partner_db->statistics_reports->edit($new_report_id, {multistate => $report->{multistate}});
                    print logstr('OK_REPORT_ID=' . $report->{id});
                }
            } else {
                print logstr('FAIL_REPORT_ID=' . $report->{id});
            }
        }
    }
   );

sub _get_level_filter {
    my ($level, $block_model, $page_model) = @_;

    return unless $level;
    return if $level eq 'mobile_mediation_block';

    if (exists $block_model->{$level}) {
        return ['block_level', 'IN', [$level]];
    }
    if (exists $page_model->{$level}) {
        return ['page_level', 'IN', [$level]];
    }
}

sub _guess_mol_field {
    my ($mol_fields, $field, $level, $block_model, $page_model) = @_;
    return unless $field && $level;

    $level =~ s/^(?:advnet_|internal_advnet_)(\w+)$/$1/;

    my $is_block = exists $block_model->{$level};
    my $is_page  = exists $page_model->{$level};

    my %h = (
        # entity_fields
        'campaign_id'      => 'page_id',
        'app_bundle_id'    => 'domain',
        'application_id'   => $level eq 'mobile_mediation_block' ? 'application_id' : 'SKIP',
        'block_type_label' => 'SKIP',
        'caption' => $level eq 'mobile_mediation_block'
        ? 'SKIP'
        : ($is_block ? 'block_caption' : ($is_page ? 'page_caption' : undef)),
        'direct_block'       => 'SKIP',
        'id'                 => $is_block ? 'public_id' : ($is_page ? 'page_id' : ($level eq 'dsp' ? 'dsp_id' : undef)),
        'media_formats_list' => 'SKIP',
        'name'               => 'SKIP',
        'owner_name'         => 'SKIP',
        'place_id'           => 'SKIP',
        'public_id'          => 'complex_block_id',
        'seller_name'        => 'SKIP',
        'short_caption'      => 'dsp_caption',
        'store_id' => $level eq 'mobile_mediation_block' ? 'store_id' : 'domain',
        "type" => 'SKIP',
        "type_label" => 'SKIP',

        # countable_fields
        'direct_context_clicks'                                    => 'clicks_direct',
        'direct_context_cpm_direct_context_all_wo_nds'             => 'cpmv_all_wo_nds',
        'direct_context_cpm_direct_context_partner_wo_nds'         => 'cpmv_partner_wo_nds',
        'direct_context_cpmh_direct_context_partner_wo_nds'        => 'ecpm_partner_wo_nds',
        'direct_search_clicks'                                     => 'clicks_direct',
        'direct_search_cpm_direct_search_all_wo_nds'               => 'cpmv_all_wo_nds',
        'direct_search_cpm_direct_search_partner_wo_nds'           => 'cpmv_partner_wo_nds',
        'direct_search_cpmh_direct_search_partner_wo_nds'          => 'ecpm_partner_wo_nds',
        'direct_search_ctr'                                        => 'ctr_direct',
        'market_api_context_clicks'                                => 'clicks_direct',
        'market_api_context_cpm_market_api_context_partner_wo_nds' => 'cpmv_partner_wo_nds',
        'market_api_context_ctr'                                   => 'ctr_direct',
        'premium_clicks'                                           => 'clicks_direct',
        'premium_ctr'                                              => 'ctr_direct',
        'stripe_clicks'                                            => 'clicks_direct',
        'stripe_ctr'                                               => 'ctr_direct',

        'direct_context_cpc_direct_context_partner_wo_nds' => 'SKIP',
        'direct_context_ctr'                               => 'ctr_direct',
        'direct_search_cpc_direct_search_partner_wo_nds'   => 'SKIP',
        'dsp_bids'                                         => 'hits',
        'instream_block_ctr'                               => 'ctr_direct',
        'instream_block_midroll_conversion_percent'        => 'SKIP',
        'instream_block_open_player'                       => 'SKIP',
        'instream_block_pauseroll_conversion_percent'      => 'SKIP',
        'instream_block_postroll_conversion_percent'       => 'SKIP',
        'instream_block_preroll_conversion_percent'        => 'SKIP',
        'instream_block_view'                              => 'hits_render',
        'market_search_clicks'                             => 'clicks_direct',
        'mcb_search_clicks'                                => 'clicks_direct',
        'mcb_search_cpm_mcb_search_partner_wo_nds'         => 'cpmv_partner_wo_nds',
        'mcb_search_ctr'                                   => 'ctr_direct',
        'premium_cpc_partner_wo_nds'                       => 'SKIP',
        'stripe_cpc_partner_wo_nds'                        => 'SKIP',

        'an_campaign_caption'      => 'page_caption',
        'categories'               => 'SKIP',
        'context_campaign_caption' => 'page_caption',
        'currency_id'              => 'SKIP',
        'internal_mobile_caption'  => 'page_caption',
        'mobile_block_type_label'  => 'SKIP',
        'mobile_caption'           => $level eq 'mobile_mediation_block' ? 'mobile_caption' : 'page_caption',
        'search_campaign_caption'  => 'page_caption',
        'tag_caption'              => 'tag_id',
        'video_caption'            => 'page_caption',

        'direct_context_cpc_direct_context_all_wo_nds'             => 'SKIP',
        'direct_search_cpc_direct_search_all_wo_nds'               => 'SKIP',
        'market_api_context_cpc_market_api_context_partner_wo_nds' => 'SKIP',
        # DSP
        'all_real_price_wo_nds'              => 'price_dsp',
        'average_rate_all_real_price_wo_nds' => 'price_dsp_average',
        'dsp_cpm_charging_wo_nds'            => 'cpmv_all_wo_nds',
        ($level eq 'dsp' ? ('dsp_charging_wo_nds' => 'all_wo_nds') : ()),
        # MobMed
        "mobile_mediation_block_calculated_revenue_original" => 'revenue_original_mm_rub',
        "mobile_mediation_block_calculated_revenue"          => 'revenue_original_mm_rub',
        "monetizer_caption"                                  => 'monetizer_caption',
        "geo_name"                                           => 'geo_name',
        "mobile_mediation_block_impressions"                 => 'impressions_mm',
        "platform"                                           => 'platform',
        "place_id"                                           => 'place_id',
        "mobile_mediation_block_visibility"                  => 'SKIP',
        'mobile_mediation_block_hits'                        => 'hits_mm',
        ($level eq 'mobile_mediation_block' ? ('page_caption' => 'mobile_caption', 'page_id' => 'mobmed_page_id') : ())
    );

    if (exists $h{$field}) {
        return $h{$field};
    } elsif ($field =~ /_cover_ratio$/) {
        return 'fillrate';
    } elsif ($field =~ /_direct_clicks$/) {
        return 'clicks_direct';
    } elsif ($field =~ /_direct_ctr$/) {
        return 'ctr_direct';
    } elsif ($field =~ /_cpm_all_wo_nds$/) {
        return 'cpmv_partner_wo_nds';
    } elsif ($field =~ /_cpm_all_wo_nds$/) {
        return 'cpmv_all_wo_nds';
    } elsif ($field =~ /_cpmh_partner_wo_nds$/) {
        return 'ecpm_partner_wo_nds';
    } elsif ($field =~ /_cpmh_all_wo_nds$/) {
        return 'ecpm_all_wo_nds';
    } elsif ($field =~ /_cpm_partner_wo_nds$/) {
        return 'cpmv_partner_wo_nds';
    }

    #$level
    my $entity_fields_map = {

    };

    $level =~ s/^(?:advnet_|internal_advnet_)(\w+)$/$1/;
    $field =~ s/^(?:\w+_block_)(\w+)$/$1/;

    if (exists $mol_fields->{$field}) {
        return $field;
    } else {
        if (my @t = grep {$field =~ /$_/} sort {$b cmp $a} keys %$mol_fields) {
            unless ($field =~ /cpm|ctr|rpm|cpc/) {
                return $t[0];
            } else {
                if (my @tt = grep {$_ =~ /cpm|ctr|rpm|cpc/} @t) {
                    return $tt[0];
                }
            }
        }
    }
}

sub _get_mol_fields {
    my ($app) = @_;

    $app->set_cur_user({id => 763068973});
    return {map {$_->{id} => 1} @{$app->bk_statistics->_get_tree2__fields()}};
}

sub _get_mol_entity_fields {
    my ($app) = @_;

    $app->set_cur_user({id => 763068973});
    return {map {$_->{id} => 1} @{$app->bk_statistics->_get_tree2__entity_fields()},};
}

sub _is_report_ok_in_mol {
    my ($app, %report) = @_;

    return FALSE unless ($report{query}{tableLevel}[0] || $report{query}{chartLevel}[0]);

    $app->set_cur_user({id => $report{owner_id}});

    my $table_chart = {};
    for my $level (qw(tableLevel chartLevel)) {
        my $query_md5 = to_json($report{query}{$level}[0], canonical => 1);
        try {
            print logstr($report{query}{$level}[0] // "no-$level-in-report");
            if ($report{query}{$level}[0]) {
                unless (exists $table_chart->{$query_md5}) {
                    $table_chart->{$level} = $table_chart->{$query_md5} =
                      $app->bk_statistics->get_statistics2(%{$report{query}{tableLevel}[0]});
                } else {
                    $table_chart->{$level} = $table_chart->{$query_md5};
                }
            } else {
                $table_chart->{$level} = "no-$level-in-report";
            }
        }
        catch {
            my ($e) = @_;
            print logstr('MOL_FAIL_REPORT_ID=' . $report{report_id});
            print logstr($e->message);
        };
    }
    return $table_chart->{tableLevel} && $table_chart->{chartLevel};
}
