#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  bin/oneshots/PI-24840_validate_native_template.pl

=cut

use lib::abs qw(../../lib);

use LWP::UserAgent;

use qbit;
use Pod::Usage;
use Utils::ScriptWrapper 'util';

my %product_id2prefix;

sub get_prefix {
    my ($app, $accessor) = @_;

    $product_id2prefix{$accessor} //= $app->$accessor->public_id_prefix;

    return $product_id2prefix{$accessor};
}

run(\&main);

sub args {
    my ($opts) = @_;
    return (
        'block_in:s'     => \$opts->{block_in},
        'block_out:s'    => \$opts->{block_out},
        'result:s'       => \$opts->{result},
        'design_part:i'  => \$opts->{design_part},
        'money_part:i'   => \$opts->{money_part},
        'design_limit:i' => \$opts->{design_limit},
        'model_limit:i'  => \$opts->{model_limit},
        'skip_money!'    => \$opts->{skip_money},
    );
}

sub prepare_args {
    my ($opts) = @_;
    $opts->{design_part} //= 10000;
    $opts->{money_part}  //= 400;
    $opts->{result}      //= 'native_block_validator_fail.txt';
    $opts->{block_out}   //= 'native_block_list.json';
}

sub main {
    my ($app, $opts) = @_;

    my $start = time;
    silent();

    my $block_list = make_block_list($app, $opts);
    my $get_list = time;

    my $date_to = date_sub(curdate(), day => 1, oformat => 'date_only_numbers');
    my $date_from = date_sub($date_to, day => 7, iformat => 'date_only_numbers', oformat => 'date_only_numbers');
    if ($opts->{skip_money}) {
        $_->{money} = 0 for values %$block_list;
    } else {
        update_money($app, $block_list, $opts->{money_part}, $date_from, $date_to);
    }
    my $get_money = time;

    out_result_tsv($block_list, $opts->{result});
    print logstr "TIMING", int(($get_list - $start) / 60), int(($get_money - $get_list) / 60);

    return 0;
}

sub make_block_list {
    my ($app, $opts) = @_;

    my $block_list;
    if ($opts->{block_in}) {
        $block_list = from_json(readfile($opts->{block_in}));
    } else {
        $block_list = {};
        get_list_block_from_templates($app, $opts, $block_list);
        for my $accessor (get_list_model($app)) {
            get_list_block_from_model($app, $opts, $accessor, $block_list);
        }
        if ($opts->{block_out}) {
            writefile($opts->{block_out}, to_json($block_list, pretty => TRUE));
            print logstr "STORE_LIST", $opts->{block_out};
        }
    }

    return $block_list;
}

sub update_money {
    my ($app, $block_list, $split, $date_from, $date_to) = @_;

    my @blocks = keys %$block_list;
    while (my @part = splice @blocks, 0, $split) {
        print logstr "ASK_MONEY", scalar(@part), scalar(@blocks);
        try {
            my $money = get_money_part($app, \@part, $date_from, $date_to);
            for my $key (keys %$money) {
                $block_list->{$key}{money} = $money->{$key};
            }
        }
        catch {
            my ($e) = @_;
            print logstr "FAIL_MONEY", $e->message;
        };
    }

    return;
}

sub get_money_part {
    my ($app, $list, $date_from, $date_to) = @_;

    my %ids;
    for my $row (@$list) {
        my @row = split /-/, $row;
        $ids{$row[0]}{$row[1]}{$row[2]} = undef;
    }

    my @filter = ({PageID => {eq => [keys %ids]},});

    my $data = $app->api_http_mol->get_data(
        countable_fields => ["Price"],
        currency         => "RUB",
        date_from        => $date_from,
        date_to          => $date_to,
        filters_pre      => {'-and' => \@filter},
        group_by         => ["PageID", "ImpID", "DesignID"],
    );

    my %h;
    for my $i (0 .. $#{$data->{header}}) {
        $h{$data->{header}[$i]} = $i;
    }

    my %result;
    my $needed = 0;
    for my $row (@{$data->{data}}) {
        next
          unless exists $ids{$row->[$h{PageID}]}{$row->[$h{ImpID}]}
              and exists $ids{$row->[$h{PageID}]}{$row->[$h{ImpID}]}{$row->[$h{DesignID}]};
        $needed++;
        my $key = join '-', @$row[@h{qw(PageID ImpID DesignID)}];
        $result{$key} = 0 + sprintf("%.2f", $row->[$h{Price}] / 1_000_000);
    }
    for my $row (@$list) {
        $result{$row} //= 0;
    }
    print logstr "FETCH_MONEY", $needed, scalar(@$list), $data->{total_rows}, $data->{duration};

    return \%result;
}

sub get_list_model {
    my ($app) = @_;

    my @models;
    for my $accessor (@{$app->product_manager->get_block_model_accessors}) {
        if ($app->$accessor->DOES('Application::Model::Role::Block::Has::ContentWidgetCommon')) {
            push @models, $accessor;
        }
    }

    return sort @models;
}

sub get_list_block_from_model {
    my ($app, $opts, $accessor, $block_list) = @_;

    print logstr "LIST_MODEL", $accessor;

    my $pn = $app->$accessor->get_page_id_field_name;

    my $table = $app->partner_db->$accessor;
    my $query = $app->partner_db->query->select(
        table  => $table,
        filter => [AND => [[multistate => '&' => \2]]],
        fields => {
            page_id           => $pn,
            block_id          => 'id',
            template          => {json_unquote => [{json_extract => ['opts', \'$.template']}]},
            is_custom_bk_data => '',
            bk_data           => '',
        },
      )->join(
        table   => $app->partner_db->all_pages,
        fields  => ['login'],
        join_on => [AND => [["page_id" => '=' => {$pn => $table}],]],
        filter => ["is_working" => "=" => \1]
      );
    $query = $query->limit($opts->{model_limit}) if $opts->{model_limit};
    my $list = $query->get_all();

    print logstr "CHECK_MODEL", $accessor, scalar(@$list);
    for my $row (@$list) {
        my $template;
        my $gm = 0;
        if ($row->{is_custom_bk_data}) {
            try {
                $template = from_json($row->{bk_data})->{CustomBlockData}{WidgetInfo}{Template};
                $gm       = 1;
            }
            catch {
                my ($e) = @_;
                print logstr "FAIL_GM", $e->message, $row;
            };
        } else {
            $template = $row->{template};
        }
        if (my $m = check_template($app, $template)) {
            my $key = join '-', $row->{page_id}, $row->{block_id}, 0;
            $block_list->{$key} = {
                page_id   => $row->{page_id},
                block_id  => $row->{block_id},
                login     => $row->{login},
                msg       => $m,
                prefix    => get_prefix($app, $accessor),
                tpl       => $template,
                msg       => $m,
                gm        => $gm,
                design_id => 0,
            };
        }
    }
}

sub get_list_block_from_templates {
    my ($app, $opts, $block_list) = @_;

    my $last_id = 0;
    my $count   = $opts->{design_limit};
    while (1) {
        last if $opts->{design_limit} && --$count < 0;
        print logstr "LIST_DESIGN", $last_id;
        my $list = get_part_list_templates($app, $last_id, $opts);
        last unless @$list;
        print logstr "CHECK_DESIGN", $last_id, scalar(@$list);

        for my $row (@$list) {
            my %by_id;
            my $gm = 0;
            my $prefix = get_prefix($app, $row->{e_id} ? $row->{e_model} : $row->{i_model});
            if ($row->{e_is_custom_bk_data} || $row->{i_is_custom_bk_data}) {
                try {
                    $gm = 1;
                    my $bk_data = from_json($row->{e_bk_data} || $row->{i_bk_data});
                    my $designs = $bk_data->{RtbDesign};
                    for my $id (sort {$a <=> $b} keys %$designs) {
                        if ($designs->{$id}{type} && $designs->{$id}{type} eq 'native') {
                            if (my $template = $designs->{$id}{design}{template}) {
                                if (my $m = check_template($app, $template)) {
                                    $by_id{$id}{msg} = $m;
                                    $by_id{$id}{tpl} = $template;
                                }
                            } else {
                                $by_id{$id}{msg} = ['empty template'];
                                $by_id{$id}{tpl} = 'empty template';
                            }
                        }
                    }
                }
                catch {
                    my ($e) = @_;
                    print logstr "FAIL_GM", "$prefix$row->{page_id}-$row->{block_id}", $e->message;
                };
            } elsif (my $m = check_template($app, $row->{template})) {
                $by_id{$row->{d_id}}{msg} = $m;
                $by_id{$row->{d_id}}{tpl} = $row->{template};
            }
            if (%by_id) {
                for my $id (keys %by_id) {
                    my $key = join '-', $row->{page_id}, $row->{block_id}, $id;
                    unless ($block_list->{$key}) {
                        $block_list->{$key} = {
                            page_id   => $row->{page_id},
                            block_id  => $row->{block_id},
                            design_id => $id,
                            login     => $row->{login},
                            prefix    => $prefix,
                            gm        => $gm,
                            msg       => $by_id{$id}{msg},
                            tpl       => $by_id{$id}{tpl},
                        };
                    }
                }
            }
            $last_id = $row->{d_id};
        }
    }

    return;
}

sub get_part_list_templates {
    my ($app, $last_id, $opts) = @_;

    my $table = $app->partner_db->design_templates;
    my $query = $app->partner_db->query->select(
        table  => $table,
        filter => [AND => [[type => '=' => \'native'], [id => '>' => \$last_id], [multistate => '=' => \0]]],
        fields => {
            d_id     => 'id',
            page_id  => '',
            block_id => '',
            template => {json_unquote => [{json_extract => ['opts', \'$.design_settings.template']}]},
        },
      )->join(
        table   => $app->partner_db->all_pages,
        fields  => ['login'],
        join_on => [AND => [["page_id" => '=' => {"page_id" => $table}],]],
        filter => ["is_working" => "=" => \1]
      )->left_join(
        table  => $app->partner_db->context_on_site_rtb,
        fields => {
            'e_model'             => \'context_on_site_rtb',
            'e_is_custom_bk_data' => 'is_custom_bk_data',
            'e_bk_data'           => 'bk_data',
            'e_id'                => 'id',
        },
        join_on => [AND => [["id" => '=' => {"block_id" => $table}], ["campaign_id" => '=' => {"page_id" => $table}],]],
        filter => [OR => [["id" => 'IS' => \undef], ["multistate" => "&" => \2],]],
      )->left_join(
        table  => $app->partner_db->internal_context_on_site_rtb,
        fields => {
            'i_model'             => \'internal_context_on_site_rtb',
            'i_is_custom_bk_data' => 'is_custom_bk_data',
            'i_bk_data'           => 'bk_data',
            'i_id'                => 'id',
        },
        join_on => [AND => [["id" => '=' => {"block_id" => $table}], ["campaign_id" => '=' => {"page_id" => $table}],]],
        filter => [OR => [["id" => 'IS' => \undef], ["multistate" => "&" => \2],]],
      )->order_by('d_id')->limit($opts->{design_part});

    return $query->get_all;
}

sub check_template {
    my ($app, $template) = @_;

    my $v;
    try {
        my $rv = $app->api_format_system->validate_native(template => $template);
        unless ($rv->{valid}) {
            $v = [map {$_->{text}} grep {$_->{type} eq 'ERR'} @{$rv->{items}{template}{messages}}];
        }
    }
    catch {
        my ($e) = @_;
        print logstr "FAIL_VALIDATE", $e->message;
    };

    return $v;
}

sub silent {

    no warnings 'redefine';
    no strict 'refs';

    require QBit::Application::Model::API::HTTP;
    require Application::Model::API::Yandex::HTTPMOL;

    *{'QBit::Application::Model::API::HTTP::INFO'}      = sub { };
    *{'Application::Model::API::Yandex::HTTPMOL::INFO'} = sub { };
}

sub out_result_tsv {
    my ($blocks, $fn) = @_;

    my %msg_idx;
    my %msg_rev_idx;
    my %msg_cnt;
    my $idx = 0;
    for my $row (values %$blocks) {
        $row->{tpl} //= 'undefined template';
        $row->{tpl} =~ s/[\n\t]/ /g;
        $row->{msg} = [$row->{msg}] unless ref $row->{msg} eq 'ARRAY';
        my @msg;
        for my $msg (@{$row->{msg}}) {
            $msg =~ s/[\n\t]/ /g;
            unless (exists $msg_idx{$msg}) {
                $msg_idx{$msg}     = $idx;
                $msg_rev_idx{$idx} = $msg;
                $idx++;
            }
            my $i = $msg_idx{$msg};
            push @{$msg[$i]}, '*';
            $msg_cnt{$i}++;
        }
        $row->{msg} = \@msg;
    }

    my @trmap = sort {$msg_cnt{$b} <=> $msg_cnt{$a}} (0 .. $idx - 1);

    my $header = ['login', 'gm', 'public_id', 'design_id', 'money', @msg_rev_idx{@trmap}, 'template'];
    open F, '>', $fn;
    print F join("\t", @$header), "\n";
    for my $row (
        sort {
                 $b->{money} <=> $a->{money}
              or $a->{login} cmp $b->{login}
              or $a->{page_id} <=> $b->{page_id}
              or $a->{block_id} <=> $b->{block_id}
              or $a->{design_id} <=> $b->{design_id}
        } values %$blocks
      )
    {
        print F join("\t",
            $row->{login}, $row->{gm}, $row->{prefix} . $row->{page_id} . '-' . $row->{block_id},
            $row->{design_id}, $row->{money}, (map {join ',', @$_} map {$_ // []} @{$row->{msg}}[@trmap]),
            $row->{tpl},),
          "\n";
    }

    close F;
}
