#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

   список полей в которых лежат признаки наличия ошибки определенного типа
   --check_fields=6,10  - берем только блоки в которых есть подобные ошибки
   --skip_fields=19,31  - пропускаем блоки в которых есть такие ошибки
   --limit=10           - сколько блоков брать из списка
   --plimit=10          - какая часть в %% блоков берется из списка

 пример запуска
 perl -Ilib bin/oneshots/PI-25162_fix_native_design.pl --input=native_block_validator_fail.txt --show_header
 perl -Ilib bin/oneshots/PI-25162_fix_native_design.pl --input=native_block_validator_fail.txt \
   --check_fields=6,10 \
   --skip_fields=19,31 \
   --skip_pages="`cat PI-25162_skip_pages.txt`" \
   --skip_blocks="`cat PI-25162_skip_blocks.txt`" \
   --limit=10 --dry_run --ticket=PI-25162

   native_block_validator_fail.txt - файл с выгрузкой получается bin/oneshots/PI-24840_validate_native_template.pl
   также есть готовый в https://st.yandex-team.ru/PI-25162#61549622ab7f6c22114c1f7f
=cut

use lib::abs qw(../../lib);

use LWP::UserAgent;

use qbit;
use Pod::Usage;
use Utils::ScriptWrapper;
use Utils::Oneshots::FixNativeTemplate;
use Utils::PublicID;

my $SV_TURBO = [qw(turbo turbo_desktop)];
run(\&main);

sub args {
    my ($opts) = @_;
    return (
        'input:s'        => \$opts->{input},
        'output:s'       => \$opts->{output},
        'check_fields:s' => \$opts->{check_fields},
        'skip_fields:s'  => \$opts->{skip_fields},
        'finished:s'     => \$opts->{finished},
        'skip_pages:s'   => \$opts->{skip_pages},
        'skip_blocks:s'  => \$opts->{skip_blocks},
        'fix_gm!'        => \$opts->{fix_gm},
        'show_header!'   => \$opts->{show_header},
        'plimit:i'       => \$opts->{plimit},
    );
}

sub prepare_args {
    my ($opts) = @_;

    unless ($opts->{input}) {
        die "need --input=<filename>";
    }
    if ($opts->{show_header}) {
        open F, $opts->{input};
        my @h = split /\t/, scalar <F>;
        close F;
        print "HEADER\n";
        for my $i (0 .. $#h) {
            printf "[%2d]: %s\n", $i, $h[$i];
        }
        die "---------STOPPED after show header------\n";
    }
    if ($opts->{skip_pages}) {
        $opts->{skip_pages} = [split /\s*,\s*/, $opts->{skip_pages}];
        $opts->{skip_pages_idx} = {map {$_ => TRUE} @{$opts->{skip_pages}}};
    }

    if ($opts->{skip_blocks}) {
        $opts->{skip_blocks} = [split /\s*,\s*/, $opts->{skip_blocks}];
    }

    if ($opts->{finished}) {
        my @list;
        for my $file (split /\s*,\s*/, $opts->{finished}) {
            open F, $file;
            while (my $str = <F>) {
                if ($str =~ /DONE_BLOCK\t(\w+-\w+-\d+-\d+)/) {
                    push @{$opts->{skip_blocks}}, $1;
                }
            }
            close F;
        }
    }
    if ($opts->{skip_blocks}) {
        my %idx;
        for my $block (@{$opts->{skip_blocks}}) {
            my @block = split /-/, $block;
            $idx{$block[2]}{$block[3]} = TRUE;
        }
        $opts->{skip_blocks_idx} = \%idx;
    }

    unless ($opts->{check_fields}) {
        die "need --check_fields=1,2";
    }
    $opts->{check_fields} = [split /\s*,\s*/, $opts->{check_fields}];
    if ($opts->{skip_fields}) {
        $opts->{skip_fields} = [split /\s*,\s*/, $opts->{skip_fields}];
    }

    open F, $opts->{input};
    my $h = <F>;
    my %list;
    while (my $str = <F>) {
        # формат строки определяется в bin/oneshots/PI-24840_validate_native_template.pl
        my @line = split /\t/, $str;
        my $skip;
        for my $f (@{$opts->{skip_fields}}) {
            if ($line[$f]) {
                $skip = TRUE;
                last;
            }
        }
        unless (join '', @line[@{$opts->{check_fields}}]) {
            $skip = TRUE;
        }
        if ($line[0] eq 'adinside') {
            $skip = TRUE;
        }
        # god mode
        if ($opts->{fix_gm}) {
            unless ($line[1]) {
                $skip = TRUE;
            }
        } elsif ($line[1]) {
            $skip = TRUE;
        }
        my @block = split /-/, $line[2];
        if ($opts->{skip_pages_idx}{$block[2]}) {
            $skip = TRUE;
        }
        if ($opts->{skip_blocks_idx}{$block[2]}{$block[3]}) {
            $skip = TRUE;
        }
        $list{$line[2]} = undef unless $skip;
    }
    close F;
    $opts->{input_blocks} = [sort keys %list];
    $opts->{output} //= 'native_block.changes.txt';
}

sub main {
    my ($app, $opts) = @_;

    silent();
    print logstr("FOUND", scalar @{$opts->{input_blocks}});

    my $by_model = group_public_ids_by_model($app, $opts->{input_blocks});
    my %changes;
    for my $accessor (keys %$by_model) {
        my @filter = [public_id => IN => $by_model->{$accessor}];
        if ($app->$accessor->DOES('Application::Model::Role::Has::SiteVersion')) {
            push @filter, [site_version => '<>' => $SV_TURBO];
        }

        my $count = 0;
        my $list;
        if ($app->$accessor->DOES('Application::Model::Role::Block::Has::ContentWidgetCommon')) {
            $list = $app->$accessor->get_all(filter => [AND => \@filter], fields => ["public_id", "template"]);
            print logstr("FOUND_MODEL", $accessor, scalar @$list);
            if ($opts->{plimit}) {
                $opts->{limit} = int @$list * $opts->{plimit} / 100;
            }
            for my $block (@$list) {
                try {
                    my $template = Utils::Oneshots::FixNativeTemplate::fix_kebab($block->{template});
                    $template = Utils::Oneshots::FixNativeTemplate::fix_category($template, 1);
                    $template = Utils::Oneshots::FixNativeTemplate::fix_age($template);
                    unless ($template eq $block->{template}) {
                        $app->$accessor->do_action($block->{public_id}, "edit", template => $template)
                          unless $opts->{dry_run};
                        print logstr("DONE_BLOCK", $block->{public_id});
                        $changes{$block->{public_id}} = {0 => {before => $block->{template}, after => $template}};
                    }
                }
                catch {
                    my ($e) = @_;
                    print logstr("FAIL_BLOCK", $block->{public_id}, $e->message);
                };
                last if $opts->{limit} && $opts->{limit} < $count++;
            }
        } elsif ($app->$accessor->DOES('Application::Model::Role::Has::Block::DesignTemplates')) {
            $list = $app->$accessor->get_all(filter => [AND => \@filter], fields => ["public_id", "design_templates"]);
            print logstr("FOUND_MODEL", $accessor, scalar @$list);
            if ($opts->{plimit}) {
                $opts->{limit} = int @$list * $opts->{plimit} / 100;
            }
            for my $block (@$list) {
                try {
                    my %ch;
                    for my $design (@{$block->{design_templates}}) {
                        next unless $design->{type} eq 'native';
                        my $template =
                          Utils::Oneshots::FixNativeTemplate::fix_kebab($design->{design_settings}{template});
                        $template = Utils::Oneshots::FixNativeTemplate::fix_category($template);
                        $template = Utils::Oneshots::FixNativeTemplate::fix_age($template);
                        unless ($design->{design_settings}{template} eq $template) {
                            my $before = $design->{design_settings}{template};
                            $design->{design_settings}{template} = $template;
                            $ch{$design->{id}} = {before => $before, after => $template};
                        }
                    }
                    if (%ch) {
                        $app->$accessor->do_action($block->{public_id}, "edit",
                            design_templates => $block->{design_templates})
                          unless $opts->{dry_run};
                        print logstr("DONE_BLOCK", $block->{public_id});
                        $changes{$block->{public_id}} = \%ch;
                    }
                }
                catch {
                    my ($e) = @_;
                    print logstr("FAIL_BLOCK", $block->{public_id}, $e->message);
                };
                last if $opts->{limit} && $opts->{limit} < $count++;
            }
        } else {
            print logstr("FAIL_ACCESSOR", $accessor);
        }
        print logstr("DONE_MODEL", $accessor, $count, scalar @{$list // []});
    }
    if (%changes) {
        open F, '>', $opts->{output};
        print F join("\t", qw(public_id design_id after before)), "\n";
        for my $block_id (sort keys %changes) {
            for my $design_id (sort {$a <=> $b} keys %{$changes{$block_id}}) {
                print F join("\t",
                    $block_id, $design_id,
                    fix_str($changes{$block_id}{$design_id}{after}),
                    fix_str($changes{$block_id}{$design_id}{before}),
                  ),
                  "\n";
            }
        }
        close F;
    }
}

sub fix_str {
    my ($str) = @_;

    $str //= '';
    $str =~ s/[\n\t]/ /g;

    return $str;
}

sub silent {

    no warnings 'redefine';
    no strict 'refs';

    require QBit::Application::Model::API::HTTP;

    *{'QBit::Application::Model::API::HTTP::INFO'} = sub { };
}
