#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  Скрипт заполняет поле  "patch" у пейджей из YT таблицы //home/yabs-cs-preprod/denkitaev/pi_patch/BSFields_json

=head1 USAGE

  ./bin/oneshots/PI-26375_fill_page_patch_field.pl  --ticket=PI-26375  --over_logs \
    --yt_table_path='//home/partner/tmp/PI-NNNN_patches'  --yt_cluster=hahn \
    --with_protected  --update_patch  --page_ids=14119,83024  --dry_run

=head1 OPTIONS

  Обязательные:
      yt_table_path      - путь к YT таблице
      yt_cluster         - YT кластер
      replace_patch      - перетирвает новым значением
      update_patch       - считает новое значение патчем и патчит текущее значнение

  Опциональные:
      dry_run            - просто выводит что на что собирается заменить
      with_protected     - меняет protected (по дефолту отключено)
      page_ids           - меняет только указанные пейджи
      start_from_page_id - позволяет продолжить после остановки скрипта

=cut

use strict;
use warnings;

use lib::abs qw(../../lib);
use PiConstants qw($PAGE_KEYS_TRANSFORM);

use qbit;
use QBit::Array qw(array_uniq);
use Utils::ScriptWrapper 'oneshot';

run(
    sub {
        my ($app, $opts) = @_;

        _allow_ssp_model_to_edit_field_patch($app);

        my $page_new_patches = get_page_patches_from_yt_table($app, $opts->{yt_table_path}, $opts->{yt_cluster});

        my $page_ids = [split(/,/, $opts->{'page_ids'} // '')];
        $page_ids = [sort keys %$page_new_patches] unless @$page_ids;

        my $model_pages = group_pages_by_model($app, $page_ids, $opts);

        my $page_cur_patches = get_page_patches_from_db($app, $model_pages);

        $opts->{start_from_page_id} //= 0;
        my $is_start_processing = $opts->{start_from_page_id} ? 0 : 1;

        my $REVERSE_KEYS_TRANSFORM = {reverse %$PAGE_KEYS_TRANSFORM};

        my $model_count = 0;
        my $pages_count = 0;
        foreach my $accessor (sort keys %$model_pages) {
            $model_count++;

            print logstr "$model_count.", 'MODEL', $accessor, scalar @{$model_pages->{$accessor}};

            unless ($opts->{dry_run}) {
                # хачим граф, потому что иначе не все площадки получится отредактировать
                my $graph = $app->$accessor->get_multistates_graph_definition;

                my ($edit_action) = grep {$_->{action} eq 'edit'} @{$graph->{multistate_actions}};
                $edit_action->{from} = '__EMPTY__ or not __EMPTY__';

                my ($set_need_update_action) =
                  grep {$_->{action} eq 'set_need_update'} @{$graph->{multistate_actions}};
                $set_need_update_action->{from} = '__EMPTY__ or not __EMPTY__';
                $app->$accessor->multistates_graph($graph);
            }

            my $model_pages_count = 0;
            foreach my $page_id (sort @{$model_pages->{$accessor}}) {
                $model_pages_count++;
                $pages_count++;

                $is_start_processing = 1 if !$is_start_processing && $page_id == $opts->{start_from_page_id};
                next unless $is_start_processing;

                my $cur_patch = from_json($page_cur_patches->{$page_id} // '{}');
                my $new_patch = from_json($page_new_patches->{$page_id} // '{}');

                if ($opts->{update_patch}) {

                    if ($new_patch->{DisabledFlags}) {
                        # потому что при get '*' 'disabled_fields' приводится к массиву
                        $new_patch->{DisabledFlags} = [sort split(/,/, $new_patch->{DisabledFlags})];
                    }

                    my $cur_patch_clone = clone($cur_patch);

                    # конвертим patch в bk_data (PageOptions -> page_options)
                    transform_keys($cur_patch_clone, $REVERSE_KEYS_TRANSFORM);

                    $new_patch = $app->$accessor->apply_bk_data_patch({'patch' => to_json($new_patch, canonical => 1)},
                        $cur_patch_clone);

                    # конвертим обратно, bk_data в patch  (page_options -> PageOptions)
                    transform_keys($new_patch, $PAGE_KEYS_TRANSFORM);

                    # Дедубликацируем PageOptions
                    my $val = $new_patch->{PageOptions};
                    if (defined $val) {
                        foreach my $key (qw(Enable Disable)) {
                            if (exists $val->{$key}) {
                                $val->{$key} = array_uniq($val->{$key});
                            }
                        }
                    }
                }

                my $cur_patch_json = to_json($cur_patch, canonical => 1);
                my $new_patch_json = to_json($new_patch, canonical => 1);

                if ($new_patch_json eq $cur_patch_json) {
                    print logstr "$model_count.$model_pages_count", 'PAGE', $page_id, 'NO CHANGES', $cur_patch_json,
                      'NEW PATCH', $page_new_patches->{$page_id};
                } else {
                    my $exception;
                    unless ($opts->{dry_run}) {
                        try {
                            $app->$accessor->do_action($page_id, 'edit', patch => $new_patch_json);
                        }
                        catch {
                            ($exception) = @_;
                        }
                    }

                    if ($exception) {
                        print STDERR logstr "$model_count.$model_pages_count", 'page', $page_id, 'FAILED', 'MODEL',
                          $accessor, 'exception', $exception->message();
                    } else {
                        print STDOUT logstr "$model_count.$model_pages_count", 'PAGE', $page_id, 'UPDATE',
                          $cur_patch_json,
                          $new_patch_json;
                    }
                }
            }
        }

        print logstr "TOTAL PAGES", $pages_count;
    }
   );

sub transform_keys {
    my ($data, $transform_keys) = @_;

    foreach my $key (keys %$data) {
        my $new_key = $transform_keys->{$key};
        if ($new_key) {
            $data->{$new_key} = delete $data->{$key};
        }
    }

    return 1;
}

sub prepare_args {
    my ($opts) = @_;

    die 'You must specify "yt_table_path"' unless $opts->{yt_table_path};
    die 'You must specify "yt_cluster"'    unless $opts->{yt_cluster};

    $opts->{replace_patch} = $opts->{replace_patch} ? 1 : 0;
    $opts->{update_patch}  = $opts->{update_patch}  ? 1 : 0;
    if ($opts->{replace_patch} == $opts->{update_patch}) {
        die 'You must specify only one of the options "--replace_patch" or "-update_patch"';
    }

    return 1;
}

sub group_pages_by_model {
    my ($app, $page_ids, $opts) = @_;

    my $data = $app->partner_db->all_pages->get_all(
        fields => ["model", "page_id", "is_protected"],
        filter => [page_id => "IN" => \$page_ids],
    );

    my $found_pages = {map {$_->{page_id} => 1} @$data};
    my $not_found_pages = {map {$_ => 1} grep {!exists $found_pages->{$_}} @$page_ids};

    die 'Unknown pages found: ', join(", ", sort keys %$not_found_pages) if %$not_found_pages;

    my $protested_pages = {map {$_->{page_id} => 1} grep {$_->{is_protected}} @$data};
    print logstr
      sprintf('Protected pages found [%d]: %s', scalar(keys %$protested_pages), join(", ", sort keys %$protested_pages))
      if %$protested_pages;

    my $model_pages = {};
    foreach my $row (@$data) {
        next if $row->{is_protected} && !$opts->{with_protected};
        my $pages = $model_pages->{$row->{model}} //= [];
        push @$pages, $row->{page_id};
    }

    my $pages = [map {@$_} values %$model_pages];
    print logstr sprintf("Pages to update [%d]: %s\n", scalar(@$pages), join(", ", sort @$pages));

    print logstr sprintf("Models [%d]: %s\n", scalar(keys %$model_pages), join(", ", sort keys %$model_pages));

    return $model_pages;
}

sub get_page_patches_from_db {
    my ($app, $model_pages) = @_;

    my $db_page_patches = {};

    foreach my $accessor (sort keys %$model_pages) {

        my $table_name         = $app->$accessor->db_table_name();
        my $page_id_field_name = $app->$accessor->get_page_id_field_name();
        my $page_ids           = $model_pages->{$accessor};

        my $data = $app->partner_db->$table_name->get_all(
            fields => [$page_id_field_name, "patch"],
            filter => [$page_id_field_name => "IN" => \$page_ids],
        );

        $db_page_patches->{$_->{$page_id_field_name}} = $_->{patch} foreach @$data;
    }

    return $db_page_patches;
}

sub _allow_ssp_model_to_edit_field_patch {
    my ($app) = @_;

    no warnings 'redefine';

    # 181341
    my $editable_fields = $app->ssp_mobile_app_settings->get_editable_fields();
    *Application::Model::Product::SSP::MobileApp::Settings::get_editable_fields = sub {
        return {%$editable_fields, patch => 1};
    };

    $editable_fields = $app->ssp_context_on_site_campaign->get_editable_fields();
    *Application::Model::Product::SSP::ContextOnSite::Campaign::get_editable_fields = sub {
        return {%$editable_fields, patch => 1};
    };

    return 1;
}

sub get_page_patches_from_yt_table {
    my ($app, $yt_table_path, $cluster) = @_;

    my $data_jsonl = $app->api_yt->read_table_from_any_replica(
        replicas => [$cluster],
        path     => $yt_table_path,
        headers  => {'X-YT-Parameters' => '{output_format=<encode_utf8=%false>json}',},
        headers  => {'Accept' => 'application/json'},
        params   => {
            timeout       => 300,
            attempts      => 1,
            delay         => 0,
            timeout_retry => 0,
        },
    );

    my $data = from_jsonl($data_jsonl);

    my $yt_page_patches = {map {$_->{PageID} => $_->{JsonPatch}} @$data};

    return $yt_page_patches;
}

sub args {
    my ($opts) = @_;

    return (
        'yt_table_path=s'      => \$opts->{yt_table_path},
        'yt_cluster=s'         => \$opts->{yt_cluster},
        'page_ids:s'           => \$opts->{page_ids},
        'start_from_page_id:s' => \$opts->{start_from_page_id},
        'with_protected!'      => \$opts->{with_protected},
        'replace_patch!'       => \$opts->{replace_patch},
        'update_patch!'        => \$opts->{update_patch},
    );
}
