#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  Скрипт для замены дизайнов

=head1 USAGE

  bin/oneshots/PI-27308_remove_carouselif_limit_is_1.pl

=head1 OPTIONS

  designes  - список дизайнов

=cut

use lib::abs qw(
  ../../lib
  ../../local/lib/perl5
  );

use Clone;
use Pod::Usage;

use qbit;
use Utils::DB;
use Utils::PublicID;
use Utils::ScriptWrapper;

my $app;
my $opts;

sub args {
    my ($opts) = @_;
    return (
        'designes=s@'        => \$opts->{designes},
        'god_mode_only!'     => \$opts->{god_mode_only},
        'designes_only!'     => \$opts->{designes_only},
        'page_ids=s@'        => \$opts->{page_ids},
        'block_ids=s@'       => \$opts->{block_ids},
        'limit=i'            => \$opts->{limit},
        'limit_per_design=i' => \$opts->{limit_per_design},
        'percent=f'          => \$opts->{percent},
        'resend=s'           => \$opts->{resend},
    );
}

sub prepare_args {
    my ($opts) = @_;

    $opts->{ticket} ||= 'PI-27308';

    if ($opts->{designes}) {
        my @designes = split /\s*,\s*/, join ',', @{$opts->{designes}};
        my %designes;
        @designes{@designes} = (TRUE) x scalar @designes;
        $opts->{designes} = \%designes;
    }

    if ($opts->{page_ids}) {
        $opts->{page_ids} = [split /\s*,\s*/, join ',', @{$opts->{page_ids}}];
    }

    if ($opts->{block_ids}) {
        $opts->{block_ids} = [split /\s*,\s*/, join ',', @{$opts->{block_ids}}];
        $opts->{block_ids_design} = [
            map {
                my (undef, $page_id, $block_id) = split_block_public_id($_);
                +{page_id => $page_id, block_id => $block_id,}
              } @{$opts->{block_ids}}
        ];
        $opts->{block_ids_godmode} = [
            map {
                my (undef, $page_id, $block_id) = split_block_public_id($_);
                +{campaign_id => $page_id, id => $block_id,}
              } @{$opts->{block_ids}}
        ];
    }

    foreach (qw(rollback resend)) {
        die sprintf('File "%s" does not esist', $opts->{$_}) if $opts->{$_} && !-f $opts->{$_};
    }
}

run(
    sub {
        ($app, $opts) = @_;

        my $pages_to_resend;

        if ($opts->{resend}) {
            my $f;
            if (open $f, '<', $opts->{resend}) {
                while (defined(my $s = <$f>)) {
                    if ($s =~ /PAGES TO RESEND\s+(.+)$/) {
                        $pages_to_resend = from_json $1;
                        last;
                    }
                }
                close $f;
            } else {
                die sprintf('Error while open file "%s": %s', $opts->{resend}, $!);
            }
        } else {
            my %pages;
            unless ($opts->{god_mode_only}) {
                my $last_id = 0;
                while (TRUE) {
                    last if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                    my $data = $app->partner_db->design_templates->get_all(
                        fields => [qw(id page_id block_id opts caption)],
                        filter => [
                            AND => [
                                ($opts->{page_ids} ? [page_id => 'IN' => \$opts->{page_ids}] : ()),
                                (
                                    $opts->{block_ids_design}
                                    ? fields_to_filter(
                                        [qw(page_id block_id)],
                                        $opts->{block_ids_design},
                                        for_db => TRUE,
                                      )
                                    : ()
                                ),
                                [type => '=' => \'tga'],
                                # Удалённые не восстанавливаются, игнорируем
                                [multistate => '=' => \0],
                                # Так быстрее, чем offset
                                [id => '>' => \$last_id],
                            ]
                        ],
                        limit    => 10000,
                        order_by => ['id'],
                    );
                    last unless scalar @$data;
                    $last_id = $data->[-1]{id};
                    my $changed_pages = update_designs($data);
                    @pages{@$changed_pages} = ();
                }
            }

            unless ($opts->{designes_only}) {
                foreach my $model (qw(context_on_site_rtb internal_context_on_site_rtb)) {
                    my $last_date = '';
                    while (TRUE) {
                        last if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                        my $data = $app->$model->partner_db_table->get_all(
                            fields => [qw(campaign_id id bk_data)],
                            filter => [
                                AND => [
                                    ($opts->{page_ids} ? [campaign_id => 'IN' => \$opts->{page_ids}] : ()),
                                    (
                                        $opts->{block_ids_godmode} ? fields_to_filter(
                                            [qw(campaign_id id)],
                                            $opts->{block_ids_godmode},
                                            for_db => TRUE,
                                          ) : ()
                                    ),
                                    [is_custom_bk_data => '=' => \1],
                                    # Так быстрее, чем offset
                                    # Возможен небольшой нахлёст,
                                    # но такие блоки проигнорируются при повторной обработке.
                                    # Добавление более сложных проверок для уникальности
                                    # увеличит время выполнения и читабельность.
                                    [create_date => '>=' => \$last_date],
                                ]
                            ],
                            limit    => 10000,
                            order_by => ['create_date'],
                        );
                        last unless scalar @$data;
                        $last_date = $data->[-1]{create_date};
                        my $changed_pages = update_god_mode($model, $data);
                        @pages{@$changed_pages} = ();
                    }
                }
            }

            $pages_to_resend = [
                sort {$a <=> $b}
                  keys %pages
            ];
        }
        print logstr 'PAGES TO RESEND', $pages_to_resend;
        $app->all_pages->mark_pages_for_async_update(page_ids => $pages_to_resend)
          if !$opts->{dry_run} && scalar @$pages_to_resend;
        print logstr 'STAT', $opts->{stat} if $opts->{stat};
    }
   );

sub update_designs {
    my ($data) = @_;

    my @result;
    foreach my $design (@$data) {
        next if defined $opts->{percent} && rand() > $opts->{percent};
        try {
            my $original = clone $design;
            $design->{opts} = from_json $design->{opts};
            my $settings = $design->{opts}{design_settings};
            my $name = $settings->{name} // '';
            return if ($opts->{designes} && !$opts->{designes}{$name});
            return
              if $opts->{limit_per_design}
                  && $opts->{stat}{$name}
                  && $opts->{stat}{$name} >= $opts->{limit_per_design};
            return if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
            my $changed;

            if ($settings->{limit} && $settings->{limit} == 1 && $settings->{carousel}) {
                delete $settings->{carousel};
                $changed = TRUE;
            }
            if ($design->{opts}{filter_tags}) {
                foreach (@{$design->{opts}{filter_tags}}) {
                    if ($_ eq 'fixed') {
                        $_       = 'adaptive';
                        $changed = TRUE;
                    }
                }
            }
            if ($changed) {
                $opts->{stat}{$name}++;
                $opts->{stat}{total}++;
                $design->{opts} = to_json $design->{opts};
                push @result, $design;
                print logstr 'DESIGN BEFORE', $original->{id}, $original;
                print logstr 'DESIGN AFTER',  $design->{id},   $design;
            }
        }
        catch {
            print logstr 'ERROR', $design->{id}, $@;
        };
    }
    $app->partner_db->design_templates->add_multi(\@result, duplicate_update => TRUE)
      if !$opts->{dry_run} && scalar @result;

    return [map {$_->{page_id}} @result];
}

sub update_god_mode {
    my ($model, $data) = @_;

    my @result;
    foreach my $block (@$data) {
        next if defined $opts->{percent} && rand() > $opts->{percent};
        try {
            my $original = clone $block;
            return unless $block->{bk_data};
            $block->{bk_data} = from_json $block->{bk_data};
            my $designes = $block->{bk_data}{RtbDesign};
            my $changed  = FALSE;
            my $old      = FALSE;
            unless (ref $designes) {
                # Где-то ещё остался старый формат (json без скобочек)
                # Надо для однотипости обработки привести его к текущему виду
                # а потом запаковать обратно
                $old      = TRUE;
                $designes = {
                    0 => {
                        design => from_json('{' . $designes . '}'),
                        type   => 'tga',
                    }
                };
            }
            foreach my $design (values %$designes) {
                my $type = $design->{type} // '';
                next if $type ne 'tga';
                my $settings = $design->{design};
                my $name = $settings->{name} // '';
                next if ($opts->{designes} && !$opts->{designes}{$name});
                if ($settings->{limit} && $settings->{limit} == 1 && $settings->{carousel}) {
                    next
                      if $opts->{limit_per_design}
                          && $opts->{stat}{$name}
                          && $opts->{stat}{$name} >= $opts->{limit_per_design};
                    $opts->{stat}{$name}++;
                    next if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                    $opts->{stat}{total}++;
                    delete $settings->{carousel};
                    $changed = TRUE;
                }
            }
            if ($changed) {
                $block->{bk_data}{RtbDesign} = substr to_json($designes->{0}{design}), 1, -1 if $old;
                $block->{bk_data} = to_json $block->{bk_data};
                push @result, $block;
                print logstr 'BLOCK BEFORE', [@$block{qw(campaign_id id)}], $original;
                print logstr 'BLOCK AFTER',  [@$block{qw(campaign_id id)}], $block;
            }
        }
        catch {
            print logstr 'ERROR', [@$block{qw(campaign_id id)}], $@;
        };
    }
    $app->$model->partner_db_table->add_multi(\@result, duplicate_update => TRUE)
      if !$opts->{dry_run} && scalar @result;

    return [map {$_->{campaign_id}} @result];
}
