#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  Скрипт для замены дизайнов

=head1 USAGE

  ./bin/oneshots/PI-28037_change_modernAdaptive_to_adaptive0418.pl --designes=adaptive,300x300

=head1 OPTIONS

  designes  - список дизайнов

=cut

use lib::abs qw(
  ../../lib
  ../../local/lib/perl5
  );

use Clone;
use Pod::Usage;

use qbit;
use Utils::DB;
use Utils::PublicID;
use Utils::ScriptWrapper;

my %OLD_DESIGN = map {$_ => TRUE} qw(
  modernAdaptive
  vertical
  horizontal
  motion
  );
my $NEW_DESIGN = 'adaptive0418';

my $app;
my $opts;
my $before;

sub args {
    my ($opts) = @_;
    return (
        'designes=s@'        => \$opts->{designes},
        'god_mode_only!'     => \$opts->{god_mode_only},
        'designes_only!'     => \$opts->{designes_only},
        'page_ids=s@'        => \$opts->{page_ids},
        'block_ids=i@'       => \$opts->{block_ids},
        'limit=i'            => \$opts->{limit},
        'limit_per_design=i' => \$opts->{limit_per_design},
        'percent=f'          => \$opts->{percent},
        'resend=s'           => \$opts->{resend},
        'rollback=s'         => \$opts->{rollback},
    );
}

sub prepare_args {
    my ($opts) = @_;

    $opts->{ticket} ||= 'PI-28037';

    if ($opts->{designes}) {
        my @designes = split /\s*,\s*/, join ',', @{$opts->{designes}};
        my %designes;
        @designes{@designes} = (TRUE) x scalar @designes;
        $opts->{designes} = \%designes;
    }

    if ($opts->{page_ids}) {
        $opts->{page_ids} = [split /\s*,\s*/, join ',', @{$opts->{page_ids}}];
    }

    if ($opts->{block_ids}) {
        $opts->{block_ids} = [split /\s*,\s*/, join ',', @{$opts->{block_ids}}];
        $opts->{block_ids_design} = [
            map {
                my (undef, $page_id, $block_id) = split_block_public_id($_);
                +{page_id => $page_id, block_id => $block_id,}
              } @{$opts->{block_ids}}
        ];
        $opts->{block_ids_godmode} = [
            map {
                my (undef, $page_id, $block_id) = split_block_public_id($_);
                +{campaign_id => $page_id, id => $block_id,}
              } @{$opts->{block_ids}}
        ];
    }

    foreach (qw(rollback)) {
        die sprintf('File "%s" does not esist', $opts->{$_}) if $opts->{$_} && !-f $opts->{$_};
    }
}

run(
    sub {
        ($app, $opts) = @_;

        my $pages_to_resend;

        if ($opts->{resend}) {
            my $f;
            if (open $f, '<', $opts->{resend}) {
                while (defined(my $s = <$f>)) {
                    if ($s =~ /PAGES TO RESEND\s+(.+)$/) {
                        $pages_to_resend = from_json $1;
                        last;
                    }
                }
                close $f;
            } else {
                die sprintf('Error while open file "%s": %s', $opts->{resend}, $!);
            }
        } else {
            if ($opts->{rollback}) {
                if (open my $f, '<', $opts->{rollback}) {
                    while (defined(my $s = <$f>)) {
                        if ($s =~ /(?:DESIGN|BLOCK) BEFORE\s+(?!ROLLBACK)(.+?)\s+(.+)$/) {
                            $before->{$1} = from_json $2;
                        }
                    }
                    close $f;
                } else {
                    die sprintf('Error while open file "%s": %s', $opts->{resend}, $!);
                }
            }

            my %pages;
            unless ($opts->{god_mode_only}) {
                my $last_id = 0;
                while (TRUE) {
                    last if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                    my $data = $app->partner_db->design_templates->get_all(
                        fields => [qw(id page_id block_id opts caption)],
                        filter => [
                            AND => [
                                ($opts->{page_ids} ? [page_id => 'IN' => \$opts->{page_ids}] : ()),
                                (
                                    $opts->{block_ids_design}
                                    ? fields_to_filter(
                                        [qw(page_id block_id)],
                                        $opts->{block_ids_design},
                                        for_db => TRUE,
                                      )
                                    : ()
                                ),
                                [type => '=' => \'tga'],
                                # Удалённые не восстанавливаются, игнорируем
                                [multistate => '=' => \0],
                                # Так быстрее, чем offset
                                [id => '>' => \$last_id],
                            ]
                        ],
                        limit    => 10000,
                        order_by => ['id'],
                    );
                    last unless scalar @$data;
                    $last_id = $data->[-1]{id};
                    my $changed_pages = update_designs($data);
                    @pages{@$changed_pages} = ();
                }
            }

            unless ($opts->{designes_only}) {

                my $last_date = '';
                while (TRUE) {
                    last if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                    my $data = $app->context_on_site_rtb->partner_db_table->get_all(
                        fields => [qw(campaign_id id bk_data)],
                        filter => [
                            AND => [
                                ($opts->{page_ids} ? [campaign_id => 'IN' => \$opts->{page_ids}] : ()),
                                (
                                    $opts->{block_ids_godmode}
                                    ? fields_to_filter(
                                        [qw(campaign_id id)],
                                        $opts->{block_ids_godmode},
                                        for_db => TRUE,
                                      )
                                    : ()
                                ),
                                [is_custom_bk_data => '=' => \1],
                                # Так быстрее, чем offset
                                # Возможен небольшой нахлёст,
                                # но такие блоки проигнорируются при повторной обработке.
                                # Добавление более сложных проверок для уникальности
                                # увеличит время выполнения и читабельность.
                                [create_date => '>=' => \$last_date],
                            ]
                        ],
                        limit    => 10000,
                        order_by => ['create_date'],
                    );
                    last unless scalar @$data;
                    $last_date = $data->[-1]{create_date};
                    my $changed_pages = update_god_mode('context_on_site_rtb', $data);
                    @pages{@$changed_pages} = ();
                }
            }

            $pages_to_resend = [
                sort {$a <=> $b}
                  keys %pages
            ];
        }

        print logstr 'PAGES TO RESEND', $pages_to_resend;
        $app->all_pages->mark_pages_for_async_update(page_ids => $pages_to_resend)
          if !$opts->{dry_run} && scalar @$pages_to_resend;
        print logstr 'STAT', $opts->{stat} if $opts->{stat};
    }
   );

sub update_designs {
    my ($data) = @_;
    my @result;

    foreach my $design (@$data) {
        next if defined $opts->{percent} && rand() > $opts->{percent};
        try {
            my $original = clone $design;
            $design->{opts} = from_json $design->{opts};
            my $settings = $design->{opts}{design_settings};
            my $name = $settings->{name} // '';
            if ($opts->{rollback}) {
                if ($before->{$original->{id}}) {
                    print logstr 'DESIGN BEFORE ROLLBACK', $original->{id}, $design;
                    $design = $before->{$original->{id}};
                    print logstr 'DESIGN AFTER ROLLBACK', $design->{id}, $design;
                    push @result, $design;
                    $opts->{stat}{$settings->{name}}++;
                    $opts->{stat}{total}++;
                }
            } elsif ($opts->{designes} && $opts->{designes}{$name}
                || !$opts->{designes} && $OLD_DESIGN{$name})
            {
                return
                  if $opts->{limit_per_design}
                      && $opts->{stat}{$name}
                      && $opts->{stat}{$name} >= $opts->{limit_per_design};
                $opts->{stat}{$name}++;
                return if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                $opts->{stat}{total}++;
                modify_settings($name, $settings);

                $design->{opts} = to_json $design->{opts};
                push @result, $design;
                print logstr 'DESIGN BEFORE', $original->{id}, $original;
                print logstr 'DESIGN AFTER',  $design->{id},   $design;
            }
        }
        catch {
            print logstr 'ERROR', $design->{id}, $@;
        };
    }
    $app->partner_db->design_templates->add_multi(\@result, duplicate_update => TRUE)
      if !$opts->{dry_run} && scalar @result;

    return [map {$_->{page_id}} @result];
}

sub update_god_mode {
    my ($model, $data) = @_;

    my @result;
    foreach my $block (@$data) {
        next if defined $opts->{percent} && rand() > $opts->{percent};
        my $block_key = sprintf("[%s]", join(',', @$block{qw(campaign_id id)}));
        try {
            my $original = clone $block;
            return unless $block->{bk_data};
            $block->{bk_data} = from_json $block->{bk_data};
            my $designes = $block->{bk_data}{RtbDesign};
            my $changed  = FALSE;
            my $old      = FALSE;
            unless (ref $designes) {
                # Где-то ещё остался старый формат (json без скобочек)
                # Надо для однотипости обработки привести его к текущему виду
                # а потом запаковать обратно
                $old      = TRUE;
                $designes = {
                    0 => {
                        design => from_json('{' . $designes . '}'),
                        type   => 'tga',
                    }
                };
            }
            if ($opts->{rollback}) {
                if ($before->{$block_key}) {
                    $block->{bk_data} = from_json $before->{$block_key}->{bk_data};
                    for my $design_key (keys %{$block->{bk_data}{RtbDesign}}) {
                        my $name = $block->{bk_data}{RtbDesign}{$design_key}{design}{name} // '';
                        next unless $OLD_DESIGN{$name};
                        $opts->{stat}{$name}++;
                        $opts->{stat}{total}++;
                    }
                    $changed = TRUE;
                }
            } else {
                foreach my $design (values %$designes) {
                    my $type = $design->{type} // '';
                    next if $type ne 'tga';
                    my $settings = $design->{design};
                    my $name = $settings->{name} // '';
                    if ($opts->{designes} && $opts->{designes}{$name}
                        || !$opts->{designes} && $OLD_DESIGN{$name})
                    {
                        next
                          if $opts->{limit_per_design}
                              && $opts->{stat}{$name}
                              && $opts->{stat}{$name} >= $opts->{limit_per_design};
                        $opts->{stat}{$name}++;
                        next if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                        $opts->{stat}{total}++;
                        modify_settings($name, $settings);
                        $changed = TRUE;
                    }
                }
            }
            if ($changed) {
                $block->{bk_data}{RtbDesign} = substr to_json($designes->{0}{design}), 1, -1 if $old;
                $block->{bk_data} = to_json $block->{bk_data};
                push @result, $block;
                print logstr sprintf('BLOCK BEFORE%s', $opts->{rollback} ? ' ROLLBACK' : ''), $block_key, $original;
                print logstr sprintf('BLOCK AFTER%s',  $opts->{rollback} ? ' ROLLBACK' : ''), $block_key, $block;
            }
        }
        catch {
            print logstr 'ERROR', [@$block{qw(campaign_id id)}], $@;
        };
    }
    $app->$model->partner_db_table->add_multi(\@result, duplicate_update => TRUE)
      if !$opts->{dry_run} && scalar @result;

    return [map {$_->{campaign_id}} @result];
}

sub modify_settings {
    my ($name, $settings) = @_;

    $settings->{name} = $NEW_DESIGN;

    if ($name eq 'modernAdaptive') {
        if ($settings->{adaptiveType} && $settings->{adaptiveType} =~ /^[ls]$/i) {
            $settings->{height} = lc($settings->{adaptiveType}) eq 's' ? 90 : 300;
        }
    } elsif ($name eq 'vertical') {
        $settings->{layout} = 'vertical';
    } elsif ($name eq 'horizontal') {
        $settings->{layout} = 'horizontal';
        if (exists($settings->{noSitelinks}) && !exists($settings->{height})) {
            $settings->{height} = $settings->{noSitelinks} ? 290 : 320;
        }
    } elsif ($name eq 'motion') {
        $settings->{limit} = 1;
    }
}
