#!/usr/bin/perl

=encoding UTF-8

=cut

=head1 DESCRIPTION

    нужно подключить прод базу

=head1 USAGE

    ./bin/oneshots/PI-28161_fix_ssp_token_duplicates.pl --ticket=PI-NNNN --over_logs --dry_run

=head1 OPTIONS

    dry_run - Ничего неудаляет, просто выводит то что хочет заменить

    Дебажные
        page-ids             - список пйджей, для дебага
        skip-fill-table      - не заполянет временную таблицу, дает ускорение при дебаге
        skip-set-duplicates  - не проставляет флаг дубликата, дает ускорение при дебаге

=cut

use lib::abs-soft => qw(
  ../../lib
  ../../local/lib/perl5/
  );

use qbit;
use Utils::ScriptWrapper 'oneshot';

my $token_field_size = 500;
my $tmp_table_name;

run(\&main);

sub args {
    my ($opts) = @_;

    return (
        'page-ids=s'           => \$opts->{'page-ids'},
        'skip-fill-table!'     => \$opts->{'skip-fill-table'},
        'skip-set-duplicates!' => \$opts->{'skip-set-duplicates'},

    );
}

sub main {
    my ($app, $opts) = @_;

    $opts->{limit} ||= 3 if $opts->{'dry_run'};

    $tmp_table_name = sprintf('%s_ssp_links', $opts->{ticket});

    _create_tmp_table($app) unless $opts->{'skip-fill-table'};

    _fill_tmp_table($app) unless $opts->{'skip-fill-table'};

    print logstr sprintf(q[Total table rows - %d], _get_table_rows_count($app, $tmp_table_name));

    _set_duplicates($app) unless $opts->{'skip-set-duplicates'};

    _allow_model_to_edit_field_tockens($app);

    _set_uniq_token_for_pages($app, $opts);

}

sub _allow_model_to_edit_field_tockens {
    my ($app) = @_;

    my $editable_fields = $app->ssp_link_mobile_app->get_editable_fields();

    no warnings 'redefine';
    *Application::Model::Product::SSP::Link::MobileApp::get_editable_fields = sub {
        return {%$editable_fields, tokens => 1};
    };

    return 1;
}

sub _set_uniq_token_for_pages {
    my ($app, $opts) = @_;

    my $pages = $app->partner_db->_get_all(
        my $sql =
          sprintf q[
            SELECT   page_id,
                     seller_id,
                     token
            FROM     `%s`
            WHERE    is_duplicate = 1
            ORDER BY token,
                     seller_id
            %s
        ], $tmp_table_name, ($opts->{limit} ? sprintf('LIMIT %d', $opts->{limit}) : '')
    );

    my $only_pages = {map {$_ => 1} split(/,/, $opts->{'page-ids'} // '')};

    my $total_count      = scalar @$pages;
    my $count            = 0;
    my $duplicates_count = 0;
    my $processed_pages  = {};
    foreach my $page (@$pages) {
        ++$count;
        my ($page_id, $seller_id, $token) = @$page{qw(page_id  seller_id  token)};

        next if $processed_pages->{$page_id};
        next if %$only_pages && !$only_pages->{$page_id};

        $app->partner_db->transaction(
            sub {

                # NOTE! добавляем  сортировку чтобы оставить созданные раньше работающие пейджи,
                #       а обновлять только созданые позже
                my $duplicates = $app->partner_db->_get_all(
                    sprintf q[
                SELECT   L.id,
                         L.mobile_app_id,
                         L.seller_id,
                         P.is_working,
                         L.tokens
                FROM     ssp_link_mobile_app L
                         INNER JOIN `%s` P ON (L.mobile_app_id = P.page_id and L.seller_id = P.seller_id)
                WHERE    L.seller_id = %d
                         and P.token = "%3$s"
                         and L.tokens LIKE '%%"%3$s"%%'
                ORDER BY P.is_working DESC,
                         L.mobile_app_id desc
                FOR UPDATE
            ], $tmp_table_name, $seller_id, $token
                );

                if (@$duplicates > 1) {
                    foreach my $i (0 .. $#$duplicates) {
                        my $row = $duplicates->[$i];
                        if ($i == 0) {
                            # первый всегда оставляем как есть, как правило он рабочий
                            print logstr(
                                sprintf
                                  '%d/%d. token "%s", seller_id=%d - Skip first duplicate for %sworking page_id=%d',
                                $count,
                                $total_count,
                                $token,
                                $seller_id,
                                ($row->{is_working} ? '' : 'not '),
                                $row->{mobile_app_id}
                            );
                        } else {
                            my $new_token = sprintf('%s_%s_%s', $token, $opts->{ticket}, $row->{mobile_app_id});

                            my $new_tokens = [map {$_ eq $token ? $new_token : $_} @{from_json($row->{tokens})}];

                            $app->ssp_link_mobile_app->do_action($row->{id}, 'edit', tokens => to_json($new_tokens))
                              unless $opts->{'dry_run'};

                            ++$duplicates_count;
                            $processed_pages->{$row->{mobile_app_id}} = 1;

                            my $link = $app->ssp_link_mobile_app->get($row, fields => [qw(id tokens)]);

                            print logstr(
                                sprintf(
'%d/%d. token "%s", seller_id=%d - Link id=%d for %sworking page_id=%d, got token changed to "%s"',
                                    $count, $total_count, $token, $seller_id, $row->{id},
                                    ($row->{is_working} ? '' : 'not '),
                                    $row->{mobile_app_id}, $new_token
                                )
                            );
                        }
                    }
                }
            }
        );
    }

    print logstr(sprintf('%d/%d pages fixed with duplicated token-seller pair', $duplicates_count, $total_count));
    print logstr('Pages to resend: ', join(',', sort {$a <=> $b} keys %{$processed_pages}));
}

sub _set_duplicates {
    my ($app) = @_;

    # 274 rows, ~ 20 sec
    my $affected_rows = $app->partner_db->_do(
        sprintf q[
            UPDATE
                `%1$s` A
                INNER JOIN (
                  select  token, seller_id, count(*) as cnt
                  from    `%1$s`
                  group   by token, seller_id
                  having cnt > 1
                ) B ON A.token = B.token and A.seller_id = B.seller_id
            SET A.is_duplicate = 1
        ], $tmp_table_name
    );
    print logstr(sprintf 'Duplicates rows updated - %d', $affected_rows);

    # максимум пйджей у дубликатов - 2
    my $duplicates_count = $app->partner_db->_get_all(
        sprintf q[
            SELECT  count(*) as cnt
            FROM    (
                SELECT  token, seller_id, count(*) as cnt
                from    `%s`
                GROUP BY token, seller_id
                having cnt > 1
            ) a;
        ], $tmp_table_name
    )->[0]->{'cnt'};
    print logstr(sprintf 'Duplicated token, seller_id found - %d', $duplicates_count);

    # максимум пйджей у дубликатов - 2
    my $max_pages_with_duplicate = $app->partner_db->_get_all(
        sprintf q[
            SELECT max(cnt) as max
            FROM (
                SELECT  count(distinct page_id) as cnt
                FROM    `%1$s`
                WHERE   is_duplicate = 1
                GROUP BY token, seller_id
            ) a;
        ], $tmp_table_name
    )->[0]->{'max'};
    print logstr(sprintf 'Max pages with diplicated token -  %d', $max_pages_with_duplicate // 0);

    return 1;
}

sub _fill_tmp_table {
    my ($app) = @_;

    foreach my $index (0 .. 5) {
        # заливаем пейджи и их токены (из массива токенов делаем строки)
        my $affected_rows = $app->partner_db->_do(
            sprintf q[
            INSERT INTO `%s` (page_id, is_working, seller_id, token)
            SELECT P.page_id,
                   if(P.multistate & 1, 1, 0) as is_working,
                   P.seller_id,
                   L.tokens->>'$[%2$d]' as token
            FROM  ssp_mobile_app_settings P
                  inner join ssp_link_mobile_app L on (P.page_id = L.mobile_app_id)
            WHERE L.tokens->>'$[%2$d]' is not null
            ON DUPLICATE KEY UPDATE is_working=VALUES(is_working)
        ], $tmp_table_name, $index
        );

        print logstr(sprintf 'Inserted "%d" rows', $affected_rows);

        # максимум токенов у пейджа 2, но на всякий пробуем наити больше, и выходим если больше уже нет
        last if $affected_rows == 0;
    }

    # проверяем что ничего не обрезалось при вставке, должно быть < 500
    my $max_token_length =
      $app->partner_db->_get_all(sprintf q[select max(length(token)) as size from `%s`], $tmp_table_name)->[0]
      ->{'size'};
    die sprintf('Max length "%d" more than field size "%d"', $max_token_length, $token_field_size)
      if $max_token_length > $token_field_size;

    print logstr(sprintf 'Max token length - %d', $max_token_length);

    return 1;
}

sub _create_tmp_table {
    my ($app) = @_;

    $app->partner_db->_do(
        sprintf q[
        CREATE TABLE IF NOT EXISTS `%s` (
            `page_id`      int(10) unsigned NOT NULL,
            `is_working`   tinyint default 0 NOT NULL,
            `seller_id`    int(10) unsigned NOT NULL,
            `token`        varchar(%d) CHARACTER SET utf8 COLLATE utf8_bin not null,
            `is_duplicate` tinyint default 0,
            UNIQUE INDEX `uniq_distribution_campaign__page_id` (`page_id`, `seller_id`, `token`)
        ) ENGINE='InnoDB' DEFAULT CHARACTER SET 'UTF8'
    ], $tmp_table_name, $token_field_size
    );

    # чистим если была не пустая
    $app->partner_db->_do(sprintf q[TRUNCATE `%s`], $tmp_table_name);

    print logstr(sprintf 'Table "%s" created', $tmp_table_name);

    return 1;
}

sub _get_table_rows_count {
    my ($app, $table_name) = @_;

    my $total_table_rows =
      $app->partner_db->_get_all(sprintf q[SELECT count(*) AS cnt FROM `%s`], $table_name)->[0]->{cnt};

    return $total_table_rows;
}
