#!/usr/bin/perl

=encoding UTF-8

=cut

=head1 DESCRIPTION

    Переводим блоки типа adaptive_banner (адаптивный стики баннер) в обычные баннеры
    для пользователей, которым подключается фича simple_inapp,
    т.к. adaptive_banner c фичёй simple_inapp не работает (см PI-28411)

=head1 USAGE

    ./bin/oneshots/PI-28430_convert_adaptive_banners_to_banners.pl --ticket=PI-NNNN --over_logs --dry_run --user-ids ids.txt
    ./bin/oneshots/PI-28430_convert_adaptive_banners_to_banners.pl --ticket=PI-NNNN --over_logs --dry_run --all_users --revert_log rev.log

=head1 OPTIONS

    dry_run - Ничего не меняет, просто выводит то что хочет заменить
    user_ids - Файл с id партнёров, которым обновляем блоки, каждый id в отдельной строке
    user_logins - Файл с логинами партнёров, которым обновляем блоки, каждый логин в отдельной строке
    all_users - обновить блоки всем партнёрам
    ignore_unexpected_blocks - не падать, если текущие настройки блока adaptive_banner отличаются от ожидаемых
    revert_log - файл с прошлой миграцией, которую нужно откатить

=cut

use lib::abs-soft => qw(
  ../../lib
  ../../local/lib/perl5/
  );

use qbit;
use Utils::ScriptWrapper 'oneshot';
use File::Slurp qw( read_file  write_file );
use PiConstants '$TECHNICAL_RTB_BLOCK_ID';

run(\&main);

sub args {
    my ($opts) = @_;

    return (
        'user_ids=s'                => \$opts->{'user_ids'},
        'user_logins=s'             => \$opts->{'user_logins'},
        'ignore_unexpected_blocks!' => \$opts->{ignore},
        'revert_log=s'              => \$opts->{revert_log},
        'all_users!'                => \$opts->{all_users},
    );
}

sub main {
    my ($app, $opts) = @_;
    my $users;
    die "usage of `all_users` with other user selectors is ambiguous"
      if $opts->{all_users} && ($opts->{user_ids} || $opts->{user_logins});
    $users = check_users($app, $opts->{user_ids}, $opts->{user_logins}) unless $opts->{all_users};

    my $dsps;
    $dsps = get_defalt_dsps($app) unless $opts->{revert_log};

    my $user_rtbs =
      !$opts->{revert_log}
      ? fetch_blocks($app, $opts->{ignore})
      : get_blocks_from_log($opts->{revert_log});
    #format: {user_id=>[{campaign_id=> ..., id=>}, ...], ...}

    $users = check_users($app, undef, undef, [keys %$user_rtbs]) if $opts->{all_users};

    my @pages_to_resend;

    for my $user (@$users) {
        unless (exists $user_rtbs->{$user->{id}}) {
            print logstr("INFO: user $user->{login} skipped: no blocks found");
            next;
        }
        push @pages_to_resend, process_user($app, $user, $user_rtbs, $dsps, $opts);
    }
    if (@pages_to_resend) {
        my $msg = join ",", @pages_to_resend;
        print logstr("blocks updated, you may want to run:"),
          logstr("bin/resend_to_bk.pl --logbroker_only 1 --split 10 --page_ids=$msg 2>&1 | tee resend_to_bk.log");
    }
}

sub check_users {
    my ($app, $ids_file, $logins_file, $ids_list) = @_;
    die "Specify users file via --user_ids or --user_logins or --all_users\n"
      unless $ids_file
          or $logins_file
          or $ids_list;
    my @result;
    if ($ids_file || $ids_list) {
        my @in_ids = $ids_file
          ? map {
            /^\s*(\d+)\s*$/ or die "user id should be numeric, got <$_>";
            $1
          } read_file($ids_file)
          : @$ids_list;
        my $out_users = $app->users->get_all(filter => [id => IN => \@in_ids], fields => [qw(login id)]);
        my %not_found;
        @not_found{@in_ids} = ();
        delete @not_found{map {$_->{id}} @$out_users};
        die "Some user ids not found in DB: " . join ",", keys %not_found if %not_found;
        push @result, @$out_users;
    }
    if ($logins_file) {
        my @in_logins = map {
            /^\s*(\S+)\s*$/ or die "user login should be non-empty, got <$_>";
            $1
        } split /\n/, read_file($logins_file);
        my $out_users = $app->users->get_all(filter => [login => IN => \@in_logins], fields => [qw(login id)]);
        my %not_found;
        @not_found{@in_logins} = ();
        delete @not_found{map {$_->{login}} @$out_users};
        die "Some user logins not found in DB: " . join ",", keys %not_found if %not_found;
        push @result, @$out_users;
    }
    return \@result;
}

sub get_defalt_dsps {
    my ($app) = @_;
    my $default = $app->mobile_app_rtb->get_default_dsps({block_type => 'banner', show_video => TRUE});
    my $result = [map {$_->{id}} @$default];
    return $result;
}

sub process_user {
    my ($app, $user, $rtbs, $default_dsps, $opts) = @_;
    my %pages;

    my $msg = !$opts->{revert_log} ? 'EDIT_BLOCK' : 'REVERT';
    $msg = 'DRY_RUN' if $opts->{dry_run};
    my @data =
      !$opts->{revert_log}
      ? (
        block_type => 'banner',
        show_video => TRUE
      )
      : (
        block_type => 'adaptive_banner',
        show_video => FALSE
      );

    try {
        $app->partner_db->transaction(
            sub {
                for my $block (@{$rtbs->{$user->{id}}}) {
                    print logstr(sprintf '%s: <%s> <%s> owner <%s>', $msg, $block->{campaign_id}, $block->{id},
                        $user->{id});
                    next if $opts->{dry_run};
                    $pages{$block->{campaign_id}} = 1;
                    $app->mobile_app_rtb->partner_db_table->edit($block, {@data});
                    unless ($opts->{revert_log}) {
                        $app->partner_db->block_dsps->add_multi(
                            [
                                map {{page_id => $block->{campaign_id}, block_id => $block->{id}, dsp_id => $_,}}
                                  @$default_dsps
                            ],
                            ignore => TRUE
                        );
                    } else {
                        my $filter = $app->partner_db->filter(
                            [
                                AND => [
                                    [page_id  => '='  => \$block->{campaign_id}],
                                    [block_id => '='  => \$block->{id}],
                                    [dsp_id   => '!=' => \1],
                                ]
                            ]
                        );
                        $app->partner_db->block_dsps->delete($filter);
                    }
                    my ($is_valid, $dtl) = run_qbit_validator_for_block($app, $block->{campaign_id}, $block->{id});
                    unless ($is_valid) {
                        print logstr($dtl);
                        throw Exception::Validation("validation problem for $user->{login}");
                    }
                }
            }
        );
        print logstr(sprintf "INFO: user <%s> processed", $user->{login} // $user->{id}) if %pages;
    }
    catch Exception::Validation with {
        %pages = ();
        print logstr"error processing user $user->{login}";
    };
    return keys %pages;
}

sub get_blocks_from_log {
    my ($file) = @_;
    my %result;
    for (read_file($file)) {
        if (/EDIT_BLOCK: <(\d+)> <(\d+)> owner <(\d+)>/) {
            push @{$result{$3} //= []}, {campaign_id => $1 + 0, id => $2 + 0};
        } elsif (/FAILED: <(\d+)> for owner <(\d+)>/) {
            delete $result{$2};
        }
    }
    die "no data found in file <$file>" unless %result;
    return \%result;
}

sub fetch_blocks {
    my ($app, $ignore) = @_;

    my $blocks = $app->partner_db->_get_all(<<"SQL");
        SELECT r.campaign_id,
               s.owner_id AS user,
               r.id,
               r.show_video,
               r.media_block,
               r.direct_block,
               GROUP_CONCAT(m.type) AS types,
               GROUP_CONCAT(d.dsp_id) AS dsps
        FROM context_on_site_rtb r
          JOIN mobile_app_settings s ON r.campaign_id = s.context_page_id
          JOIN block_dsps d
            ON r.campaign_id = d.page_id
           AND d.block_id = r.id
          JOIN media_sizes m
            ON r.campaign_id = m.page_id
           AND r.id = m.block_id
        WHERE r.model = 'mobile_app_rtb'
        AND   r.block_type = 'adaptive_banner'
        AND   r.multistate & 1 = 0
        GROUP BY s.owner_id,
                 r.campaign_id,
                 r.id,
                 r.media_block,
                 r.direct_block,
                 r.show_video
SQL

    my $error_flag = 0;
    my %result;
    for my $block (@$blocks) {
        my ($user, $show_video, $dsps, $media, $direct, $types) =
          map {delete $block->{$_}} qw/user show_video dsps media_block direct_block types/;
        if (   $show_video
            || $dsps   ne '1'
            || $media  ne 'adaptive0418'
            || $direct ne 'adaptive0418'
            || $types  ne 'adaptive0418')
        {
            print logstr(sprintf "ERROR: strange block <%s> <%s>", $block->{campaign_id}, $block->{id});
            $error_flag = 1;
        }
        push @{$result{$user} //= []}, $block;
    }
    die "exiting due to strange blocks" if ($error_flag && !$ignore);
    return \%result;
}

# based on copy&paste from ./bin/fix_page_id.pl
sub run_qbit_validator_for_block {
    my ($app, $page_id, $block_id) = @_;

    my $model        = $app->mobile_app_rtb();
    my $model_fields = $model->get_model_fields;
    my %pk           = map {$_ => TRUE} grep {$model_fields->{$_}{'pk'}} keys(%$model_fields);

    my @ignore_fields = $app->qbit_validator_checker->_get_field_names_to_ignore('mobile_app_rtb');
    my $fields = arrays_difference([keys(%$model_fields)], \@ignore_fields);

    next if $block_id == $TECHNICAL_RTB_BLOCK_ID;

    my $pk_value = {
        campaign_id => $page_id,
        id          => $block_id,
    };

    my $block_data = $model->get_all(
        fields => array_uniq(@$fields, keys(%pk)),
        filter => $pk_value,
    )->[0];

    my $qv = QBit::Validator->new(
        data => $block_data,
        app  => $model,
        $model->get_template(fields => $fields),
    );

    if ($qv->has_errors) {
        my @fields_with_errors = $qv->get_fields_with_error();

        my %error_fields;
        foreach my $f (@fields_with_errors) {
            #pk поля как правило проверяются при добавлении get_all_campaigns_for_adding
            #затем площадка может быть удалена или находиться в другом статусе.
            my $field_name = $f->{'path'}[0] // 'ERROR_IN_ROOT';
            next if $pk{$field_name};

            $error_fields{$field_name} = {
                msg   => join('; ', @{$f->{'msgs'}}),
                value => $block_data->{$field_name},
                path  => $f->{'path'},
            };
        }
        my $details = {};
        if (%error_fields) {
            my $data = {
                pk           => $pk_value,
                error_fields => \%error_fields,
            };
            push @{$details->{error_elements} //= []}, $data;
        } else {
            die 'this should never happen';
        }
        return FALSE, $details;
    } else {
        return TRUE;
    }

}
