#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  ./bin/oneshots/PI-29514_add_mincpm_to_godmod.pl --ticket=PI-29514 --over_logs --use_cache --page-block-ids=28471-21,28471-91 --dty_run

=cut

use lib::abs qw(../../lib);

use File::stat;
use LWP::UserAgent;

use qbit;
use Pod::Usage;
use Utils::DB qw(fields_to_filter);
use Utils::ScriptWrapper;
use Utils::Logger qw(WARN WARNF);

our $BLOCKS_CHUNK_SIZE    = 20;
our $CACHE_FILE_PATH      = sprintf '%s.cache.json', $0;
our $CACHE_TTL            = 600;
our $MINCMP_RELEASED_DATE = '2022-06-30 16:10:00';

run(\&main);

sub modify_bk_data {
    my ($app, $bk_data, $opts) = @_;

    # берем первый потому что он одинаковый для всех для "минимальный cpm"
    # (а для стратегий "максимальный доход" и "раздельный cpm" там всегда нули)
    if ($bk_data->{DSPInfo}) {
        $bk_data->{MinCPM} = 0;
        if (@{$bk_data->{DSPInfo}}) {
            $bk_data->{MinCPM} = $bk_data->{DSPInfo}->[0]->{CPM};
        }
    }

    return $bk_data;
}

sub args {
    my ($opts) = @_;
    return (
        'page-block-ids:s'   => \$opts->{'page-block-ids'},
        'start_from_model=s' => \$opts->{'start_from_model'},
        'start_from_page=s'  => \$opts->{'start_from_page'},
        'limit:i'            => \$opts->{limit},
        'model_limit=i'      => \$opts->{model_limit},
        'use_cache!'         => \$opts->{use_cache},
        'force_cache!'       => \$opts->{force_cache},
        'reset_cache!'       => \$opts->{reset_cache},
    );
}

sub prepare_args {
    my ($opts) = @_;
    $opts->{page_blocks} = [map {[split /-/, $_]} split(/,/, $opts->{'page-block-ids'} // '')];
    $opts->{start_from_model} //= '';
    $opts->{start_from_page}  //= 0;
}

sub main {
    my ($app, $opts) = @_;

    my $start = time;
    silent();

    my $blocks = get_cached_data($CACHE_FILE_PATH, $opts, $CACHE_TTL);
    if (!$blocks || $opts->{reset_cache}) {
        $blocks = _get_block_by_models($app, $opts);
        update_cache_data($CACHE_FILE_PATH, $blocks) if $blocks;
    }

    my $model_count    = 0;
    my $is_start_model = 0;
    my $is_start_page  = 0;
    foreach my $accessor (sort keys %$blocks) {
        my $model = $app->$accessor;
        $model_count += 1;

        my $page_id_field_name = $model->get_page_id_field_name;

        my $model_blocks = $blocks->{$accessor};

        $is_start_model = 1 if !$opts->{start_from_model} || $opts->{start_from_model} eq $accessor;
        unless ($is_start_model) {
            print logstr(sprintf('%d. MODEL=%s', $model_count, $accessor), 'SKIP');
            next;
        } else {
            print logstr(sprintf '%d. MODEL=%s [%d]', $model_count, $accessor, scalar(@$model_blocks));
        }

        my $blocks_count = 0;
        while (my @blocks_chunk = splice(@$model_blocks, 0, $BLOCKS_CHUNK_SIZE)) {

            my $filter = $app->partner_db->filter(
                fields_to_filter(
                    [$page_id_field_name, 'id'],
                    [
                        map {
                            {$page_id_field_name => $_->{page_id}, id => $_->{block_id}}
                          } @blocks_chunk
                    ],
                    for_db => TRUE
                )
            );
            $filter->and({is_custom_bk_data => 1});

            if ($model->is_block_table_with_multiple_models()) {
                $filter->and(['model' => '=' => \$accessor]);
            }

            $app->partner_db->transaction(
                sub {

                    my $blocks_data = $model->partner_db_table->get_all(
                        fields => {
                            page_id  => $page_id_field_name,
                            block_id => 'id',
                            bk_data  => 'bk_data',
                        },
                        filter     => $filter,
                        order_by   => [qw(page_id block_id)],
                        for_update => 1
                    );

                    for (my $i = 0; $i <= $#$blocks_data; $i++) {
                        my $row = $blocks_data->[$i];

                        my $prefix = sprintf '%d.%d. model=%s, %spage_id=%s, block_id=%s: ',
                          $model_count,
                          ++$blocks_count,
                          $accessor,
                          ($i == 0 ? '** ' : ''),
                          @$row{qw(page_id block_id)};

                        $is_start_page = 1
                          if $is_start_model
                              && (!$opts->{start_from_page} || $row->{page_id} >= $opts->{start_from_page});
                        unless ($is_start_page) {
                            print logstr($prefix, 'SKIP');
                            next;
                        } else {
                            print logstr($prefix);
                        }

                        my $bk_data;
                        if ($row->{bk_data}) {
                            $bk_data = eval {from_json($row->{bk_data})};
                            if ($@) {
                                print logstr("\t", 'ERROR', $@->message());
                                print STDERR logstr($prefix, 'ERROR', $@->message());
                                next;
                            }
                        } else {
                            print logstr("\t", 'ERROR', 'bk_data is NULL');
                            print STDERR logstr($prefix, 'ERROR',, 'bk_data is NULL');
                            next;
                        }

                        my $bk_data_old_json = to_json($bk_data, canonical => 1);

                        my $new_bk_data = modify_bk_data($app, $bk_data, $opts);

                        my $bk_data_new_json = to_json($new_bk_data, canonical => 1);

                        if ($bk_data_old_json ne $bk_data_new_json) {
                            my $exception;
                            try {
                                my $pk = {
                                    $page_id_field_name => $row->{page_id},
                                    id                  => $row->{block_id},
                                };

                                if ($app->$accessor->can('edit')) {
                                    $app->$accessor->do_action($pk, 'edit',
                                        'bk_data' => to_json($new_bk_data, pretty => TRUE))
                                      unless $opts->{'dry_run'};
                                } else {
                                    my $filter = $pk;
                                    if ($model->is_block_table_with_multiple_models()) {
                                        $filter->{model} = $accessor;
                                    }

                                    $model->partner_db_table->edit($app->partner_db->filter($filter),
                                        {bk_data => to_json($new_bk_data, pretty => TRUE)})
                                      unless $opts->{'dry_run'};
                                }
                                print logstr(sprintf "\tOLD bk_data: %s", $bk_data_old_json);
                                print logstr(sprintf "\tNEW bk_data: %s", $bk_data_new_json);
                            }
                            catch {
                                ($exception) = @_;
                                print logstr("\t", 'ERROR', ref($exception));
                                print STDERR logstr($prefix, 'ERROR', ref($exception), $exception->message());
                            };

                        } else {
                            print logstr("\tNO CHANGES");
                        }
                    }
                }
            );
        }
    }

    my $now = time;

    print logstr(sprintf "elapsed %s sec", int(($now - $start) / 60));

    return 0;
}

sub _get_block_by_models {
    my ($app, $opts) = @_;

    my $found_blocks = {};
    if (@{$opts->{page_blocks}}) {

        my $filter = $app->partner_db->filter(
            fields_to_filter(
                ['page_id', 'id'],
                [map {{page_id => $_->[0], id => $_->[1]}} @{$opts->{page_blocks}}],
                for_db => TRUE
            )
        );

        my $blocks = $app->partner_db->all_blocks->get_all(
            fields => {
                model    => 'model',
                page_id  => 'page_id',
                block_id => 'id',
            },
            filter   => $filter,
            order_by => [qw(model page_id block_id)]
        );

        foreach my $row (@$blocks) {
            my $model_blocks = $found_blocks->{$row->{model}} //= [];
            push @$model_blocks, $row;
        }
    } else {

        foreach my $accessor (sort @{$app->product_manager->get_block_model_accessors}) {
            my $model = $app->$accessor;
            if ($model->DOES('Application::Model::Role::Has::CustomBkData')) {

                my $page_id_field_name = $model->get_page_id_field_name;

                my $filter = $app->partner_db->filter({'is_custom_bk_data' => 1});
                if ($model->is_block_table_with_multiple_models()) {
                    $filter->and(['model' => '=' => \$accessor]);
                }

                my $table = $model->partner_db_table();
                my $query = $app->partner_db->query->select(
                    table  => $table,
                    fields => {
                        page_id  => $page_id_field_name,
                        block_id => 'id',
                    },
                    filter   => $filter,
                    order_by => [$page_id_field_name, 'id']
                  )->join(
                    table   => $app->partner_db->all_pages,
                    fields  => ['login', 'update_time'],
                    join_on => [AND => [["page_id" => '=' => {$page_id_field_name => $table}],]],
                    filter => [AND => [["is_protected" => "=" => \0], ['update_time' => '<' => \$MINCMP_RELEASED_DATE]]]
                  );
                $query = $query->limit($opts->{model_limit}) if $opts->{model_limit};

                $found_blocks->{$accessor} = $query->get_all();
            }
        }
    }

    return $found_blocks;
}

sub silent {

    no warnings 'redefine';
    no strict 'refs';

    require QBit::Application::Model::API::HTTP;
    require Application::Model::API::Yandex::HTTPMOL;

    *{'QBit::Application::Model::API::HTTP::INFO'}      = sub { };
    *{'Application::Model::API::Yandex::HTTPMOL::INFO'} = sub { };
}

sub get_cached_data {
    my ($cache_file_path, $args, $ttl_sec) = @_;

    my $is_use_cache = $args->{use_cache} || $args->{force_cache};

    my $data;
    if ($is_use_cache && -e $cache_file_path) {
        my $cache_timestamp = File::stat::stat($cache_file_path)->mtime;
        my $cur_timestamp   = time();
        my $sec_left        = $cur_timestamp - $cache_timestamp;

        if ($args->{force_cache} || $sec_left < $ttl_sec) {
            $data = from_json(readfile($cache_file_path));
        }
    }

    $data = undef unless ($data && %$data);
    WARN 'Got data from cache file ' . $cache_file_path if $data;

    return $data;
}

sub update_cache_data {
    my ($cache_file_path, $data) = @_;
    writefile($cache_file_path, to_json($data, pretty => TRUE));
    WARNF 'cache file updated %s', $cache_file_path;
}
