#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт добавляет технический блок 100500. (Необходимо подключить ПРОД БАЗУ)

=head1 USAGE

perl bin/PI-8684_new_block.pl --page_ids=12345,54321,..  2>&1 | tee PI-8684_new_block.pl.log

perl bin/PI-8684_new_block.pl --page_ids=all_pages --dry_run

=head1 OPTIONS

  - dry_run - ничего не добавляет и не удаляет, просто выводит что собирается сделать
  - page_ids - Список ID площадок через запятую (или all_pages для всех пейджей)
  - delete - флаг, если передали удаляем технический блок

=cut

use lib::abs '../../lib';

use qbit;

use Application;
use PiConstants qw($TECHNICAL_RTB_BLOCK_ID);
use Utils::Logger qw/ INFO  INFOF  ERROR/;

use Cwd qw(getcwd);
use File::Basename qw( basename );
use File::Temp qw(tempfile);
use FindBin qw($Bin);
use File::Spec;
use File::Slurp qw(write_file);
use Pod::Usage;
use Getopt::Long qw();
use List::Util qw(min);
use Term::ANSIColor qw(colored);

main();

sub main {
    my ($page_ids, $delete, $is_dry_run) = _get_args();

    INFO 'START';

    my $app = Application->new();

    $app->pre_run();

    $app->set_cur_user({id => 0});    #Cron

    no strict 'refs';
    no warnings 'redefine';
    local *{'QBit::Application::check_rights'} = sub {TRUE};

    my %rtb_model_by_campaign = (
        context_on_site_campaign          => 'context_on_site_rtb',
        internal_context_on_site_campaign => 'internal_context_on_site_rtb',
    );

    my %pages_with_tech_blocks_by_type = ();
    my %pages_with_tech_blocks         = ();
    {
        foreach my $block_model (values %rtb_model_by_campaign) {

            my $page_id_field_name = $app->$block_model->get_page_id_field_name();

            INFOF(q[Get existing technical blocks for model '%s'], $block_model);
            my $tech_blocks = $app->partner_db->$block_model->get_all(
                fields => [qw(campaign_id)],
                filter => {
                    id => $TECHNICAL_RTB_BLOCK_ID,
                    (
                        @$page_ids == 1 && $page_ids->[0] eq 'all_pages'
                        ? ()
                        : ($page_id_field_name => $page_ids)
                    )
                },
            );

            push @{$pages_with_tech_blocks_by_type{$block_model}}, map {$_->{'campaign_id'}} @$tech_blocks;
        }

        %pages_with_tech_blocks = map {$_ => 1} map {@$_} values %pages_with_tech_blocks_by_type;

        INFOF(
            '%d pages already has technical blocks: %s',
            scalar(keys %pages_with_tech_blocks),
            substr(join(',', sort {$a <=> $b} keys %pages_with_tech_blocks), 0, 100)
        );
    }

    my @pages_affected = ();
    if ($delete) {

        INFOF('Starting process %d pages:', scalar(keys %pages_with_tech_blocks));

        foreach my $block_model (keys %pages_with_tech_blocks_by_type) {

            my $page_ids = $pages_with_tech_blocks_by_type{$block_model};

            next unless @$page_ids;

            INFOF('   %s (%d pages)', $block_model, scalar(@$page_ids));

            my $dsps = $block_model;
            $dsps =~ s/rtb$/dsps/;
            INFOF('        delete from %s (block_id=%s, campaign_id=%s...)',
                $dsps, $TECHNICAL_RTB_BLOCK_ID, join(',', @$page_ids[0 .. min(10, $#$page_ids)]));
            $app->partner_db->$dsps->delete(
                $app->partner_db->filter({campaign_id => $page_ids, block_id => $TECHNICAL_RTB_BLOCK_ID}))
              unless $is_dry_run;

            INFOF('        delete from %s (block_id=%s, page_id=%s...)',
                media_sizes, $TECHNICAL_RTB_BLOCK_ID, join(',', @$page_ids[0 .. min(10, $#$page_ids)]));
            $app->partner_db->media_sizes->delete(
                $app->partner_db->filter({campaign_id => $page_ids, block_id => $TECHNICAL_RTB_BLOCK_ID}))
              unless $is_dry_run;

            my $logs = $block_model . '_action_log';
            INFOF('        delete from %s (elem_id=%s, elem_campaign_id=%s...)',
                $logs, $TECHNICAL_RTB_BLOCK_ID, join(',', @$page_ids[0 .. min(10, $#$page_ids)]));
            $app->partner_db->$logs->delete(
                $app->partner_db->filter({elem_campaign_id => $page_ids, elem_id => $TECHNICAL_RTB_BLOCK_ID}))
              unless $is_dry_run;

            INFOF(
                '        delete from %s (elem_campaign_id=%s, id=%s)', $block_model,
                join(',', @$page_ids[0 .. min(10, $#$page_ids)]), $TECHNICAL_RTB_BLOCK_ID
            );
            $app->partner_db->$block_model->delete(
                $app->partner_db->filter({campaign_id => $page_ids, id => $TECHNICAL_RTB_BLOCK_ID}))
              unless $is_dry_run;

            push @pages_affected, @$page_ids;

        }
    } else {

        INFO('Start processing');

        foreach my $model (keys %rtb_model_by_campaign) {

            my $page_id_field_name = $app->$model->get_page_id_field_name();

            my $pages = $app->$model->get_all(
                fields => [$page_id_field_name],
                filter => [
                    'AND',
                    [
                        {multistate => 'working and not deleted'},
                        (@$page_ids == 1 && $page_ids->[0] eq 'all_pages')
                        ? [$page_id_field_name => 'IS NOT' => undef]
                        : {$page_id_field_name => $page_ids}
                    ]
                ],
                order_by => [$page_id_field_name],
            );

            $pages = [grep {!$pages_with_tech_blocks{$_->{$page_id_field_name}}} @$pages];

            INFOF('    %s (%d pages)', $model, scalar(@$pages)) if @$pages;

            foreach my $camp (@$pages) {

                my $page_id = $camp->{$page_id_field_name};

                my $block_model = $rtb_model_by_campaign{$model} // throw "Неизвестная модель $model";

                INFOF('        Start process page_id=%d', $page_id);

                my $rtb_settings = $app->$block_model->get_stat_rtb_block($page_id);

                $rtb_settings->{'caption'}       = 'Технический RTB Блок';
                $rtb_settings->{'id'}            = $TECHNICAL_RTB_BLOCK_ID;
                $rtb_settings->{'direct_block'}  = '240x400';
                $rtb_settings->{'media_block'}   = '240x400';
                $rtb_settings->{'strategy'}      = 3;
                $rtb_settings->{'media_active'}  = 1;
                $rtb_settings->{'media_blocked'} = 1;
                $rtb_settings->{'text_active'}   = 0;

                INFOF(q[            add technical block (page_id='%d')], $page_id);
                $app->partner_db->$block_model->add($rtb_settings) unless $is_dry_run;

                INFOF(q[            add %s (page_id='%d')], 'media_sizes', $page_id);
                $app->partner_db->media_sizes->add(
                    {
                        page_id  => $page_id,
                        block_id => $TECHNICAL_RTB_BLOCK_ID,
                        type     => '240x400'
                    }
                ) unless $is_dry_run;

                my $dsps = $block_model;
                $dsps =~ s/rtb$/dsps/;
                INFOF(q[            add %s (page_id='%d')], $dsps, $page_id);
                $app->partner_db->$dsps->add(
                    {
                        campaign_id => $page_id,
                        block_id    => $TECHNICAL_RTB_BLOCK_ID,
                        dsp_id      => 1,
                        is_deleted  => 0
                    }
                ) unless $is_dry_run;

                push @pages_affected, $page_id;
            }
        }
    }

    if (@pages_affected) {

        my $resend_page_ids_file_path;
        {
            my $name = basename($0, qw(.pl .pm .cgi));
            (undef, $resend_page_ids_file_path) = tempfile(
                $name . '_resend_page_ids_XXXX',
                DIR    => $Bin,
                OPEN   => 0,
                SUFFIX => '.txt'
            );
        }

        write_file($resend_page_ids_file_path, {binmode => ':utf8',}, join("\n", sort {$a <=> $b} @pages_affected));

        print colored(
            ["cyan"],
            sprintf('Необходимо переотправить %d пейджи в БК', scalar(@pages_affected))
          ),
          "\n";
        print colored(
            ["yellow"],
            sprintf('resend_to_bk.pl --file_path=./%s  2>&1 | tee resend_to_bk.log',
                File::Spec->abs2rel($resend_page_ids_file_path, getcwd()))
          ),
          "\n\n";
    }

    INFO '#END';
}

sub _get_args {

    my $page_id_str = '';
    my $is_dry_run  = '';
    my $help        = 0;
    my $delete      = 0;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'page_ids=s' => \$page_id_str,
        'dry_run!'   => \$is_dry_run,
        #---
        'delete'   => \$delete,
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    my @page_ids = ();

    #-- Проверяем зн-ия входных параметров
    my $errors = [];
    if ($page_id_str && $page_id_str eq 'all_pages') {
        push(@page_ids, $page_id_str);
    } elsif ($page_id_str && $page_id_str !~ /^(\d+,?)+$/) {
        push @$errors,
          'В списке пейджей должны быть тоолько цифры через запятую ';
    } else {
        @page_ids = split /,/, $page_id_str;
    }

    push @$errors, 'Ожидалось "all_pages" или список пейджей через запятую '
      unless @page_ids;

    if (@$errors) {
        foreach my $error (@$errors) {
            ERROR $error;
        }
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    return (\@page_ids, $delete, $is_dry_run);
}
