#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт переносит статистику из mysql в clickhouse.
    Необходимо подключить ПРОД базы MYSQL и CLICKHOUSE

=head1 USAGE

perl bin/oneshots/PI-9842_clickhouse_collector.pl --accessors=statistics_advnet_context_on_site_rtb

perl bin/oneshots/PI-9842_clickhouse_collector.pl --accessors=statistics_advnet_context_on_site_rtb --start_date=2017-12-15

perl bin/oneshots/PI-9842_clickhouse_collector.pl --accessors=statistics_advnet_context_on_site_rtb --fix_date=2017-12-15

=head1 OPTIONS

  - accessors  - аксессоры через запятую для которых перекладываем статистику
  - start_date - дата после которой перекладывается статистика
  - fix_date   - дата которую нужно переложить повторно (не работает вместе с start_date)
  - continue   - актуально с параметром fix_date, после повторного перекладывания одного дня скрипт продолжает работу в штатном режиме
  - clean      - перед заливкой данных удалить строки с нужным product_id

=cut

use feature 'say';
use lib::abs qw(
  ../../lib
  );

use Pod::Usage;
use Getopt::Long qw();

use qbit;
use Application;

{
    no warnings 'redefine';
    *QBit::Application::check_rights = sub {1};
}

my $LIMIT = 10_000;

main();

sub main {
    my $opts = _get_args();

    my ($accessors, $start_date, $fix_date, $continue, $clean) =
      @{$opts}{qw(accessors start_date fix_date continue clean)};

    my @accessors = split(',', $accessors);

    throw 'start_date, fix_date, continue only for one accessor'
      if @accessors > 1 && (defined($start_date) || defined($fix_date) || $continue);

    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0});

    foreach my $accessor (@accessors) {
        move_stat_for_level($app, $accessor, $start_date, $fix_date, $continue, $clean);
    }

    say '#END';
}

sub move_stat_for_level {
    my ($app, $accessor, $start_date, $fix_date, $continue, $clean) = @_;

    my $stat_level = $app->$accessor;
    my $table_obj  = $stat_level->_get_stat_table();

    say 'Table: ' . $table_obj->name();

    if (defined($fix_date)) {
        update_statistics($stat_level, $fix_date);

        return unless $continue;

        $start_date = $fix_date;
    }

    unless (defined($start_date)) {
        $start_date = curdate(oformat => 'db');
        $start_date =~ s/\d{2}$/01/;
    }

    my $end_date = $app->partner_db->query->select(
        table  => $table_obj,
        fields => {min_date => {MIN => ['dt']}}
    )->get_all()->[0]{'min_date'};

    if ($clean) {
        say $stat_level->{'accessor'} . ' - ' . curdate(oformat => 'db_time') . " - delete stat";

        $stat_level->refresh_tmp_stat_table();

        $stat_level->move_statistics('statistics', 'statistics_temporary', from => $end_date, to => $start_date);
        $stat_level->move_statistics('statistics_temporary', 'statistics', from => $end_date, to => $start_date);

        $stat_level->clickhouse_db->statistics_temporary->drop();
        $stat_level->clickhouse_db->_do('OPTIMIZE TABLE `statistics`');
    }

    while ($start_date gt $end_date) {
        $start_date = date_sub($start_date, day => 1, iformat => 'db', oformat => 'db');

        move_one_day($stat_level, 'statistics', $start_date);
    }
}

sub move_one_day {
    my ($stat_level, $table_name, $date) = @_;

    say $stat_level->{'accessor'} . ' - ' . curdate(oformat => 'db_time') . " - date: $date";

    my $table_obj = $stat_level->_get_stat_table();

    my $data;
    my $offset = 0;
    while (($data = $table_obj->get_all(filter => {dt => $date}, offset => $offset, limit => $LIMIT))
        && @$data)
    {
        $stat_level->app->clickhouse_db->$table_name->add_multi($stat_level->convert_mysql_to_clickhouse($data));

        last if @$data < $LIMIT;

        $offset += $LIMIT;
    }
}

sub update_statistics {
    my ($stat_level, $date) = @_;

    $stat_level->refresh_tmp_stat_table();

    $stat_level->move_statistics('statistics', 'statistics_temporary', from => $date, to => $date);

    move_one_day($stat_level, 'statistics_temporary', $date);

    $stat_level->move_statistics('statistics_temporary', 'statistics', from => $date, to => $date);
    $stat_level->clickhouse_db->statistics_temporary->drop();
    $stat_level->clickhouse_db->_do('OPTIMIZE TABLE `statistics`');
}

sub _get_args {

    my $opts = {
        accessors  => '',
        start_date => undef,
        fix_date   => undef,
        continue   => 0,
        clean      => 0,
        help       => 0,
    };

    Getopt::Long::GetOptions(
        'accessors=s'  => \$opts->{'accessors'},
        'start_date=s' => \$opts->{'start_date'},
        'fix_date=s'   => \$opts->{'fix_date'},
        'continue'     => \$opts->{'continue'},
        'clean'        => \$opts->{'clean'},
        'help|?|h'     => \$opts->{'help'},
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $opts->{'help'};

    unless ($opts->{'accessors'}) {
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    return $opts;
}
