#!/usr/bin/perl

=head1 DESCRIPTION
 Скрипт считает количетсво разных настроек для каждого блока в пределах логина

=head2 USAGE
 Передать выборочные логины --logins='["drive2russia", "asdasd"]'
 Задать ограничение по количеству выданных пейджов для логина --limit_for_pages=10
 Задать ограничение по количеству выданных блоков для пейджа --limit_for_blocks=10
 Задать ограничение по количеству рассматриваемых логинов --limit_for_logins=10
=cut

# project modules
use lib::abs qw(
  ../../lib
  );

use qbit;
use Application;

use POSIX;

use PiConstants qw($TECHNICAL_RTB_BLOCK_ID);
use Digest::MD5 qw(md5_hex);

use feature 'say';

my $ALLOWED_MODELS = [qw(context_on_site_campaign internal_context_on_site_campaign)];

my $CPMS_HASH = {
    mincpm          => TRUE,
    percent_traffic => TRUE,
    text_cpm        => TRUE,
    media_cpm       => TRUE,
};

my $id_excluded_1 = 0;

# main
sub main {
    my $args = _get_args();

    my $app = delete $args->{'app'};

    my $tmp_rights = $app->add_all_tmp_rights();

    my $block_models = [];

    foreach my $page_model (@{$ALLOWED_MODELS}) {
        push @{$block_models}, @{$app->$page_model->get_block_model_names()};
    }

    my $preprocess_block_models       = {};
    my $public_id_prefixses           = {};
    my $page_id_field_name_for_blocks = {};
    my $multistates                   = {};
    my @filters_for_blocks            = ('OR');

    foreach my $block_model (@{$block_models}) {

        $preprocess_block_models->{$block_model} = [];

        $public_id_prefixses->{$block_model} = $app->$block_model->public_id_prefix();

        $page_id_field_name_for_blocks->{$block_model} = $app->$block_model->get_page_id_field_name();

        my $model_fields = $app->$block_model->get_model_fields();

        $preprocess_block_models->{$block_model} = [
            grep {exists $model_fields->{$_}}
              qw(
              brands
              geo
              articles
              strategy
              mincpm
              percent_traffic
              text_cpm
              media_cpm
              )
        ];

        push @{$filters_for_blocks[1]},
          [
            'AND',
            [
                ['multistate', 'IN', \$app->$block_model->get_multistates_by_filter('working')],
                ['model',      '=',  \$block_model]
            ],
          ];

        $multistates->{$block_model} = $app->$block_model->get_multistates_by_filter('working');
    }

    my $tsv =
      Text::CSV_XS->new({binary => 1, sep_char => "\t", eol => "\n", escape_char => undef, quote_char => undef});

    write_in_the_start($tsv);

    #Забираем все логины из users
    my $logins = _get_logins($app, $args);

    my $logins_count = scalar @{$logins};

    my $i = 0;

    foreach my $login (@{$logins}) {

        $i = $i + 1;

        say "Comlete : " . $i . '/' . $logins_count;

        #Для каждого логина достаем все его площадки
        my $page_ids_for_login = _get_page_ids_for_login($app, $login->{'id'}, $args);

        next unless scalar @{$page_ids_for_login};

        my $page_ids = [];

        $page_ids = [map {$_->{'page_id'}} @{$page_ids_for_login}];

        my $blocks_count = {};

        map {$blocks_count->{$_} = {}} @{$block_models};

        my $block_ids_for_page = $app->partner_db->query->select(
            table  => $app->partner_db->all_blocks,
            fields => [qw(id model)],
            filter => [
                'AND',
                [
                    ['page_id' => 'IN' => \$page_ids],
                    ['id'      => '<>' => \$TECHNICAL_RTB_BLOCK_ID],
                    ['id'      => '<>' => \$id_excluded_1],
                    \@filters_for_blocks
                ]
            ],
        );

        $block_ids_for_page =
          defined $args->{'limit_for_blocks'}
          ? $block_ids_for_page->limit($args->{'limit_for_blocks'})->get_all()
          : $block_ids_for_page->get_all();

        next unless scalar @{$block_ids_for_page};

        my $block_ids_by_model = {};

        foreach (@{$block_ids_for_page}) {
            push @{$block_ids_by_model->{$_->{'model'}}}, $_->{'id'};
        }

        my $block_ids = [];

        foreach my $block_model (keys %{$block_ids_by_model}) {

            my $page_id_field_name_for_block = $page_id_field_name_for_blocks->{$block_model};

            my $block_ids_for_model = $app->$block_model->get_all(
                fields => [$page_id_field_name_for_block, 'id', @{$preprocess_block_models->{$block_model}}],
                filter => [
                    'AND',
                    [
                        ['id'                          => 'IN' => $block_ids_by_model->{$block_model}],
                        [$page_id_field_name_for_block => 'IN' => $page_ids],
                        ['multistate'                  => '='  => $multistates->{$block_model}]
                    ]
                ]
            );

            map {$_->{'model'} = $block_model} @{$block_ids_for_model};

            push @{$block_ids}, @{$block_ids_for_model};

            $blocks_count->{$block_model}->{'blocks_count'}->{'all'} += scalar @{$block_ids_for_model};
        }

        foreach my $block (@{$block_ids}) {

            my $block_model = delete $block->{'model'};

            my $public_id_perfix = $public_id_prefixses->{$block_model};

            my $page_id_field_name_for_block = $page_id_field_name_for_blocks->{$block_model};

            my $settings = _normalize_settings($block);

            my $public_id = $public_id_perfix . $block->{$page_id_field_name_for_block} . '-' . $block->{'id'};

            my $block_settings = {};

            foreach (qw(articles geo strategy brands)) {

                if (defined $settings->{$_}) {

                    my $json = to_json($settings->{$_}, canonical => TRUE);

                    my $md5 = md5_hex($json);

                    $block_settings->{$_} = $settings->{$_};

                    push @{$blocks_count->{$block_model}->{$md5}->{'public_ids'}}, $public_id;

                    $blocks_count->{$block_model}->{$md5}->{'json'} = $json;

                    $blocks_count->{$block_model}->{$md5}->{'type'} = $_;
                }
            }

            if (%{$block_settings}) {
                my $json_all_settings = to_json($block_settings, canonical => TRUE);

                my $md5_all_settings = md5_hex($json_all_settings);

                push @{$blocks_count->{$block_model}->{$md5_all_settings}->{'public_ids'}}, $public_id;

                $blocks_count->{$block_model}->{$md5_all_settings}->{'json'} = $json_all_settings;

                $blocks_count->{$block_model}->{$md5_all_settings}->{'type'} = 'all';

                $blocks_count->{$block_model}->{'blocks_count'}->{'with_cpm'} += 1;
            }
        }

        write_to_files($blocks_count, $login->{'login'}, $tsv);
    }

    $app->post_run();
}

sub _get_args {
    my $app = Application->new();
    $app->pre_run();

    $app->set_option('find_app_mem_cycle', 0);

    my $logins           = undef;
    my $limit_for_pages  = undef;
    my $limit_for_blocks = undef;
    my $limit_for_logins = undef;
    my $help             = 0;
    Getopt::Long::GetOptions(
        'logins=s'           => \$logins,
        'limit_for_pages=s'  => \$limit_for_pages,
        'limit_for_blocks=s' => \$limit_for_blocks,
        'limit_for_logins=s' => \$limit_for_logins,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    return {
        logins           => $logins,
        limit_for_pages  => $limit_for_pages,
        limit_for_blocks => $limit_for_blocks,
        limit_for_logins => $limit_for_logins,
        app              => $app,
    };
}

sub write_in_the_start {
    my $tsv = shift;

    foreach my $file (qw(block_settings block_count blocks_count_for_login)) {

        my $filename = "$file.tsv";

        open(my $fh, '>>', $filename) or die "Cannot open: $!";

        if ($file eq 'block_settings') {
            $tsv->print($fh, [qw(login model md5 type json)]);
        } elsif ($file eq 'block_count') {
            $tsv->print($fh, [qw(login model md5 type blocks_count public_ids)]);
        } else {
            $tsv->print($fh, [qw(login model blocks_count_with_cpm blocks_count_all)]);
        }
    }
}

sub _get_logins {
    my ($app, $args) = @_;

    if (defined $args->{'logins'}) {
        my $logins = from_json($args->{'logins'});

        return $app->partner_db->query->select(
            table  => $app->partner_db->users,
            fields => [qw(id login)],
            filter => [login => 'IN' => \$logins]
        )->get_all();
    } else {
        return defined $args->{'limit_for_logins'}
          ? $app->partner_db->query->select(
            table  => $app->partner_db->users,
            fields => [qw(id login)],
          )->limit($args->{'limit_for_logins'})->get_all()
          : $app->partner_db->query->select(
            table  => $app->partner_db->users,
            fields => [qw(id login)],
          )->get_all();
    }
}

sub _get_page_ids_for_login {
    my ($app, $owner_id, $args) = @_;

    my $page_ids_for_login = $app->partner_db->query->select(
        table  => $app->partner_db->all_pages,
        fields => [qw(page_id)],
        filter => [
            'AND',
            [['owner_id' => '=' => \$owner_id], ['model' => 'IN' => \$ALLOWED_MODELS], ['is_working' => '=' => \1]]
        ],
    );

    $page_ids_for_login =
      defined $args->{'limit_for_pages'}
      ? $page_ids_for_login->limit($args->{'limit_for_pages'})->get_all()
      : $page_ids_for_login->get_all();

    return $page_ids_for_login;
}

sub _normalize_settings {
    my $block = shift;

    my $strategy = delete $block->{'strategy'};

    foreach my $field (keys %{$block}) {

        #Округление в до ближайшего числа кратного 25
        if (exists $CPMS_HASH->{$field}) {
            if (defined $block->{$field}) {
                $block->{$field} =
                  ($block->{$field} % 25 > 13)
                  ? $block->{$field} + (25 - $block->{$field} % 25)
                  : $block->{$field} - $block->{$field} % 25;

                $block->{'strategy'}->{$field} = delete $block->{$field} if $block->{$field} > '0';
            } else {
                delete $block->{$field};
            }
        }

        if (defined $block->{$field}) {

            if (ref($block->{$field}) eq '') {

                my $from_json = from_json($block->{$field});

                if (ref($from_json) eq 'ARRAY') {

                    if (!scalar @{$from_json}) {

                        $block->{$field} = undef;

                    } else {

                        $block->{$field} = _get_settings_without_zeros($from_json, 'id');
                    }
                }
            } elsif (ref($block->{$field}) eq 'ARRAY') {

                $block->{$field} = _get_settings_without_zeros($block->{$field}, 'bid');
            }
        }
    }

    $block->{'strategy'}->{'strategy'} = $strategy if defined $block->{'strategy'};

    return $block;
}

sub _get_settings_without_zeros {
    my ($settings, $field_to_sort) = @_;

    my $settings_without_zeros = [];

    foreach (@{$settings}) {
        next unless defined $_->{'cpm'};
        $_->{'cpm'} =
          ($_->{'cpm'} % 25 > 13)
          ? $_->{'cpm'} + (25 - $_->{'cpm'} % 25)
          : $_->{'cpm'} - $_->{'cpm'} % 25;

        push @{$settings_without_zeros}, $_ if $_->{'cpm'} > '0';
    }

    return
      scalar @{$settings_without_zeros}
      ? [sort {$a->{$field_to_sort} cmp $b->{$field_to_sort}} @{$settings_without_zeros}]
      : undef;
}

sub write_to_files {
    my ($data, $login, $tsv) = @_;

    foreach my $file (qw(block_settings block_count blocks_count_for_login)) {
        my $filename = "$file.tsv";
        open(my $fh, '>>', $filename) or die "'$filename' $!";

        foreach my $block_model (keys %{$data}) {

            foreach my $md5 (keys %{$data->{$block_model}}) {

                if ($file eq 'block_settings') {
                    $tsv->print(
                        $fh,
                        [
                            $login, $block_model, $md5,
                            $data->{$block_model}->{$md5}->{'type'},
                            $data->{$block_model}->{$md5}->{'json'}
                        ]
                    ) unless $md5 eq 'blocks_count';
                } elsif ($file eq 'block_count') {
                    $tsv->print(
                        $fh,
                        [
                            $login,
                            $block_model,
                            $md5,
                            $data->{$block_model}->{$md5}->{'type'},
                            scalar @{$data->{$block_model}->{$md5}->{'public_ids'}},
                            to_json($data->{$block_model}->{$md5}->{'public_ids'}, canonical => TRUE)
                        ]
                    ) unless $md5 eq 'blocks_count';
                } else {
                    $tsv->print(
                        $fh,
                        [
                            $login,                                      $block_model,
                            $data->{$block_model}->{$md5}->{'with_cpm'}, $data->{$block_model}->{$md5}->{'all'}
                        ]
                    ) if $md5 eq 'blocks_count' && defined $data->{$block_model}->{$md5}->{'with_cpm'};
                }
            }
        }

        close $fh or die "$filename: $!";
    }
}

main();
__END__
