#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт добавляет Instream блоки с категориями на пейджи по шаблону

=head1 USAGE

  perl -Ilib bin/oneshots/add_video_blocks_on_categories.pl --page_ids=12312,534142 --dry_run
  perl -Ilib bin/oneshots/add_video_blocks_on_categories.pl --dry_run
  perl -Ilib bin/oneshots/add_video_blocks_on_categories.pl --rollback_log=add_video_blocks_on_categories.log
  perl -Ilib bin/oneshots/add_video_blocks_on_categories.pl \
        --rollback_log=add_video_blocks_on_categories.log \
        --exclude_page_ids=12312

=head1 OPTIONS

  sample_block_public_id - Public ID блока который нужно использовать как шаблон
  page_ids               - Список ID площадок через запятую (необязательный)
  exclude_user_ids       - Список пользователей через запятую для которых не нужно создавать блоки (необязательный)
  exclude_page_ids       - Список ID площадок через запятую для которых не нужно создавать блоки (необязательный)
  category_ids           - Список ID категорий через запятую (необязательный)

=cut

use lib::abs qw(../../lib);

use qbit;

use Utils::PublicID;
use Utils::ScriptWrapper;

sub args {
    my ($opts) = @_;

    return (
        'exclude_page_ids=s'       => \$opts->{exclude_page_ids},
        'page_ids=s'               => \$opts->{'page_ids'},
        'rollback_log=s'           => \$opts->{'rollback_log'},
        'exclude_user_ids=s'       => \$opts->{'exclude_user_ids'},
        'sample_block_public_id=s' => \$opts->{'sample_block_public_id'},
        'category_ids=s'           => \$opts->{'category_ids'},

    );
}

run(
    sub {
        my ($app, $opts) = @_;

        _prepare_opts($opts);

        if ($opts->{rollback_log}) {
            _rollback__archive_blocks($app, $opts);
        } else {
            _add_blocks($app, $opts);
        }
    }
   );

sub _prepare_opts {
    my ($opts) = @_;

    foreach my $opt_name (qw(exclude_page_ids page_ids exclude_user_ids category_ids)) {
        $opts->{$opt_name} = [split(/,/, $opts->{$opt_name} // '')];
    }
}

sub _add_blocks {
    my ($app, $opts) = @_;

    throw Exception 'Expected "sample_block_public_id"' unless defined($opts->{'sample_block_public_id'});

    my $dry_run = $opts->{'dry_run'};

    my $add_fields = $app->video_an_site_instream->get_add_fields();
    delete $add_fields->{$_} for (qw(adfox_user_id picategories));

    my $sample_block =
      $app->video_an_site_instream->get($opts->{'sample_block_public_id'}, 'fields' => [keys %$add_fields]);
    delete $sample_block->{page_id};

    my $category_ids = @{$opts->{'category_ids'}} ? $opts->{'category_ids'} : [$sample_block->{'category_id'}];
    my %category_name_by_id =
      map {$_->{'id'} => $_->{'name'}{'ru'}} @{$app->video_an_site_categories->get_default_video_categories()};

    my $filter = [
        (@{$opts->{'exclude_user_ids'}} ? ['owner_id', 'NOT IN', \$opts->{'exclude_user_ids'}] : ()),
        ['multistate', 'IN', \$app->video_an_site->get_multistates_by_filter('not (deleted or protected)')],
    ];

    if (@{$opts->{'page_ids'}}) {
        push(@$filter, ['id', 'IN', \$opts->{'page_ids'}]);
    }

    if (@{$opts->{exclude_page_ids}}) {
        push(@$filter, ['id', 'NOT IN', \$opts->{'exclude_page_ids'}]);
    }

    my $page_list = $app->partner_db->video_an_site->get_all(
        fields => [qw(id)],
        filter => ['AND' => $filter],
    );

    if (@{$opts->{'page_ids'}} && @{$opts->{'page_ids'}} != @$page_list) {
        my %found_pages = map {$_ => TRUE} @$page_list;
        print STDERR logstr('PAGES NOT FOUND:', join(', ', grep {!$found_pages{$_}} @{$opts->{'page_ids'}}));
    }

    foreach my $page (@$page_list) {
        foreach my $category_id (@$category_ids) {
            my $block2add = {
                %$sample_block,
                page_id       => $page->{id},
                category_id   => $category_id,
                category_path => $category_name_by_id{$category_id},
            };

            print logstr('BLOCK SETTINGS:', $block2add);

            unless ($dry_run) {
                try {
                    my $public_id = $app->video_an_site_instream->add(%$block2add);

                    print logstr('ADDED BLOCK:', $public_id);
                }
                catch {
                    print STDERR logstr('ERROR:', $block2add, shift->message);
                };
            }
        }
    }
}

sub _rollback__archive_blocks {
    my ($app, $opts) = @_;

    my %exclude = map {$_ => TRUE} @{$opts->{exclude_page_ids}};

    unless (open(FH, '<', $opts->{rollback_log})) {
        throw Exception "Could not open rollback_log file: $!";
    }

    my %pages;
    my @block_public_id_list =
      map {$_ =~ /ADDED BLOCK:\t(\w+-\w+-\d+-\d+)/; $1;} grep {$_ =~ /ADDED BLOCK:\t(\w+)/} <FH>;

    for my $block_public_id (@block_public_id_list) {
        my $page_id = (split_block_public_id($block_public_id))[1];

        if ($exclude{$page_id}) {
            print logstr('SKIP BLOCK:', $block_public_id);
            next;
        }

        try {
            $app->video_an_site_instream->do_action($block_public_id, 'delete') unless ($opts->{dry_run});
            $pages{$page_id}++;
            print logstr('DELETED BLOCK:', $block_public_id);
        }
        catch {
            print STDERR logstr('ERROR:', $block_public_id, $_[0]->message);
        };
    }

    print logstr('PAGES LIST:', join(',', sort keys %pages));
}
