#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

    sets or unsets dsps for blocks

=head1 USAGE

    perl -I./lib ./bin/oneshots/block_dsps_set_unset.pl --blocks=blocks.in --leave=1,2
    perl -I./lib ./bin/oneshots/block_dsps_set_unset.pl --blocks=blocks.in --remove=3,4

=head1 OPTIONS

    blocks   - file with block public_ids, one id per line
    page_ids - blocks on these pages will be updated
    leave    - dsp_id to leave on blocks, other will be removed
    remove   - dsp_id to remove on blocks

=cut

use strict;
use warnings;

use File::Slurp;

use qbit;

use Utils::DB qw(fields_to_filter);
use Utils::PublicID;
use Utils::ScriptWrapper;

sub args {
    my ($opts) = @_;

    return (
        'blocks:s'   => \$opts->{blocks},
        'leave:s'    => \$opts->{leave},
        'remove:s'   => \$opts->{remove},
        'page_ids:s' => \$opts->{page_ids},
    );
}

run(
    sub {
        my ($app, $opts) = @_;

        my %pages_to_update;

        my %dsps_to_remove = map {$_ => 1} split(',', $opts->{remove} // '');
        my %dsps_to_leave  = map {$_ => 1} split(',', $opts->{leave}  // '');
        if (((keys %dsps_to_remove) && (keys %dsps_to_leave)) || (0 == (keys %dsps_to_remove) + (keys %dsps_to_leave)))
        {
            print logstr 'please pass --remove or --leave option: comma-separated dsps to remove or leave on blocks';
            return;
        }
        my $is_removing = 0 < keys %dsps_to_remove;

        my @page_id_list = split(',', $opts->{page_ids} // '');
        my @block_list;
        unless (($opts->{blocks} && -f $opts->{blocks})) {
            unless (@page_id_list) {
                print logstr
                  'please pass --blocks option: file name with list of blocks public_ids or --page_ids option';
                return;
            }
        } else {
            my @public_id_list = map {trim($_)} read_file($opts->{blocks});
            @block_list = map {
                my ($prefix, $page_id, $block_id) = split_block_public_id($_);
                {
                    page_id => $page_id,
                    id      => $block_id,
                }
              } grep {
                '' ne $_
              } @public_id_list;
        }

        my $all_blocks_filter;
        if (@block_list) {
            $all_blocks_filter =
              $app->partner_db->filter(fields_to_filter(['page_id', 'id'], \@block_list, for_db => TRUE));
            if (@page_id_list) {
                $all_blocks_filter->or({page_id => \@page_id_list});
            }
        } else {
            $all_blocks_filter = $app->partner_db->filter({page_id => \@page_id_list});
        }

        my @model_list = map {$_->{model}} @{
            $app->partner_db->all_blocks->get_all(
                fields   => ['model'],
                distinct => TRUE,
                filter   => _get_filter($app, \@block_list, \@page_id_list),
            )
          };

        for my $model (@model_list) {
            # Если у продукта нет настройки DSP, скипаем
            next unless $app->$model->DOES('Application::Model::Role::Has::DSPS');

            my @regular_blocks;
            my @godmode_blocks;

            my $block_list = $app->$model->get_all(
                filter => _get_filter($app, \@block_list, \@page_id_list, $model),
                fields => [qw(page_id id bk_data is_custom_bk_data)],
            );

            unless (@$block_list) {
                print logstr("SKIP: blocks not found in $model");
                next;
            }

            for my $block (@$block_list) {
                # Если блок в годмоде - апдейтим bk_data
                if ($block->{is_custom_bk_data}) {
                    my $bk_data = from_json($block->{bk_data});
                    my @new_dsp_info;
                    for my $dsp (@{$bk_data->{DSPInfo}}) {
                        # Отфильтровываем DSP
                        if ($is_removing && exists($dsps_to_remove{$dsp->{DSPID}})
                            || (!$is_removing && !exists $dsps_to_leave{$dsp->{DSPID}}))
                        {
                            print logstr(
                                'DELETE_GODMODE_DSP',
                                {
                                    model   => $model,
                                    page_id => $block->{page_id},
                                    id      => $block->{id},
                                    dsp_id  => $dsp->{DSPID},
                                }
                            );
                        } else {
                            push @new_dsp_info, $dsp;
                        }
                    }
                    if (0 == @new_dsp_info) {
                        print logstr(
                            'SKIP: no DSP left on block',
                            {
                                model   => $model,
                                page_id => $block->{page_id},
                                id      => $block->{id},
                            }
                        );
                    }
                    # Если дспшек стало меньше, добавляем блок в список к апдейту годмода
                    elsif (@{$bk_data->{DSPInfo}} > @new_dsp_info) {
                        $bk_data->{DSPInfo} = \@new_dsp_info;
                        $block->{bk_data} = to_json($bk_data, pretty => 1);
                        push @godmode_blocks, $block;
                    }
                }
                # В список к удалению из block_dsps добавляем в любом случае
                # Даже если блок в годмоде, когда его вернут, внешних дсп там тоже не должно быть
                push @regular_blocks, [$block->{page_id}, $block->{id}];

                $pages_to_update{$block->{page_id}} = 1;
            }

            # Апдейтим годмоды
            $app->partner_db->$model->add_multi(\@godmode_blocks, duplicate_update => TRUE)
              if @godmode_blocks && !$opts->{dry_run};

            # Чистим block_dsps
            my $dsp_list = scalar keys %dsps_to_remove ? [keys %dsps_to_remove] : [keys %dsps_to_leave];
            my $filter = $app->partner_db->filter(
                fields_to_filter(
                    ['page_id', 'block_id'],
                    [map {{page_id => $_->{page_id}, block_id => $_->{id},}} @block_list],
                    for_db => TRUE
                )
            );
            $filter->or({page_id => \@page_id_list}) if @page_id_list;
            $filter->and(['dsp_id', (keys %dsps_to_leave ? 'NOT IN' : 'IN'), \$dsp_list]);

            if (@regular_blocks && !$opts->{dry_run}) {
                $app->partner_db->block_dsps->delete($filter);
            }
        }

        print "PAGES TO UPDATE (" . scalar keys(%pages_to_update) . ")\n";
        print join(',', sort {$a <=> $b} keys(%pages_to_update)), "\n";
    }
   );

sub _get_filter {
    my ($app, $block_list, $page_id_list, $model) = @_;

    my $filter;
    if ($model) {
        if (@$block_list) {
            $filter = fields_to_filter(['page_id', 'id'], $block_list);
            if (@$page_id_list) {
                $filter = ['OR', [$filter, [$app->$model->get_page_id_field_name() => 'IN' => $page_id_list]]];
            }
        } else {
            $filter = [$app->$model->get_page_id_field_name() => 'IN' => $page_id_list];
        }
    } else {
        if (@$block_list) {
            $filter = $app->partner_db->filter(fields_to_filter(['page_id', 'id'], $block_list, for_db => TRUE));
            if (@$page_id_list) {
                $filter->or({page_id => $page_id_list});
            }
        } else {
            $filter = $app->partner_db->filter({page_id => $page_id_list});
        }
    }
    return $filter;
}
