#!/usr/bin/perl

=encoding UTF-8
=cut

=head1 DESCRIPTION

=cut

# project modules
use lib::abs qw(
  ../../lib
  );
use qbit;
use Application;
use Utils::ScriptWrapper;

# common modules
use feature 'say';
use Carp;
use DDP;

# subs
sub sync_block_multistates {
    my ($app, $model_name) = @_;

    my $model        = $app->$model_name;
    my $page_id_name = $model->get_page_id_field_name();

    #
    # Sync: deleted page -> do stop and then delete_with_page for all blocks
    #
    say "# Sync: deleted page -> do stop and then delete_with_page for all blocks if they are not already deleted";

    foreach my $page (
        @{
            $model->get_all(
                fields => [$page_id_name, qw(id multistate)],
                filter   => {multistate => 'deleted'},
                order_by => [$page_id_name],
            )
        }
      )
    {

        my $page_id = $page->{$page_id_name};
        next unless defined($page_id);

        foreach my $block_model ($model->get_block_models()) {
            my $block_page_id_name = $block_model->get_page_id_field_name();

            my $blocks = $block_model->get_all(
                fields => [$block_page_id_name, qw(id public_id multistate)],
                filter => {$block_page_id_name => $page_id},
                order_by => [$block_page_id_name, 'id'],
            );

            foreach my $block (@$blocks) {
                next if $block->{'id'} == 0;    # don't touch 0-blocks

                if ($block_model->check_action($block, 'stop')) {
                    say "$block->{'public_id'} - stop";
                    $block_model->do_action($block, 'stop');
                }

                # Allowed state for delete_with_page action does not depend on stop action results
                # so we use the same $block state in check_action() as in the step above.
                # delete_with_page will not touch individually deleted blocks.

                if ($block_model->check_action($block, 'delete_with_page')) {
                    say "$block->{'public_id'} - delete_with_page";
                    $block_model->do_action($block, 'delete_with_page');
                }
            }
        }
    }

    #
    # Sync: working or testing page -> start all blocks except those deleted individually (not deleted_with_page)
    #
    say '# Sync: working or testing page -> start all blocks except working or deleted';
    foreach my $page (
        @{
            $model->get_all(
                fields => [$page_id_name, qw(id multistate)],
                filter   => {multistate => 'working or testing'},
                order_by => [$page_id_name],
            )
        }
      )
    {

        my $page_id = $page->{$page_id_name};
        next unless defined($page_id);

        foreach my $block_model ($model->get_block_models()) {
            my $block_page_id_name = $block_model->get_page_id_field_name();

            # Take working or testing pages.
            # Take not working and not deleted blocks (the block state normally checked as a prerequisite for start)
            # and then forcefully start on a page ignoring possible can_action() results.
            # This way we can sync with protected pages.
            #
            # "Working" block state is currently only for PI2 reference.
            # It does not affect sending to BK.
            # All _not deleted_ blocks on a page are sent to BK.
            #
            # So we don't resend the page to BK after starting blocks here.
            #
            my $blocks = $block_model->get_all(
                fields => [$block_page_id_name, qw(id public_id multistate)],
                filter => {$block_page_id_name => $page_id, multistate => 'not working and not deleted'},
                order_by => [$block_page_id_name, 'id'],
            );

            foreach my $block (@$blocks) {
                next if $block->{'id'} == 0;    # don't touch 0-blocks

                say "$block->{'public_id'} - start";
                $block_model->force_do_action($block, 'start');    # FORCE, ignore can_action() check results
            }
        }
    }

    return TRUE;
}

# main
run(
    sub {
        my ($app, $opts) = @_;

        my @models = qw(
          context_on_site_campaign
          search_on_site_campaign
          mobile_app_settings
          internal_context_on_site_campaign
          internal_search_on_site_campaign
          internal_mobile_app
          video_an_site
          );

        foreach my $model_name (@models) {

            say "";
            say "## Model - $model_name";

            sync_block_multistates($app, $model_name);
        }

        $app->post_run();

        say "#END";
    }
   )

__END__
