#!/usr/bin/perl -w

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для переотправки пейджей в БК.

=head1 USAGE

  pi_oneshot_runner --branch=<branch> --jb=<branch> --type=perl|sql

  см https://wiki.yandex-team.ru/partner/w/projects/support/oneshots

=head1 OPTIONS

    Обязательные
        type    - тип скрипта sql|perl|task
        format  - форматировать вывод для ST (всегда передается в бою)
        ticket  - куда писаьть коментарии при format=1

    Необязательные
        stage   - в каком окружении запускается (testing,production,dev,docker,keep)
        branch  - ветка на которой выполнять
        jb      - java branch

    Дебажные (необязательные)
        path                 - где развернут partner2 (по дефолту /home/yabs/partner2)
        check_pages          - включен по дефолту, позволяет отключать сбор пейджей из базы
        validate             - включен по дефолту, позволяет отключать валидацию
        resend_pages         - включен по дефолту, позволяет отключать отправку в БК
        single               - SQL одним куском, нужно для хранимых процедур, когда лопается парсер
        nodb                 - позволяет не поднимать докерную базу, полезно при дебаге при stage=keep
        skip-infra-event     - не создает событие в infra
        skip-checkout        - не чекаутит версию прода, полезно при дебаге при stage=keep
        skip-build-java      - не билдид java, полезно при дебаге при stage=keep
        skip-start-java      - не поднимаем java jsonapi, полезно при дебаге при stage=keep
        skip-ya-upload       - не заливает логи в sanbox, бывает полезно при дебаге
        skip-comment-ticket  - не добавляет каменты в тикет, бывает полезно при дебаге
        force-resend-pages   - всегда перепосылает пейджи в БК, бывает полезно при дебаге
        force-java-jsonap    - всегда поднимаие java jsonapi, бывает полезно при дебаге

=cut

use lib::abs qw(
  ../lib
  ../local/lib/perl5
  );

use qbit;
use PiConstants qw($TRANSPORTS_KVSTORE_KEYS);

use Capture::Tiny qw(capture);
use Errno qw(EAGAIN);
use Fcntl qw(:flock);
use File::Slurp qw(read_file write_file);
use Getopt::Long();
use IPC::Open3;
use POSIX qw(strftime);
use PiSecrets;
use Pod::Usage;
use Time::HiRes qw(gettimeofday);
use Yandex::StarTrek;
use LWP::UserAgent;
use IO::Socket::SSL qw(SSL_VERIFY_NONE);
use HTTP::Request;
use Data::Dumper;
use Utils::Logger qw(INFO INFOF ERRORF);
use Utils::MakeFile;
use utf8;

# особенность шебанга - все параметры идут одной строкой, вторым параметром идет имя файла
my $cmd_args_str     = shift;
my $self_script_name = shift;

my $INFRA_SERVICE_ID = 2034;
my %INFRA_ENV        = (prod => 3163, test => 3164);
my $INFRA_ESTIMATE   = 30 * 60;

my $ROOT           = "/home/yabs";
my $ARCADIA_ROOT   = $ENV{ARCADIA_ROOT} // "$ROOT/arcadia";
my $DEFAULT_PATH   = "$ARCADIA_ROOT/partner/perl/partner2";
my $ARTEFACTS_PATH = "%s/oneshot_artifacts";
my $LOCK_FILE      = "$ARTEFACTS_PATH/lockfile";
my $LOGS_PATH      = "$ARTEFACTS_PATH/%s";

my $DEFAULT_STAGE = 'testing';
my $STAGE         = $DEFAULT_STAGE;
my %stages        = (
    $DEFAULT_STAGE => 'config_oneshot_deploy_db',
    production     => 'config_production',
    dev            => 'config_dev',
    docker         => 'config_docker',
    keep           => 'config_oneshot_keep_local',
);

my $oneshot_types = {
    perl => 1,
    sql  => 1,
    task => 1
};

my $timer = {actions => []};

my $RESEND_OPTION_SPLIT = 10;

my $TICKET;
my $STEPS_TOTAL = 5;
my $STEPS_COUNT = 0;

my $partner2_path         = $DEFAULT_PATH;
my $jsonapi_start_invoked = '';
my $jsonapi_pid_file      = "";

our (@result_out, @result_err);
my $result_err_ref = \@result_err;
my $result_log_ref = \@result_out;
my ($oneshot_log, $oneshot_err_log, $lock_fh);

my $format;
my $infra_id;
my $mysql_command;
my $is_skip_ya_upload;
my $is_skip_comment_ticket;
my $is_help_mode = 0;

$SIG{TERM} = sub {
    CORE::die 'Got termination request, killed by TERM signal (timeout probaly)';
};

$SIG{__DIE__} = sub {
    unless ($^S) {
        # Пушить нужно в ссылку на оригинальный массивd, чтобы после `local @result_err` ошибка не пропала при выводе
        push @$result_err_ref, ['ERROR: ', $_[0]];
        CORE::die $_[0];
    }
};

main();

sub main {
    $timer->{start} = gettimeofday;

    my $args = _get_args($cmd_args_str, $self_script_name);

    _init($args);

    $lock_fh = get_lock(\@result_err, $TICKET);

    $infra_id = infra_open_event($STAGE eq 'production' ? 'prod' : 'test') unless $args->{'skip-infra-event'};

    _arc_checkout($args->{branch}, $args->{'skip-checkout'}) unless @result_err;

    _buid_java($args->{jb}, $args->{'skip-build-java'}) unless @result_err;

    _init_database($STAGE) if !$args->{nodb} && !@result_err;

    _init_java_json_api($args->{validate}, $STAGE, $args->{'skip-start-java'}, $args->{'force-java-jsonapi'})
      unless @result_err;

    unless (@result_err) {
        create_oneshot_pages_and_block_tables() if $args->{check_pages};

        # run oneshot itself
        run_oneshot($args);

        my $pages = check_affected_pages_and_blocks($args) if $args->{check_pages};

        if ($args->{resend_pages} || $args->{validate}) {
            (undef, my $pages_path) = get_log_file_path('page_list', 'txt');
            write_file($pages_path, {binmode => ':utf8'}, $pages);

            # отправляем пейджи только в production
            if ($args->{resend_pages}) {
                resend_pages($args->{resend_pages}, $pages_path);
            }

            # валидируем как на ТС так и в production
            if ($args->{validate}) {
                validate_pages($pages_path);
            }
        }
    } else {
        push @result_err, ["PROCESS NOT RUNNING", ''];
    }

    # NOTE! далее см sub END
}

sub END {

    return if $is_help_mode;

    _kill_java_jsonapi() if $jsonapi_start_invoked;

    free_lock($lock_fh);

    infra_close_event($infra_id) if $infra_id;

    _cleanup_old_logs();

    _print_results();
}

sub _init {
    my ($args) = @_;

    binmode(STDOUT, ':utf8');
    binmode(STDERR, ':utf8');

    if ($args->{ticket}) {
        $TICKET = $args->{ticket};
    } elsif ($ENV{ONESHOT_TICKET}) {
        $TICKET = $ENV{ONESHOT_TICKET};
    } else {
        $TICKET = $args->{file};
        $TICKET =~ s/.*?([A-Z]+-\d+)_.*/$1/g;
    }

    chdir $partner2_path;
    $ARTEFACTS_PATH = sprintf $ARTEFACTS_PATH, $partner2_path;
    $LOCK_FILE      = sprintf $LOCK_FILE,      $partner2_path;
    $LOGS_PATH      = sprintf $LOGS_PATH,      $partner2_path, $TICKET;

    foreach my $path ($ARTEFACTS_PATH, $LOGS_PATH) {
        my (undef, $std_err, $code) = run_command("mkdir -p $path");
        push @result_err, ['ERROR mkdir', $path . ". " . $std_err] if $code;
    }

    $args->{resend_pages} = 0 unless $STAGE eq 'production';
    $args->{resend_pages} = 'java' if $args->{'force-resend-pages'};

    foreach my $key (qw(resend_pages validate check_pages)) {
        $STEPS_TOTAL -= 1 unless $args->{$key};
    }
    $STEPS_TOTAL -= 1 if $is_skip_comment_ticket || $is_help_mode || !$format;

    $STAGE = delete $args->{stage};
    if ($STAGE eq 'production') {
        $ENV{ENVIRONMENT} = 'production';
    }

    return 1;
}

sub _add_comment {
    my ($comment, @params) = @_;

    $comment = sprintf($comment, @params) if @params;

    my $time_prefix = strftime('[%F %T]', localtime time);
    push @result_out, {comment => $time_prefix . ' ' . $comment};
    if ($result_log_ref != \@result_out) {
        push @$result_log_ref, {comment => $time_prefix . ' ' . $comment};
    }

    INFO $comment, "\n";
}

sub _arc_checkout {
    my ($branch, $is_keep_code) = @_;

    my $start = gettimeofday;

    if (!$branch && !$is_keep_code) {
        _add_comment('Getting perl prod version');
        my $prod_git_tag = _get_prod_version('s');
        _add_comment('Got prod perl git tag %s', $prod_git_tag);
        $branch = "tags/releases/partner/perl/$prod_git_tag";
    }

    $branch //= '';

    _add_comment('Checkout branch %s', $branch);

    # todo execute pull in case of branch
    my ($std_out, $std_err, $code) = run_command("arc reset --hard HEAD && arc checkout $branch && arc pull")
      unless $is_keep_code;
    if ($code) {
        push @result_err, ['ERROR checkout', $std_err];
    } else {
        push @result_out, ["checkout $branch", ''];
    }

    push @{$timer->{actions}},
      {
        name   => 'Checkout branch',
        start  => $start,
        end    => scalar(gettimeofday()),
        status => ($code ? 'FAIL' : 'OK')
      };

    return 1;
}

sub _get_prod_version {
    my ($json_key) = @_;

    my ($prod_git_tag, $std_err, $code) =
      run_command("curl -s 'https://partner.yandex-team.ru/partner2_production_version.json' | jq -r .$json_key");

    chomp($prod_git_tag);

    push @result_err, ['ERROR getting perl production version', $std_err] if $std_err;

    return $prod_git_tag;
}

sub _init_database {
    my ($stage, $result_out, $result_err) = @_;

    if ($stage) {
        my $start = gettimeofday;

        _add_comment('Start init database stage=%s', $stage);

        my $make_target = $stages{$stage};
        my ($std_out, $std_err, $code) = run_command("make $make_target");
        if ($code) {
            push @result_err, ["ERROR make $make_target", $std_err];
        } else {
            push @result_out, ["make $make_target", ''];
        }

        push @{$timer->{actions}},
          {
            name   => 'Init database',
            start  => $start,
            end    => scalar(gettimeofday()),
            status => $code ? 'FAIL' : 'OK'
          };
    }

    # прогреваем кеш - читаем всю context_on_site_rtb с джойнами
    my $tries = 3;
    while (--$tries > 0) {
        my ($std_out, $std_err, $code) = run_sql_command(
            q[
                SELECT  count(id)
                FROM (
                        SELECT  R.id,
                                C.page_id,
                                U.id as uid
                        FROM    context_on_site_rtb R
                                INNER JOIN context_on_site_campaign C ON (R.campaign_id = C.page_id)
                                INNER JOIN users U ON (C.owner_id=U.id)
                        WHERE   R.model = 'context_on_site_rtb'
                ) b;
            ],
            1,
            1
        );
        last if $code == 0;
    }

    return 1;
}

sub get_log_file_path {
    my ($file_name, $suffix) = @_;

    local $$ = 10500 if $STAGE eq 'keep';

    my $log_path = sprintf '%s/%s_%s_%s.%s', $LOGS_PATH, $$, strftime("%Y-%m-%d", localtime), $file_name,
      $suffix // 'log';

    open(my $fh, '>', $log_path) or die "open: $!";

    return ($fh, $log_path);
}

sub _init_java_json_api {
    my ($is_validate, $stage, $is_skip_start_java, $is_force_java_jsonapi) = @_;

    $jsonapi_pid_file = "$partner2_path/oneshot_jsonapi.pid";

    # Поднимаем java_json_api в тестовом окружении (в проде уже и так все поднято)
    # так как при валидации, при получении bk_data, дергается ручка enrich локально
    if (!@result_err && (($is_validate && $stage ne 'production') || $is_force_java_jsonapi)) {

        my $start = gettimeofday;

        (undef, my $java_api_run_log) = get_log_file_path('java_jsonapi_run_log');
        (undef, my $java_api_log) = get_log_file_path('java_jsonapi_log', 'json');

        my $java_jsonapi_port  = +Utils::MakeFile::get_java_jsonapi_port();
        my $java_actuator_port = $java_jsonapi_port + 100;
        my $java_command       = "$partner2_path/bin/java_start.sh -o $java_api_log"
          . " --pid_file $jsonapi_pid_file --port $java_jsonapi_port -- --api  2>&1 1>$java_api_run_log";

        _add_comment("Starting java jsonapi: \n\t%s", $java_command);

        $jsonapi_start_invoked = !system("/bin/bash $java_command &") unless $is_skip_start_java;

        my $jsonapi_started = '';
        if (!$jsonapi_start_invoked) {
            push @result_err, ["Could not start JsonAPI", ''] unless $is_skip_start_java;
        } else {
            my $count = 0;
            while (++$count <= 12 && !$jsonapi_started) {
                sleep(5);
                push @result_out, ["$count. starting JsonAPI", ''];
                $jsonapi_started =
                  !system("curl -s http://localhost:$java_actuator_port/actuator/health | grep '{\"status\":\"UP\"}'");
            }

            if (!$jsonapi_started) {
                push @result_err, ["Could not start JsonAPI within 120 seconds", ''];
            }
        }

        my $is_error = !$is_skip_start_java && !$jsonapi_started;
        push @{$timer->{actions}},
          {
            name     => 'Start javа JSONAPI',
            start    => $start,
            end      => scalar(gettimeofday()),
            log_file => $java_api_run_log,
            status   => $is_error ? 'FAIL' : 'OK'
          };
    }

    return 1;
}

sub run_oneshot {
    my ($args) = @_;

    my $start = gettimeofday;

    local @result_out;
    local @result_err;

    my @args = ($args, get_content($args->{file}));
    if ($args->{type} eq 'perl') {
        _add_comment('Start running perl script');
        make_perl(@args);
    } elsif ($args->{type} eq 'sql') {
        _add_comment('Start runing mysql script');
        make_sql(@args);
    } elsif ($args->{type} eq 'task') {
        _add_comment('Start queue task');
        make_queue_task(@args);
    }

    my $out = _format_out(\@result_out, \@result_err);

    (my $out_fh, $oneshot_log) = get_log_file_path('oneshot');
    (my $err_fh, $oneshot_err_log) = get_log_file_path('oneshot', 'err');
    _dump_logs($out_fh, \@result_out, $err_fh, \@result_err);

    my $done = gettimeofday;
    my $time_stat_str = sprintf("Executing:  %s\n", _sec_to_time($done - $start));

    my $is_error = @result_err > 0;
    add_result_to_ticket(
        $is_error ? $oneshot_err_log : $oneshot_log,
        'Результат выполнения',
        $is_error, $out, $time_stat_str
    );

    push @{$timer->{actions}},
      {
        name     => 'Run oneshot',
        start    => $start,
        end      => scalar(gettimeofday()),
        log_file => $is_error ? $oneshot_err_log : $oneshot_log,
        status   => $is_error ? 'FAIL' : 'OK',
      };

    return 1;
}

sub _print_results {

    _add_comment('Start print results');

    # Dump log to file
    (my $stdout_fh, my $stdout_log) = get_log_file_path('oneshot_runner');
    (my $stderr_fh, my $stderr_log) = get_log_file_path('oneshot_runner', 'err');
    _dump_logs($stdout_fh, \@result_out, $stderr_fh, \@result_err);

    # Dump log to ticket
    if ($format) {
        my $out = _format_out(\@result_out, \@result_err);

        print("\n%%}>\n", $out, "\n\n<{Status\n%%\n");

        my ($first_failed_log) =
          map {$_->{log_file}} grep {$_->{status} ne 'OK' && $_->{log_file}} @{$timer->{actions}};
        my $is_error = scalar(@result_err || (grep {$_->{status} ne 'OK'} @{$timer->{actions}}));

        my $done          = gettimeofday;
        my $time_stat_str = join "\n",
          sprintf("%-23s %s", 'Total:', _sec_to_time($done - $timer->{start})),
          map {sprintf(" -%-21s %s", $_->{name} . ':', _sec_to_time($_->{end} - $_->{start}))}
          @{$timer->{actions} // []};

        add_result_to_ticket(
            $is_error ? $first_failed_log : $stdout_log,
            'Результат работы',
            $is_error, $out, $time_stat_str
        );
    }

    return 1;
}

sub _dump_logs {
    my ($stdout_fh, $result_out, $stderr_fh, $result_err) = @_;

    for my $out ([$stdout_fh, \@result_out], [$stderr_fh, \@result_err]) {
        my $h = $out->[0];
        for my $row (@{$out->[1]}) {
            if (ref $row eq 'ARRAY') {
                print $h "-----\n", $row->[0], "\n-----\n";
                print $h $row->[1], "\n=====\n" if $row->[1];
            } elsif (ref $row eq 'HASH') {
                if ($row->{command}) {
                    print $h "-----\n", $row->{command}, "\n-----\n";
                    print $h $row->{result},  "\n=====\n" if defined $row->{result};
                    print $h $row->{warning}, "\n=====\n" if defined $row->{warning};
                } elsif ($row->{comment}) {
                    print $h $row->{comment}, "\n",;
                }
            }
        }
    }

    return 1;
}

sub _format_out {
    my ($result_out, $result_err) = @_;

    my $out = '';
    open(my $h, '>', \$out) or die "open: $!";

    for my $row ((@$result_out, @$result_err)) {
        if (ref $row eq 'ARRAY') {
            if ($row->[1]) {
                print $h "<{", $row->[0], "\n\n";
                print $h "%%\n", quote_out($row->[1]), "%%\n}>\n";
            } else {
                print $h $row->[0], "\n";
            }
        } elsif (ref $row eq 'HASH') {
            if ($row->{command}) {
                my $is_big = length($row->{command}) > 1000 ? 1 : 0;
                printf $h "1. !!(%s)%s!! !!(grey)%s - %s сек!!\n",
                  $row->{color},
                  $row->{title},
                  _time_to_str($row->{startTime}),
                  $row->{elapsed};
                printf $h "  <{%s...\n", $row->{command} =~ m/^(.{0,80})/ if $is_big;
                print $h "  %%";
                print $h "($row->{type} nomark)" if $row->{type};
                print $h "\n";
                print $h $row->{command}, "%%\n";
                print $h "}>\n" if $is_big;

                _print_table($h, $row, 'result', 'Вывод');
                _print_table($h, $row, 'warning', 'Примечания') if $row->{warning};
            }
        }
    }
    close($h) or die "close: $!";

    return $out;

}

sub _print_table {
    my ($h, $row, $key, $name) = @_;
    my $is_big = length($row->{$key}) > 1000 ? 1 : 0;
    printf $h "  %s!!(%s)%s%s!!\n", ($is_big ? '<{' : ''), $row->{color}, $name,
      ($is_big ? sprintf(' (%s строк)', scalar(map {1} $row->{$key} =~ m/(\n)/g) - 4) : '');
    print $h "  ", to_table($row->{$key});
    print $h "\n}>\n" if $is_big;
}

sub get_content {
    my ($filename) = @_;

    my $content = eval {read_file($filename, binmode => ':utf8')};
    push @result_err, ["ERROR", $@] if $@;

    $content //= '';
    $content =~ s/^.+?\n//;

    return $content;
}

sub make_perl {
    my ($args, $data) = @_;

    my $code = "
package OneShot;
use lib qw(lib);
use qbit;
use Application;
\$ENV{FORCE_LOGGER_TO_SCREEN} = 1;
my \$app = Application->new;
\$app->pre_run();
\$app->set_cur_user({id => 0});
{
    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {TRUE};
}
\$app->system_events->start('oneshot', ticket => '$TICKET', oneshot_type => '$args->{type}');
#line 2 \"$args->{file}\"
$data;
\$app->post_run();
";

    my $error;
    my ($std_out, $std_err, $exit_status) = capture {
        eval $code;
        if ($@) {
            $error = $@;
        }
    };

    push @result_err, ["STDERR", $std_err] if defined $std_err and $std_err =~ /\S/;
    push @result_out, ["STDOUT", $std_out] if defined $std_out and $std_out =~ /\S/;
    push @result_err, ["ERROR",  $error]   if defined $error   and $error   =~ /\S/;

    return 1;
}

sub get_sql_command {
    unless ($mysql_command) {
        my $config = from_json(read_file('lib/DatabaseConfig.json', binmode => ':utf8'))->{partnerdb2};
        if (my $password = $config->{password}) {
            $ENV{MYSQL_PWD} = $password;
        }
        $mysql_command =
            "mysql -v -v "
          . '--batch'
          . " -u $config->{user}"
          . " --host=$config->{host}"
          . " --port=$config->{port}"
          . " --default-character-set=utf8"
          . " -D partner";
    }

    return $mysql_command;
}

sub make_sql {
    my ($args, $data) = @_;

    my @commands = $args->{single} ? ($data) : (split /;\s*?\n\s*/, $data);

    for my $command (@commands) {
        $command =~ s/^\s+//g;
        $command =~ s/\s+$//g;
        next unless $command;

        process_sql_command($command);

        last if @result_err;
    }

    return 1;
}

sub create_oneshot_pages_and_block_tables {

    _add_comment('Start create tables "oneshot_pages" & "oneshot_blocks"');

    run_sql_command(
        q[
        DROP TABLE IF EXISTS `oneshot_pages`, `oneshot_blocks`;
        CREATE TABLE `oneshot_pages` (
            `page_id` BIGINT UNSIGNED NOT NULL,
            PRIMARY KEY (`page_id`)
        ) ENGINE='InnoDB' DEFAULT CHARACTER SET 'UTF8';
        CREATE TABLE `oneshot_blocks` (
            `page_id` BIGINT UNSIGNED NOT NULL,
            `block_id` INT UNSIGNED NOT NULL,
            `prefix` VARCHAR(4) NOT NULL,
            `public_id` VARCHAR(20) GENERATED ALWAYS AS (CONCAT(`prefix`, '-', `page_id`, '-', `block_id`)) VIRTUAL NOT NULL,
            PRIMARY KEY (`page_id`, `block_id`)
        ) ENGINE='InnoDB' DEFAULT CHARACTER SET 'UTF8';
    ],
        undef,
        1
    );
}

sub check_affected_pages_and_blocks {
    my ($args) = @_;

    my $start = gettimeofday;

    local @result_out;
    local @result_err;

    my $pages = '';

    my $command =
      q[SELECT count(DISTINCT `public_id`) as `block count`, GROUP_CONCAT(DISTINCT `public_id`) FROM `oneshot_blocks`];
    my $result = process_sql_command($command, 'Blocks affected');
    unless ($result->{error}) {
        if ($result->{result} =~ /Empty set|NULL/) {
            pop @result_out;
        }
    }

    $command =
q[SELECT count(DISTINCT `page_id`) as `page count` ,GROUP_CONCAT(DISTINCT `page_id`) AS `page list` FROM `oneshot_pages`];
    $result = process_sql_command($command, 'Pages affected');
    unless ($result->{error}) {
        if ($result->{result} =~ /Empty set|NULL/) {
            $result->{error} = 1;
            $result->{color} = 'red';
            pop @result_out;
            push @result_err, $result;
        } elsif ($args->{resend_pages} || $args->{validate}) {
            my @rows = split /\n/, $result->{result};
            my @cols = split /\t/, $rows[2];
            $pages = $cols[1];
        }
    }

    $command = q[
        SELECT
            GROUP_CONCAT(DISTINCT `page_id`) AS `page list`
        FROM
            `oneshot_blocks`
        LEFT JOIN
            `oneshot_pages` USING(`page_id`)
        WHERE
            `oneshot_pages`.`page_id` IS NULL
        ];
    $result = process_sql_command($command, 'Missed pages');
    unless ($result->{error}) {
        pop @result_out;
        if ($result->{result} !~ /Empty set|NULL/) {
            $result->{error} = 1;
            $result->{color} = 'red';
            push @result_err, $result;
        }
    }

    my $out = _format_out(\@result_out, \@result_err);

    my $done = gettimeofday;
    my $time_stat_str = sprintf("Executing:  %s\n", _sec_to_time($done - $start));

    my $is_error = @result_err > 0;
    add_result_to_ticket(undef, 'Затронутые пейджи и блоки', $is_error, $out, $time_stat_str);

    push @{$timer->{actions}},
      {
        name   => 'Get affected pages',
        start  => $start,
        end    => scalar(gettimeofday()),
        status => $is_error ? 'FAIL' : 'OK',
      };

    return $pages;
}

sub validate_pages {
    my ($page_list) = @_;

    my $start = gettimeofday;

    _add_comment('Start validation');

    (undef, my $validator_out_log) = get_log_file_path('validator');
    (undef, my $validator_err_log) = get_log_file_path('validator', 'err');

    my $command =
"$partner2_path/bin/validator.pl --file_path=$page_list --split=$RESEND_OPTION_SPLIT --result=$validator_err_log --timeout=5 --tries=10 2>&1 > $validator_out_log";

    my $result = process_command($command, "Run validator $command", 'shell');

    my $is_error = $result->{error} || $result->{code};

    # запускаем валидацию ртб блоков в java
    (undef, my $java_validator_log) = get_log_file_path('validator_java_log', 'json');
    my $dir          = $partner2_path;
    my $extra_args   = "--split=$RESEND_OPTION_SPLIT --filePath=$page_list";
    my $java_command = "$dir/bin/java_start.sh -o $java_validator_log -- --validator $extra_args";
    my $java_code    = system("/bin/bash $java_command");
    my $json_text    = readfile($java_validator_log);
    my @logs         = map {from_json($_)} split("\n", $json_text);
    @logs = grep($_->{level} eq "ERROR", @logs);

    if (@logs || $java_code) {
        $is_error = 1;
        writefile($validator_err_log, join("\n", map {$_->{'message'}} @logs), append => TRUE);
    }

    my $done = gettimeofday;
    my $time_stat_str = sprintf("Executing:  %s\n", _sec_to_time($done - $start));

    add_result_to_ticket(
        $is_error ? $validator_err_log : '',
        'Результат валидации',
        $is_error, undef, $time_stat_str
    );

    push @{$timer->{actions}},
      {
        name     => 'Validate pages',
        start    => $start,
        end      => scalar(gettimeofday()),
        log_file => $is_error ? $validator_err_log : $validator_out_log,
        status   => $is_error ? 'FAIL' : 'OK'
      };

    return 1;
}

sub resend_pages {
    my ($arg_resend_pages, $page_list) = @_;

    my $start = gettimeofday;

    _add_comment('Start resend pages');

    my $result_out;
    my $result_err;
    (undef, my $resend_log) = get_log_file_path('resend_to_bk',);
    (undef, my $resend_failed_log) = get_log_file_path('resend_to_bk', 'err');

    my $command =
"$partner2_path/bin/resend_to_bk.pl --file_path=$page_list --split=$RESEND_OPTION_SPLIT --log_failed=$resend_failed_log --timeout=5 --tries=10 ";

    if ($arg_resend_pages eq 'java') {
        $command .= " --from_java --oneshot=$STAGE";
    }

    $command .= " 2>&1 > $resend_log";

    my $result = process_command($command, "Resend to BK", 'shell');
    my $is_error = $result->{error} || $result->{code};

    my $done = gettimeofday;
    my $time_stat_str = sprintf("Executing:  %s\n", _sec_to_time($done - $start));

    add_result_to_ticket(
        $is_error ? $resend_failed_log : '',
        'Результат переотправки пейджей в БК',
        $is_error, undef, $time_stat_str
    );

    push @{$timer->{actions}},
      {
        name     => 'Resend pages',
        start    => $start,
        end      => scalar(gettimeofday()),
        log_file => $is_error ? $resend_failed_log : $resend_log,
        status   => $is_error ? 'FAIL' : 'OK'
      };

    return 1;
}

sub add_result_to_ticket {
    my ($filename, $message, $is_error, $body, $time_stat_str) = @_;

    return if $is_skip_comment_ticket;

    my $comment = sprintf("%d/%d. %s", ++$STEPS_COUNT, $STEPS_TOTAL, $message);

    my $link = _upload_file_and_get_link($filename) if $filename && !$is_skip_ya_upload;

    $comment .= $is_error ? " - !!FAIL!!" : ' - !!(green)OK!!';
    $comment .= "\n$link (%%$filename%%)" if $link;

    my $st       = _get_yandex_startrek();
    my $issue    = $st->get_issue($TICKET);
    my $assignee = $issue->{assignee}{id};

    if ($body) {
        utf8::decode($body);
        $comment .= "\n\n<{Body\n$body\n}>\n";
    }
    $comment .= "\n" . '%%' . $time_stat_str . '%%' if $time_stat_str;

    INFOF 'Add comment to ticket %s', $TICKET;
    $st->add_comment($TICKET, $comment, $assignee);
}

sub _upload_file_and_get_link {
    my ($filename) = @_;

    my $link;
    if (-e $filename) {
        my $token   = get_secret('sandbox-token');
        my $command = "ya upload $filename --token=$token --ttl=inf --json-output --owner=PARTNER";
        my $result  = process_command($command, 'ya upload', 'shell', 1);

        my $upload_result = from_json($result->{result});
        $link = $upload_result->{download_link};
        INFOF 'Log uploaded - %s', $link;
    } else {
        ERRORF 'File doesnt exists "%s"', $filename;
    }

    return $link;
}

sub quote_out {
    my ($str) = @_;
    $str =~ s/%%/~%%/g;
    return $str;
}

sub process_sql_command {
    my ($command, $title) = @_;

    return process_command($command, $title, 'sql');
}

sub process_command {
    my ($command, $title, $type, $dont_print_cmd) = @_;

    $type //= 'sql';
    my $result = {
        title => ($title || 'SQL'),
        color => 'green',
        type  => $type,
        command    => $command,
        startTime  => time(),
        finishTime => 0,
        elapsed    => '00:00:00',
    };

    INFOF 'Run command "%s"', $command;
    my ($std_out, $std_err, $code) = (
        $type eq 'sql'
        ? run_sql_command($command, $dont_print_cmd)
        : run_command($command, $dont_print_cmd)
    );
    $result->{code}       = $code;
    $result->{finishTime} = time();
    $result->{elapsed}    = _sec_to_time($result->{finishTime} - $result->{startTime});

    if ($code) {
        $result->{color}  = 'red';
        $result->{result} = $std_err // 'unknown error';
        $result->{error}  = 1;
        push @result_err, $result;
    } else {
        $std_out //= '';
        if ($type eq 'sql') {
            my ($res, $warn) = split_result($std_out);
            if ($std_out =~ /\S/) {
                $result->{result}  = $res;
                $result->{warning} = $warn;
                if ($warn) {
                    $result->{color} = 'yellow';
                }
                push @result_out, $result;
            }
        } else {
            $result->{result} = $std_out;
        }
    }

    return $result;
}

sub run_sql_command {
    my ($command, $without_warnings, $dont_print_cmd) = @_;

    # экранируем спец-символы bash-а: " ` $ \
    $command =~ s/(["`\$\\])/\\$1/g;
    $command .= '; show warnings' unless $without_warnings;

    my $mysql_cmd = get_sql_command();

    return run_command(qq[$mysql_cmd -e "$command"], $dont_print_cmd);
}

sub _buid_java {
    my ($java_branch, $is_skip_build_java) = @_;

    my $start = gettimeofday;

    my $java_commit;
    unless ($java_branch) {
        _add_comment('Getting Java prod version');
        my $prod_arc_commit = _get_prod_version('j');
        _add_comment('Got java arc commit %s', $prod_arc_commit);

        ($java_commit) = $prod_arc_commit =~ /^\d+\.(\w+)-\d+$/;
    }

    _add_comment('Start build java form %s=%s', $java_branch ? 'branch' : 'commit', $java_branch // $java_commit);

    my ($code, $std, $err, $build_java_log, $build_java_err_log) =
      run_java_build($java_branch, $partner2_path, 2, $java_commit, $is_skip_build_java);

    if ($code) {
        push @result_err, ['ERROR checkout java', ($java_branch // $java_commit)];
        push @result_err, ['ERROR checkout java', $err];
    }

    push @{$timer->{actions}},
      {
        name     => 'Build java',
        start    => $start,
        end      => scalar(gettimeofday()),
        log_file => $code ? $build_java_err_log : $build_java_log,
        status   => $code ? 'FAIL' : 'OK'
      };

    return 1;
}

sub run_java_build {
    my ($java_branch, $path, $tries, $arc_commit, $is_skip_build_java) = @_;

    (undef, my $build_java_log) = get_log_file_path('build_java');
    (undef, my $build_java_err_log) = get_log_file_path('build_java', 'err');

    my $branch_or_commit_arg = $arc_commit ? "--commit=$arc_commit" : "--branch=$java_branch";
    my $command =
"$path/bin/build_java.pl --rebuild $branch_or_commit_arg --app=runner --app=jsonapi 1>$build_java_log 2>$build_java_err_log";

    _add_comment($command);

    my $code = system($command) unless $is_skip_build_java;

    while ($code && $tries) {
        $code = system($command) unless $is_skip_build_java;
        $tries = $tries - 1;
    }
    my $out = readfile($build_java_log);
    my $err = readfile($build_java_err_log);

    return ($code, $out, $err, $build_java_log, $build_java_err_log);
}

sub run_command {
    my ($command, $dont_print_cmd) = @_;

    _add_comment('   run command "%s"', $command) unless $dont_print_cmd;

    local *CHLD_OUT;
    local *CHLD_ERR;
    my $std_out = '';
    my $std_err = '';
    my $pid     = open3(undef, \*CHLD_OUT, \*CHLD_ERR, $command);
    binmode(CHLD_OUT, ':utf8');
    binmode(CHLD_ERR, ':utf8');
    while (defined(my $out = <CHLD_OUT>) or defined(my $err = <CHLD_ERR>)) {
        $std_out .= $out if defined $out;
        $std_err .= $err if defined $err;
    }
    waitpid($pid, 0);
    my $code = $? >> 8;

    return ($std_out, $std_err, $code);
}

sub get_lock {
    my ($result_err, $ticket) = @_;

    my $lock_fh;
    if (open $lock_fh, '>', $LOCK_FILE) {
        unless (flock $lock_fh, LOCK_EX | LOCK_NB) {
            if ($! == EAGAIN) {
                push @result_err, ['ERROR lock', "another oneshot running"];
            } else {
                push @result_err, ['ERROR lock', "cannot get lock on ($LOCK_FILE):" . $!];
            }
            undef $lock_fh;
        } else {
            print $lock_fh sprintf("Ticket: %s\nPID: %s\n", $ticket // '', $$);
        }
    } else {
        push @result_err, ['ERROR lock', "cannot open lockfile($LOCK_FILE):" . $!];
    }

    return $lock_fh;
}

sub free_lock {
    my ($lock_fh) = shift;

    if ($lock_fh) {
        open $lock_fh, '>', $LOCK_FILE;
        print $lock_fh '-';

        flock $lock_fh, LOCK_UN;
        close $lock_fh;
    }
}

sub split_result {
    my ($str) = @_;
    my @rows = split /-{14}\n/, $str // '';

    my $warn = $rows[4] // '';
    $warn =~ s/\s*Bye\s*//       if $warn;
    $warn =~ s/\s*Empty set\s*// if $warn;

    return ($rows[2] // '', $warn);
}

sub to_table {
    my ($str) = @_;
    my $res = '';
    $str =~ s/^\s+//g;
    for my $row (split /\n/, $str) {
        $row =~ s/\t/|/g;
        $row =~ s/\|(?=\|)/\| /g;
        $res .= "||$row||\n";
    }
    return "#|\n" . $res . "|#\n";
}

sub _get_yandex_startrek {

    my $ys = Yandex::StarTrek->new(
        oauth_token => get_secret('startrek-token'),
        retry       => 3,
        delay       => 3,
    );

    return $ys;
}

sub infra_open_event {
    my ($env) = @_;

    my $start = gettimeofday;

    my ($json) = infra_get_response(
        method       => 'POST',
        base         => 'https://infra-api.yandex-team.ru/v1',
        url          => '/events',
        token        => get_secret('infra-token'),
        content_data => {
            title         => 'Oneshot',
            description   => sprintf('Oneshot for %s on %s', $TICKET, $env),
            startTime     => time(),
            finishTime    => time() + $INFRA_ESTIMATE,
            type          => 'issue',
            severity      => 'minor',
            serviceId     => $INFRA_SERVICE_ID,
            environmentId => $INFRA_ENV{$env},
            tickets       => $TICKET,
        },
    );

    push @{$timer->{actions}},
      {
        name   => 'Open infra event',
        start  => $start,
        end    => scalar(gettimeofday()),
        status => 'OK'
      };

    return $json->{id};
}

sub infra_close_event {
    my ($id) = @_;

    infra_get_response(
        method       => 'PUT',
        base         => 'https://infra-api.yandex-team.ru/v1',
        url          => '/events/' . $id,
        token        => get_secret('infra-token'),
        content_data => {finishTime => time(),},
    );
}

sub infra_get_response {
    my (%opts) = @_;

    my $lwp = LWP::UserAgent->new(ssl_opts => {verify_hostname => 0, SSL_verify_mode => SSL_VERIFY_NONE});
    my $request = HTTP::Request->new(
        $opts{method} => $opts{base} . $opts{url},
        [
            'Authorization' => 'OAuth ' . $opts{token},
            'Content-Type'  => 'application/json',
        ],
        $opts{content_data} ? to_json($opts{content_data}) : ()
    );

    my $retry = 3;
    my $response;
    while (1) {
        $response = $lwp->request($request);
        last if $response->is_success || --$retry == 0;
        sleep(3);
    }

    my $content = $response->decoded_content // 'No content';

    my $json;
    if ($response->is_success) {
        $json = eval {from_json($content)};
        warn $@ if $@;
    }

    utf8::decode($content) unless utf8::is_utf8($content);

    return ($json, $response->code, $content);
}

sub make_queue_task {
    my ($args, $data) = @_;

    try {
        $data = from_json($data);
    }
    catch {
        my ($e) = @_;
        push @result_err, ["ERROR", $e->message];
    };

    if (@result_err) {
        return (\@result_out, \@result_err);
    } else {
        $data->{params}{ticket} = $TICKET;
        my $p = to_json($data, pretty => TRUE);
        $p =~ s/\\/\\\\/g;
        $p =~ s/'/\\'/g;
        my $cmd = "
my \$data = from_json('$p');
my \$result = \$app->support->process_from_json(\$data);
";
        $cmd .= "
\$app->support->fill_tables(\$result);
" if $args->{check_pages};

        return make_perl($cmd);
    }
}

sub _cleanup_old_logs {
    my $command = 'find ./oneshot_artifacts/*  -maxdepth 1  -type d  -mtime +90  -prune  -exec rm -rf {} \; || true';
    run_command($command);
}

sub _get_args {
    my ($cmd_args_str, $self_script_name) = @_;

    my $args = {file => $self_script_name};

    Getopt::Long::GetOptionsFromString(
        $cmd_args_str,
        'ticket=s'             => \$args->{ticket},
        'stage=s'              => \$args->{stage},
        'path=s'               => \$partner2_path,
        'type=s'               => \$args->{type},
        'branch=s'             => \$args->{branch},
        'jb=s'                 => \$args->{jb},
        'format!'              => \$format,
        'check_pages!'         => \$args->{check_pages},
        'validate!'            => \$args->{validate},
        'resend_pages=s'       => \$args->{resend_pages},
        'single!'              => \$args->{single},
        'nodb!'                => \$args->{nodb},
        'skip-infra-event'     => \$args->{'skip-infra-event'},
        'skip-checkout!'       => \$args->{'skip-checkout'},
        'skip-build-java!'     => \$args->{'skip-build-java'},
        'skip-start-java!'     => \$args->{'skip-start-java'},
        'skip-ya-upload!'      => \$is_skip_ya_upload,
        'skip-comment-ticket!' => \$is_skip_comment_ticket,
        'force-resend-pages!'  => \$args->{'force-resend-pages'},
        'force-java-jsonapi!'  => \$args->{'force-java-jsonapi'},
        'help|?!'              => \$is_help_mode,
    ) or pod2usage(2);

    pod2usage(1) if $is_help_mode;

    warn "Undefined option: type\n" and pod2usage(2) unless $args->{type};
    warn "Invalid value option: type\n" and pod2usage(2) unless $oneshot_types->{$args->{type}};
    die "Invalid call: need <filename> second args\n" unless $args->{file};

    $args->{check_pages}  //= 1;
    $args->{resend_pages} //= 'java';
    $args->{validate}     //= 1;

    # NOTE! здесь проверяется наличие файлика "production" (который накатывается плейбуком pi-oneshot-runner)
    unless ($args->{stage}) {
        for (keys %stages) {
            if (-e "$ROOT/$_") {
                $args->{stage} = $_;
                last;
            }
        }
        $args->{stage} //= $DEFAULT_STAGE;
    }

    return $args;
}

sub _sec_to_time {
    my ($sec) = @_;
    return sprintf("%02d:%02d:%02d", $sec / 3600, $sec / 60 % 60, $sec % 60);
}

sub _time_to_str {
    my ($time) = @_;
    my @t = localtime($time);
    return sprintf("%04d-%02d-%02d %02d:%02d:%02d", $t[5] + 1900, $t[4] + 1, $t[3], $t[2], $t[1], $t[0]);
}

sub _kill_java_jsonapi {
    if ($jsonapi_start_invoked) {
        _add_comment('Killing java jsonapi');
        system("$partner2_path/bin/deploy/kill_process.pl --pid_file=$jsonapi_pid_file --wait");
    }
}
