#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для переотправки пейджей в БК.

=head1 USAGE

  perl bin/resend_to_bk.pl --page_ids=12345,54321,..  2>&1 | tee resend_to_bk.log
  perl bin/resend_to_bk.pl --file_path=pageids.txt    2>&1 | tee resend_to_bk.log

=head1 OPTIONS

  page_ids        - Список ID площадок через запятую (необязательный)
  file_path       - Путь к файлу со списком площадок (разделитель перевод строки или запятая)
  send_deleted    - Позволяет отправлять удаленные площадки
  send_rejected   - Позволяет отправлять отклоненные площадки
  send_protected  - Позволяет отправлять защищенные площадки площадки
  split           - Разбить задание на указанное количество потоков
  log_failed      - имя файла, для хранения неотправленных pages
  age             - datetime, посылать только отправлявшиеся до указанной даты
  protobuf_only   - отослать javа-ой protobuf в LB топик bk_proto_page_update
  logbroker_only  - отослать perl-ом json в LB топик bk_page_update
                    (+ protobuf если выставлен 'java_transport_turn_on_key' в кv_store)
  do_not_change_multistate - отсылает в БК не меняя статусы (частичный функционал logbroker_only)
  timeout         - Сколько ждать перед повторным запуском в случае ошибки
  tries           - Количество попыток в случае неудачи
  from_java       - Собирать bkdata с участием java jsonapi
  oneshot         - сигнализирует приложению о запуске в контексте ваншота со стейджем

=cut

# В интерфейсе ПИ2 есть форма для переотправки пейджей в БК - https://partner2.yandex.ru/queue/update_in_bk
# Но если нужно переотправить больше сотни пейджей, то лучше использовать скрипт - с помощью скрипта удобнее прорабатывать ошибки, которые возникли пре отправке.
#
# Для того чтобы переотправить нужно:
#
#  * получить тикет на переотправку
#  * Либо передавать список пейджей списком в парметре ---page_ids, либо путь к файлу с пейджами в параметре --file_path=...
#  * запустить команду `perl bin/resend.pl --page_ids=... 2>&1 | tee resend_to_bk.log`
#  * после того как скрипт отработал посмотреть на ошибки `cat resend_to_bk.log | grep ERROR`
#  * отписать все в тикет

use lib::abs-soft => qw(../lib ../local/lib/perl5);

use Pod::Usage;

use qbit;
use Utils::ScriptWrapper 'util';
use Utils::Oneshots::ResendToBk;

run(\&Utils::Oneshots::ResendToBk::main);

sub prepare_args {
    my ($opts) = @_;

    my @page_ids;
    unless ($opts->{page_ids} || $opts->{file_path}) {
        die "You must specify 'page_ids' or 'file_path'\n";
    } elsif ($opts->{file_path}) {
        unless (-r $opts->{file_path}) {
            die "Wrong file path '$opts->{file_path}'\n";
        } else {
            open(my $FH, "<", $opts->{file_path});
            while (my $page_id = <$FH>) {
                push @page_ids, grep {$_} split /\s*,\s*/, $page_id;
            }
            close $FH;
        }
    } elsif ($opts->{page_ids}) {
        @page_ids = split /,/, $opts->{page_ids};
    }

    $opts->{allow_special} = $opts->{is_send_deleted} || $opts->{is_send_rejected} || $opts->{is_send_protected};
    $opts->{special_flags} = {
        rejected  => $opts->{is_send_rejected},
        deleted   => $opts->{is_send_deleted},
        protected => $opts->{is_send_protected}
    };

    $opts->{page_ids} = \@page_ids;
}

sub args {
    my ($opts) = @_;
    return (
        'log_failed:s'              => \$opts->{log_failed},
        'page_ids:s'                => \$opts->{page_ids},
        'file_path:s'               => \$opts->{file_path},
        'send_deleted!'             => \$opts->{is_send_deleted},
        'send_rejected!'            => \$opts->{is_send_rejected},
        'send_protected!'           => \$opts->{is_send_protected},
        'age:s'                     => \$opts->{age},
        'logbroker_only!'           => \$opts->{logbroker_only},
        'protobuf_only!'            => \$opts->{protobuf_only},
        'do_not_change_multistate!' => \$opts->{do_not_change_multistate},
        'timeout:i'                 => \$opts->{timeout},
        'tries:i'                   => \$opts->{tries},
        'from_java!'                => \$opts->{from_java},
        'oneshot:s'                 => \$opts->{oneshot},
    );
}

sub key_data_for_split {
    my ($opts) = @_;
    return 'page_ids';
}

sub prepare_app {
    my ($app, $opts) = @_;

    $app->set_option('oneshot', $opts->{oneshot} // 'dev');
    $app->api_bk->{'__SOAP__'}->proxy->timeout(7200);
}
