#!/usr/bin/perl

=encoding utf8

=head1 DESCRIPTION

Cкрипт принимает параметры запроса в RestApi и выполняет его как если бы это был настоящий вызов API

=head1 PARAMS

 token            - токен к RESTAPI (обязателен, для fakelogin нужен токен менеджера)
 fakelogin        - логин под которым делать запрос
 X                - метод GET|POST|PATCH|DELETE
 query_string     - GET параметры запроса (если хочется передавать отдельно)
 data             - json параметров запроса
 format           - [json; pretty; dumper;] ("json" is default)
 show_cmd         - показывать команду
 no_lazy          - не использовать ленивую загрузку
 debug_exceptions - выводить все exceptions

=head1 USAGE

 ./bin/restapi_call.pl --token=$TOKEN  --fakelogin=vsheff  '/v1/bkdata/R-A-21503-1' --debug_exceptions
 ./bin/restapi_call.pl --token=$TOKEN         '/v1/context_on_site_rtb/defaults?fields=direct_block&attributes={"page_id":"60067","site_version":"desktop"}'
 ./bin/restapi_call.pl --token=$TOKEN         '/v1/context_on_site_rtb/defaults --query_string='fields=direct_block&attributes={"page_id":"60067","site_version":"desktop"}'
 ./bin/restapi_call.pl --token=$TOKEN -X POST '/v1/context_on_site_campaign' --data '{"data":{"type":"context_on_site_campaign","attributes":{"login":"autoru-manager","domain_id":3,"caption":"Test"}}}'

=cut

use lib::abs qw( ../lib );

use qbit;

use Pod::Usage;
use Getopt::Long qw();
use URI::Escape qw();
use Utils::Safe;

use feature 'say';

####
_run();
#####

sub _run {

    my (
        $token,  $fakelogin,    $resource,         $query_string, $data, $method,
        $format, $with_headers, $debug_exceptions, $show_cmd,     $is_no_lazy_load
       ) = _get_args();

    $ENV{'LAZY_LOAD'} = $is_no_lazy_load ? FALSE : TRUE;

    my $additional_headers = '';
    $additional_headers .= sprintf('CONTENT_LENGTH=%s',          length($data)) if $data;
    $additional_headers .= sprintf(q[ QUERY_STRING='%s'],        $query_string) if $query_string;
    $additional_headers .= sprintf(q[ HTTP_COOKIE=fakelogin=%s], $fakelogin)    if $fakelogin;
    $additional_headers .= ' DEBUG_EXCEPTIONS=1  FORCE_LOGGER_TO_SCREEN=1' if $debug_exceptions;

    confirm_connect_on_production();
    my $cmd = sprintf q[
        %s
        HTTP_AUTHORIZATION="token %s"
        HTTP_ACCEPT='application/vnd.api+json'
        CONTENT_TYPE='application/vnd.api+json'

        HTTP_X_REAL_IP='5.255.228.81'
        HTTP_HOST='jsonapi.dev-partner2.yandex.com'

        REQUEST_METHOD='%s'
        PATH_INFO='%s'

        %s

        perl -I. -I./lib -I./local/lib/perl5 ./bin/rest_api.psgi cgi
     ], ($data ? sprintf(q[echo '%s' | ], $data) : ''), $token, uc($method), $resource, $additional_headers;

    $cmd =~ s|\n\s+|  |g;
    $cmd =~ s|^\s+||g;
    $cmd =~ s|\s+$||g;

    if ($show_cmd) {
        (my $cmd2 = $cmd) =~ s/token [^ "']+/token XXX/;
        say $cmd2 ;
    }

    my $out = `$cmd`;

    my ($header, $body) = split(/\r?\n\r?\n/, $out, 2);

    unless ($body) {
        $body = $header // '';
        $header = '';
    }

    say $header if $with_headers;

    if ($format eq 'json') {
        say $body;
    } elsif ($format eq 'pretty') {
        my $data = from_json($body);
        say to_json($data, pretty => TRUE);
    } elsif ($format eq 'dumper') {
        my $data = from_json($body);
        no warnings;
        local $Data::Dumper::Indent     = 2;
        local $Data::Dumper::Sortkeys   = 1;
        local $Data::Dumper::Terse      = 1;
        local $Data::Dumper::Sparseseen = 1;
        print Data::Dumper::Dumper($data);
        print STDERR "#END\n";
    }

}

sub _get_args {

    my $token        = undef;
    my $fakelogin    = undef;
    my $method       = 'GET';
    my $query_string = '';
    my $data         = '';

    my $format           = 'json';
    my $with_headers     = 0;
    my $debug_exceptions = 0;
    my $show_cmd         = 0;
    my $is_no_lazy_load  = 0;

    my $help = 0;
    Getopt::Long::GetOptions(
        #--- Obligatory
        'token=s' => \$token,
        #--- Optional
        'fakelogin:s'    => \$fakelogin,
        'X:s'            => \$method,
        'query_string:s' => \$query_string,
        'data:s'         => \$data,

        'format:s'         => \$format,
        'with_headers!'    => \$with_headers,
        'debug_exceptions' => \$debug_exceptions,
        'show_cmd!'        => \$show_cmd,
        'no_lazy!'         => \$is_no_lazy_load,

        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    my $args = shift @ARGV;
    pod2usage(-verbose => 2, -noperldoc => 1) if $help || !defined $args;

    my ($resource, $query_string2) = split /\?/, $args, 2;
    $query_string = join '&', map {join '=', $_->[0], URI::Escape::uri_escape($_->[1])}
      map {[split /=/, $_]} map {split /&/} grep {$_} $query_string, $query_string2;
    $query_string =~ s/%22/"/g;

    #-- Проверяем зн-ия входных параметров
    my $errors = [];
    push @$errors, 'Wrong resource' unless $resource =~ m|^/v1/|;
    push @$errors, 'Extra data' if @ARGV;

    push @$errors, 'Token required' unless $token;

    if ($method && $method !~ /^(GET|POST|PATCH|DELETE)$/) {
        push @$errors, 'Unknown method';
    }

    unless (grep {$format eq $_} qw( dumper  json  pretty)) {
        push @$errors, "Wrong format $format";
    }

    if (@$errors) {
        print join("\n", @$errors), "\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    return (
        $token,  $fakelogin,    $resource,         $query_string, $data, $method,
        $format, $with_headers, $debug_exceptions, $show_cmd,     $is_no_lazy_load
    );
}
