#!/usr/bin/perl

=encoding utf8

=head1 DESCRIPTION

Cкрипт принимает параметры запроса в Rosetta сервер и выполняет его как если бы это был настоящий вызов с фронта

=head1 PARAMS

 request     - JSON запрос целиком
 model       - либо по отдельности: модель
 method      - метод
 args        - массив параметров или
 opts        - хэш параметров
 login       - логин пользователя
 format      - [json; dumper; json_pretty] (default json_pretty)
 coro/nocoro - принудительно включать или выключать флаг use_coro в конфиге приложения
 no_lazy     - не использовать ленивую загрузку

=head1 USAGE

 rosetta_call --request='{"model":"rbac","method":"get_roles_by_user_id","args":[155664066]}'
 rosetta_call --model=rbac --method=get_roles_by_user_id --args='[155664066]'
 rosetta_call --model=rbac --method=get_roles_rights     --opts='{"fields":["right"],"role_id":[2,17,18,19,22,30]}'

=cut

use strict;
use warnings;

use lib::abs-soft => qw(../lib ./lib);

use qbit;
use Rosetta;
use PiConstants qw($AUTHORIZATION_USER_FIELDS);

use Pod::Usage;
use Getopt::Long qw();
use Utils::Safe;

use feature 'say';

####
_run();
#####

sub _run {

    my ($login, $request, $format, $is_use_coro, $is_no_lazy_load) = _get_args();

    confirm_connect_on_production();
    $ENV{'LAZY_LOAD'} = $is_no_lazy_load ? FALSE : TRUE;

    my $app = _get_app($login);

    $app->request(['call', $request]);

    if (defined($is_use_coro)) {
        if ($is_use_coro) {
            $app->set_option(use_coro => 1);
        } else {
            $app->set_option(use_coro => 0);
        }
    }

    my $res = $app->call(
        model      => $request->{model},
        method     => $request->{method},
        args       => $request->{args} // [%{$request->{opts}}],
        request_id => $request->{request_id},
    );

    $app->post_run();

    if ($format eq 'json_pretty') {
        say to_json($res, pretty => TRUE);
    } elsif ($format eq 'dumper') {
        no warnings;
        local $Data::Dumper::Indent     = 2;
        local $Data::Dumper::Sortkeys   = 1;
        local $Data::Dumper::Terse      = 1;
        local $Data::Dumper::Sparseseen = 1;
        print Data::Dumper::Dumper($res);
    } else {
        say to_json($res);
    }

    print STDERR "#END\n" unless $format =~ /^json/;
}

sub _get_app {
    my ($login) = @_;

    my $app = Rosetta->new();

    $app->pre_run();

    _login_as_user(
        app   => $app,
        login => $login,
    );

    return $app;
}

sub _login_as_user {
    my (%opts) = @_;

    my $app   = delete $opts{app};
    my $login = delete $opts{login};

    my $user = $app->partner_db->_get_all('select id from users where login = ?', $login,);

    if (@$user == 1) {
        my $user_id = $user->[0]->{id};

        my $tmp_rights =
          $app->add_tmp_rights(qw(users_view_all users_view_field_multistate users_view_field_client_id));

        my $cur_user = $app->users->get($user_id, fields => $AUTHORIZATION_USER_FIELDS);

        # have no blackbox here to get a real value
        # instead doing like in fake_login
        $cur_user->{display_login} = $cur_user->{login};

        # to be removed when adfox checking goes into set_cur_user
        my $user_adfox = $app->partner_db->user_adfox->get_all(
            fields => ['adfox_id'],
            filter => $app->partner_db->filter({user_id => $cur_user->{id}}),
            limit  => 1
        );

        $cur_user->{has_adfox} = (@$user_adfox && $user_adfox->[0]->{adfox_id}) ? 1 : 0;

        $app->set_cur_user($cur_user);
    } else {
        throw sprintf('Can\'t find login %s in database', $login);
    }
}

sub _get_args {

    my $json        = '';
    my $model       = '';
    my $method      = '';
    my $args        = undef;
    my $opts        = undef;
    my $login       = 'yndx-bessarabov';
    my $format      = 'json_pretty';
    my $is_use_coro = undef;

    my $is_no_lazy_load = 0;

    my $help = 0;
    Getopt::Long::GetOptions(
        #--- Obligatory
        'request:s' => \$json,
        'model:s'   => \$model,
        'method:s'  => \$method,
        'args:s'    => \$args,
        'opts:s'    => \$opts,
        'login:s'   => \$login,
        'format=s'  => \$format,
        'no_lazy!'  => \$is_no_lazy_load,
        'coro!'     => \$is_use_coro,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    #-- Проверяем зн-ия входных параметров
    my $errors = [];
    if (!$json && !$model && !$method && !$args && !$opts) {
        push @$errors, 'request is empty';
    }

    unless (grep {$format eq $_} qw( dumper  json  json_pretty)) {
        push @$errors, "Wrong format $format";
    }

    unless ($json) {
        push @$errors, 'model  is empty' unless $model;
        push @$errors, 'method is empty' unless $method;

        if (!$args && !$opts) {
            push @$errors, 'You must specify "args" or "opts"';
        }
    }

    if (@$errors) {
        print join("\n", @$errors), "\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    unless ($json) {
        my $v = $args // $opts;
        utf8::decode($v);
        $json = sprintf '{"model":"%s","method":"%s","%s":%s}', $model, $method, ($args ? 'args' : 'opts'), $v;
    }

    my $request = from_json($json);

    return ($login, $request, $format, $is_use_coro, $is_no_lazy_load);
}
