#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт выставления CPM для брендов на блоках (#PI-9730)

=head1 USAGE

  perl bin/set_brands_cpm.pl --block_public_ids_file_path=./block_ids.txt --brand_ids_file_path=./bids.txt --cpm=130

=head1 OPTIONS

  block_public_ids_file_path - Путь к файлу со список Public ID блоков (разделитель запятая или перенос строк)
  brand_ids_file_path        - Путь к файлу со список BID (разделитель запятая или перенос строк)

=cut

use strict;
use warnings;

use lib::abs qw(
  ../lib
  );

# NOTE! потому что модели frontend нет в Application, а там нужен метод get_model_name_from_public_id
use Rosetta;

# common modules
use Carp;
use File::Slurp;
use Pod::Usage;
use Getopt::Long qw();
use Time::HiRes qw(gettimeofday tv_interval);
use Term::ANSIColor qw(colored);

use Exception::Validation::BadArguments;
use qbit qw( TRUE from_json  to_json );

# global vars
my $RIGHTS;

main();

# main
sub main {

    my $t0 = [gettimeofday()];

    my $app = Rosetta->new();
    $app->pre_run();

    $app->set_cur_user({id => 0, login => 'system-cron'});

    no strict 'refs';
    no warnings 'redefine';
    local *{'QBit::Application::check_rights'} = sub {TRUE};

    my ($public_ids, $brand_ids, $cpm) = _get_args();

    my $page_blocks = _resolve_public_ids($app, $public_ids);

    my $total_pages = 0;
    map {$total_pages += keys(%$_)} values %$page_blocks;

    print "START sending $total_pages pages\n";

    my $i = 1;
    foreach my $page_accessor (sort keys %$page_blocks) {

        foreach my $page_id (sort {$a <=> $b} keys %{$page_blocks->{$page_accessor}}) {

            my $t1 = [gettimeofday()];

            printf "    %d/%d - %s\n", $i++, $total_pages, colored($page_id, 'yellow');

            my $page_internal_id = undef;
            {
                my $found_pages = $app->all_pages->get_all(fields => ['id'], filter => {page_id => $page_id});
                if (!@$found_pages) {
                    throw Exception gettext('Page with Page ID %s not found', $page_id);
                } else {
                    $page_internal_id = $found_pages->[0]->{id};
                }
            }

            my $error = '';
            my $is_page_changed;
            my $j = 1;
            foreach my $block_accessor (sort keys %{$page_blocks->{$page_accessor}->{$page_id}}) {

                my $page_id_field_name = $app->$block_accessor->get_page_id_field_name();

                foreach my $block_id (sort {$a <=> $b} @{$page_blocks->{$page_accessor}->{$page_id}->{$block_accessor}})
                {

                    my $block_ids = {
                        $page_id_field_name => $page_id,
                        id                  => $block_id
                    };

                    my $public_id = $app->$block_accessor->public_id($block_ids);
                    printf "        %d. %s \n", $j++, $public_id;

                    try {

                        $app->partner_db->transaction(
                            sub {

                                my $block_brands = $app->$block_accessor->get(
                                    $public_id,
                                    fields       => ['brands'],
                                    'for_update' => TRUE
                                );

                                if (defined $block_brands) {

                                    my $brands = $block_brands->{brands} // [];

                                    my $brand_indexes = {map {$brands->[$_]->{bid} => $_} 0 .. $#$brands};

                                    my $is_changed;
                                    foreach my $brand_id_to_change (@$brand_ids) {
                                        my $index = $brand_indexes->{$brand_id_to_change};

                                        if (defined $index) {
                                            my $curr_cpm = $brands->[$index]->{cpm};
                                            if ($brands->[$index]->{blocked}) {
                                                printf "            bid=%-6s %s brand is blocked\n",
                                                  $brand_id_to_change,
                                                  colored("ERROR", 'red');
                                            } elsif ($curr_cpm != $cpm) {
                                                printf "            bid=%-15s cpm %s -> %s\n",
                                                  colored($brand_id_to_change, 'yellow'), $curr_cpm, $cpm;
                                                $brands->[$index]->{cpm} = $cpm;
                                                $is_changed = 1;
                                            }
                                        } else {
                                            printf "            bid=%-15s cpm %s\n",
                                              colored($brand_id_to_change, 'green'),
                                              $cpm;
                                            push @$brands,
                                              {
                                                bid     => $brand_id_to_change,
                                                blocked => 0,
                                                cpm     => $cpm
                                              };
                                            $is_changed = 1;
                                        }
                                    }

                                    if ($is_changed) {
                                        $is_page_changed = 1;
                                        $app->$block_accessor->brands->add(
                                            {
                                                page_id  => $block_ids->{$page_id_field_name},
                                                block_id => $block_ids->{id}
                                            },
                                            $brands
                                        );
                                    } else {
                                        print colored("\t\t nothing to change \n", 'cyan');
                                    }
                                } else {
                                    throw Exception gettext('Block %s not found', $public_id);
                                }
                            }
                        );

                    }
                    catch {
                        $error = shift->message();
                        $error =~ s/\n/ /g;
                        print colored($error, 'red');
                    };
                }
            }

            if ($is_page_changed) {
                try {

                    my $page_id_field = $app->$page_accessor->get_page_id_field_name();

                    # NOTE! Транзакция нужна чтобы выставленный 'need_update' не подхватил скрипт асинхронной отправки
                    $app->partner_db->transaction(
                        sub {
                            unless ($app->$page_accessor->isa('Application::Model::SSP')) {
                                $app->$page_accessor->do_action($page_internal_id, 'set_need_update');
                            }

                            $app->$page_accessor->update_in_bk({$page_id_field => $page_id});
                        }
                    );
                }
                catch {
                    $error = shift->message();
                    $error =~ s/\n/ /g;
                };

                my $elapsed = sprintf '%.2f', tv_interval($t1, [gettimeofday()]);
                print "        ",
                  (
                    $error
                    ? colored("ERROR (page_id=$page_id, model: $page_accessor) - $error", 'red')
                    : colored('OK',                                                       'green') . " ($elapsed sec)"
                  ),
                  "\n";
            } else {
                print colored("        not changed\n", 'magenta');
            }
        }
    }

    my $elapsed = sprintf '%.2f', tv_interval($t0, [gettimeofday()]);
    print "END ($elapsed sec)\n";

    $app->post_run();
}

sub _resolve_public_ids {
    my ($app, $public_ids) = @_;

    my $page_blocks = {
        #  <page_accessor> => [
        #      <page_id> => {
        #          <block_accessor> => [ <block_id>, ... ],
        #          ...
        #      },
        #      ...
        #   ],
        #   ...
    };
    foreach my $public_id (@$public_ids) {

        my $block_accessor = $app->frontend->get_model_name_from_public_id($public_id);
        if (ref $block_accessor eq 'HASH') {
            throw Exception::Validation::BadArguments gettext(
                "Wrong public_id $public_id. " . $block_accessor->{'error_message'});
        }

        my $page_accessor      = $app->$block_accessor->get_campaign_model_name();
        my $page_id_field_name = $app->$block_accessor->get_page_id_field_name();

        my $ids     = $app->$block_accessor->_split_id($public_id);
        my $page_id = $ids->{$page_id_field_name};

        push @{$page_blocks->{$page_accessor}->{$page_id}->{$block_accessor}}, $ids->{id};
    }

    return $page_blocks;
}

sub _get_args {

    my $public_ids_file_path = '';
    my $brand_ids_file_path  = '';
    my $cpm                  = undef;
    my $help                 = 0;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'block_public_ids_file_path:s' => \$public_ids_file_path,
        'brand_ids_file_path:s'        => \$brand_ids_file_path,
        'cpm:i'                        => \$cpm,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    #-- Проверяем зн-ия входных параметров
    my $errors = [];
    unless ($public_ids_file_path && $brand_ids_file_path && defined $cpm) {
        push @$errors, 'You must specify "block_public_ids_file_path"' unless $public_ids_file_path;
        push @$errors, 'You must specify "brand_ids_file_path"'        unless $brand_ids_file_path;
        push @$errors, 'You must specify "cpm"'                        unless defined $cpm;
    } else {
        push @$errors, "Cannot read file $public_ids_file_path" unless -r $public_ids_file_path;
        push @$errors, "Cannot read file $brand_ids_file_path"  unless -r $brand_ids_file_path;

        push @$errors, 'Wrong "cpm" format' unless $cpm =~ /^\d+(\.\d+)?$/;
    }

    my $public_ids = [];
    my $brand_ids  = [];
    unless (@$errors) {
        $public_ids = _get_file_items($public_ids_file_path);
        $brand_ids  = _get_file_items($brand_ids_file_path);
    }

    foreach my $public_id (@$public_ids) {
        unless ($public_id =~ /^[A-Z]{1,2}-[A-Z]{1,2}-\d+-\d+$/) {
            push @$errors, 'Wrong public_id="' . $public_id . '"';
        }
    }

    foreach my $brand_id (@$brand_ids) {
        unless ($brand_id =~ /^\d+$/) {
            push @$errors, 'Wrong brand_id="' . $brand_id . '"';
        }
    }

    if (@$errors) {
        print join("\n", map {colored('ERROR! ' . $_, 'red')} @$errors), "\n\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    return ($public_ids, $brand_ids, $cpm);
}

sub _get_file_items {
    my ($file_path) = @_;

    my @items = ();

    open(my $FH, "<", $file_path);
    while (my $line = <$FH>) {
        push @items, grep {$_ ne ""} map {s/^\s+//; s/\s+$//; $_} split /,/, $line;
    }
    close $FH;

    return \@items;
}
