#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для вывода возможных значений "multistate" и переходов в них
либо для генерации JSON для отрисовки графов переходов в d3

=head1 USAGE

perl bin/show_model_multistates.pl context_on_site_rtb

perl bin/show_model_multistates.pl generate_json

=head1 OPTIONS

  context_on_site_rtb - Акксессор продуктовой модели
  generate_json       - запускает генерацию JSON графоф для d3

=cut

# common modules
use feature 'say';
use Carp;
use File::Slurp qw(write_file);
use File::Path qw(make_path);
use Pod::Usage;

# project modules
use lib::abs qw(
  ../lib
  );
use qbit;
use Application;

main();

# main
sub main {

    my ($accessor, $is_make_d3_json) = get_args();

    my $app = Application->new();
    $app->pre_run();

    no strict 'refs';
    no warnings 'redefine';
    local *{'QBit::Application::check_rights'} = sub {TRUE};

    my $multistate_models = [];
    if ($accessor) {
        push @$multistate_models, $accessor;
    } elsif ($is_make_d3_json) {
        my $models = $app->get_models();
        while (my ($accessor, $package) = each(%$models)) {
            push(@$multistate_models, $accessor) if $package->isa('QBit::Application::Model::Multistate');
        }
    }

    setup_d3($app->get_option('error_dump_dir')) if $is_make_d3_json;

    foreach my $accessor (sort @$multistate_models) {

        my ($bits_hash, $multistates) = get_model_data($app->$accessor);

        unless ($is_make_d3_json) {
            print_data($app->$accessor, $bits_hash, $multistates);
        } else {
            create_d3_json($app->$accessor, $bits_hash, $multistates, $app->get_option('error_dump_dir'));
        }
    }
}

sub get_model_data {
    my ($model) = @_;

    my $bits = $model->get_multistates_bits();

    my $bits_hash;
    my $i = 1;
    foreach my $bit (@{$bits}) {
        $bits_hash->{$i}->{name} = $bit->[0];
        $bits_hash->{$i}->{opts} = $bit->[2];
        $i++;
    }

    my $multistates = $model->get_multistates();

    return ($bits_hash, $multistates);
}

sub print_data {
    my ($model, $bits_hash, $multistates) = @_;

    say "# ", $model->accessor();

    say '';
    say '## bits';
    say '';

    foreach my $bit (sort {$a <=> $b} keys %$bits_hash) {
        say sprintf '%4s - %-20s %s', $bit, $bits_hash->{$bit}->{name},
          (
            %{$bits_hash->{$bit}->{opts}}
            ? to_json($bits_hash->{$bit}->{opts})
            : ''
          ),
          ;
    }

    say '';
    say '## Multistates';
    say '';

    foreach my $start (sort {$a <=> $b} keys %{$multistates}) {
        my $name = $model->get_multistate_name($start, private => TRUE);
        $name =~ s/\n/ /g;
        say sprintf "%5s %13b - %s", $start, $start, $name,;
    }

    say '';
    say '## Actions';
    say '';

    say sprintf('%20s %6s   %-40s %20s %6s', 'from', '', 'action', 'to', '');
    say '-' x 100;

    my $prev_start = -1;
    foreach my $start (sort {$a <=> $b} keys %{$multistates}) {
        foreach my $action (sort keys %{$multistates->{$start}}) {
            printf "%20s %6s   %-40s %20b (%4s)\n",
              $start,
              $start,
              $action,
              $multistates->{$start}->{$action},
              $multistates->{$start}->{$action},
              ;

            $prev_start = $start;
        }

    }

    return 1;

}

sub create_d3_json {
    my ($model, $bits_hash, $multistates, $error_dump_dir) = @_;

    my $json = {
        accessor   => $model->accessor,
        graph      => [],
        directed   => FALSE,
        multigraph => FALSE,
        links      => [],
        nodes      => [],
        bits       => []
    };

    my $bits = $json->{bits};
    foreach my $bit (sort {$a <=> $b} keys %$bits_hash) {
        push @$bits,
          {
            bit  => $bit,
            name => $bits_hash->{$bit}->{name}
          };
    }

    my $types = {
        'new' => {
            score => 0,
            type  => 'circle',
        },
        'deleted' => {
            score => 0.5,
            type  => 'square',
        },
        'bad' => {
            score => 1,
            type  => 'triangle-up',
        },
    };

    my $nodes    = $json->{nodes};
    my $node_ids = {};
    my $idx      = 0;
    foreach my $start (sort {$a <=> $b} keys %{$multistates}) {
        my $name = $model->get_multistate_name($start, private => TRUE);
        $name =~ s/\n/ /g;

        $node_ids->{$start} = $idx;

        my $type =
          (      $name =~ /Работает/
              && $name =~
              /(Заархивирован|Удален|Заблокирован|Отклонен|Остановлен)/
          ) ? 'bad'
          : ($name =~
              /(Заархивирован|Удален|Заблокирован|Отклонен|Остановлен)/
          ) ? 'deleted'
          :   'new';

        push @$nodes,
          {
            id    => "$start - $name",
            score => $types->{$type}->{score},
            type  => $types->{$type}->{type},
            size  => 10,
          };
        $idx++;
    }

    my $links = $json->{links};
    foreach my $start (sort {$a <=> $b} keys %{$multistates}) {
        foreach my $action (sort keys %{$multistates->{$start}}) {
            push @$links,
              {
                source => $node_ids->{$start},
                target => $node_ids->{$multistates->{$start}->{$action}},
                action => $action
              };
        }
    }

    my $file_path = sprintf('%s/d3_multistate_graph/%s.json', $error_dump_dir, $model->accessor);
    File::Slurp::write_file($file_path, {binmode => ':utf8'}, to_json($json, pretty => TRUE),);

    File::Slurp::write_file(
        "$error_dump_dir/d3_multistate_graph/index.html",
        {binmode => ':utf8', append => 1},
        sprintf(qq[<li><a href="main.html?accessor=%s">%s</a>\n], $model->accessor, $model->accessor)
    );

    say "$file_path - OK";

}

sub setup_d3 {
    my ($error_dump_dir) = @_;

    File::Path::make_path("$error_dump_dir/d3_multistate_graph", chmod => 0777);

    File::Slurp::write_file(
        "$error_dump_dir/d3_multistate_graph/index.html", {binmode => ':utf8'}, q[
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8"/>
    <title>Яндекс партнер</title>
</head>
<h2>Index</h2>
<ol>
]
    );

}

sub get_args {
    my $accessor = $ARGV[0];

    if (!$accessor || $accessor =~ /^(-h|--help)$/) {
        pod2usage(-verbose => 2, -noperldoc => 1);
    }

    my $is_make_d3_json = 1 if $accessor eq 'generate_json';

    $accessor = $ARGV[1] // '' if $is_make_d3_json;

    return ($accessor, $is_make_d3_json);
}
