#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для раскидывания API тестов по подпапкам по логинам (token в tjson файле)

=head1 USAGE

bin/split-api_tests.pl <PATH>

=head1 OPTIONS

 - PATH - путь для обработки, один файл или папка

=cut

# common modules
use feature 'say';
use Carp;

use File::Basename;
use File::Path qw(make_path);
use FindBin qw($Bin);
use Pod::Usage;

# project modules
use lib::abs '../lib';
use qbit;

main();

sub main {
    pod2usage(-verbose => 2, -noperldoc => 1) unless $ARGV[0];

    process_item($ARGV[0]);
}

sub process_item {
    my ($name) = @_;

    if (-f $name) {
        process_file($name);
    } elsif (-d $name) {
        process_dir($name);
    }
}

sub process_file {
    my ($fname) = @_;

    open F, $fname or die "can't open file $fname: $!\n";
    local undef $/;
    my $json = <F>;
    close F;
    $json =~ s/^(#.*)$//m;
    my $shebang = $1;
    # print "s:$shebang\n";

    my $dir  = dirname($fname);
    my $name = basename($fname);

    # parse and check json
    my $t;
    eval {$t = from_json($json);};
    if ($@) {
        warn "$fname: bad json, can't parse, skipping";
        return;
    }
    unless (exists $t->{tests}) {
        warn "$fname: bad json structure, no 'tests' found, skipping";
        return;
    }

    # collect tests by tokens
    my $tokens = {};
    foreach my $test (@{$t->{tests}}) {
        unless (exists $test->{request}) {
            warn "$fname: bad json structure, test w/o 'request' found, skipping";
            return;
        }
        unless (exists $test->{request}->{token} and length($test->{request}->{token}) > 0) {
            warn "$fname: bad json structure, test w/o 'token' found, skipping";
            return;
        }
        push @{$tokens->{$test->{request}->{token}}}, $test;
    }
    unless (keys %$tokens) {
        warn "$fname: no tokens found, skipping";
        return;
    }
    if (scalar(keys %$tokens) == 1) {
        # corner case - if we are already in a dir with the token name, we should skip moving
        my ($token) = keys %$tokens;
        if ($dir =~ m#/$token$#) {
            warn "$fname: already in a token dir, skipping";
            return;
        }
    }

    # create new tests
    delete $t->{tests};
    foreach my $token (keys %$tokens) {
        my $test = clone($t);
        $test->{tests} = $tokens->{$token};
        unless (-d "$dir/$token") {
            make_path("$dir/$token");
        }
        my $path = qq[dir/$token/$name];
        open F, ">", $path;
        print F $shebang . "\n";
        print F to_json($test, pretty => 1);
        close F;
        `chmod 755 $path`;
    }
    unlink $fname;
    print "$fname converted to " . scalar(keys %$tokens) . " file(s)\n";
}

sub process_dir {
    my ($dir) = @_;

    opendir DIR, $dir or die "can't open dir $dir: $!\n";
    my @names = readdir DIR;
    closedir DIR;

    foreach my $name (@names) {
        next if $name =~ /^\./;
        process_item($dir . '/' . $name);
    }
}
