#!/usr/bin/perl

=encoding utf8

=head1 DESCRIPTION

Cкрипт получает и заливает ключи и их переводы в танкер

=head1 PARAMS

 nocache      - не использует кэши (кейсыты кешируются на полчаса, поиск новых преводов на минуту)
 only_update  - не ищет новые переводы, просто обновляет файлы переводов в проекте
 force_update - обновляет файлы переводов в проекте, даже если нет новых ключей
 force_cache  - игнорировать ttl (для дебага)

=head1 USAGE

 ./bin/tanker-sync.pl
 ./bin/tanker-sync.pl --only_update --nocache
 ./bin/tanker-sync.pl --nocache
=cut

use Config::IniFiles;
use Curses::UI;
use File::Find;
use File::Slurp qw(write_file read_file);
use File::stat qw();
use File::Temp qw(tempfile);
use Net::INET6Glue::INET_is_INET6;
use Getopt::Long;
use Pod::Usage;

use lib::abs qw(../lib/);
use Curses::UI::Common;
use Curses::UI::CommonUTF8;

use PiSecrets;
use QBit::Gettext::Extract;
use Yandex::Tanker;

$SIG{__DIE__} = sub {fatal($_[0]);};

my $CACHE_KEYSET_PATH   = lib::abs::path('../tanker_sync_keysets_cache.json');
my $CACHE_NEW_KEYS_PATH = lib::abs::path('../tanker_sync_new_keys_cache.json');

my $TANKER_SEPARATOR = '+++MSGCTXT+++:';

my %LANG_CODES = (
    be => 'be_BY',
    en => 'en_GB',
    kk => 'kk_KZ',
    ru => 'ru_RU',
    tr => 'tr_TR',
    uk => 'uk_UA',
);

my $startrek_base_url = 'https://st.yandex-team.ru/';

my $cui;

sub push_hs(\[$%]@) {
    my ($h1, @args) = @_;

    $h1 = $$h1 if ref($h1) eq 'REF';
    my $h2 = @args == 1 ? $args[0] : {@args};

    @$h1{keys(%$h2)} = values(%$h2);
}

sub ngettext($$$;@) {
    my ($text, $plural, $n, @params) = @_;
    utf8::encode($text);
    utf8::encode($plural);
    my $msg = Locale::Messages::turn_utf_8_on(Locale::Messages::ngettext($text, $plural, $n));
    return @params ? sprintf($msg, @params) : $msg;
}

sub fatal {
    if ($cui) {
        $cui->error($_[0]);
    } else {
        print "$_[0]\n";
    }
    exit(1);
}

sub tanker_quote {
    my ($str) = @_;

    for ($str) {
        s/\r//g;
        s/\n/\\n/g;
        s/"/\\"/g;
    }

    return $str;
}

sub get_ticket_link {
    my $arc_info = `arc info --json`;
    my $cur_branch_name = eval {JSON::decode_json(Encode::encode_utf8($arc_info))}->{"branch"};
    my ($ticket) = ($cur_branch_name =~ m!^(?:[^/]*/)*([A-Z]{2,}-[0-9]{3,})(\D|$)!);
    return $ticket;
}

sub to_tjson_file {
    my ($keysets, $ticket_link) = @_;

    my %result;
    foreach my $keyset (keys(%$keysets)) {
        foreach my $key_id (keys(%{$keysets->{$keyset}})) {

            my $key_data = $keysets->{$keyset}->{$key_id};

            my $msgid = $key_id;
            $msgid .= $TANKER_SEPARATOR . $key_data->{'context'} if defined $key_data->{'context'};
            $msgid = tanker_quote($msgid);

            $result{$keyset}->{'keys'}{$msgid} = {
                info => {
                    context   => $ticket_link,
                    is_plural => (defined($key_data->{'plural'}) ? JSON::true() : JSON::false()),
                },
                translations => {
                    ru => {
                        (map {$_ => tanker_quote($key_data->{$_})} grep {/^form/} keys %$key_data),
                        status => 'approved',
                    }
                }
            };
        }
    }

    my ($tmp_fh, $tmp_filename) = tempfile('tankerXXXXXX', SUFFIX => '.tjson', TMPDIR => TRUE, UNLINK => TRUE);
    binmode($tmp_fh, ":utf8");
    print $tmp_fh JSON::encode_json({keysets => \%result});
    close($tmp_fh);

    return $tmp_filename;
}

sub update_cache_data {
    my ($cache_file_path, $data) = @_;
    write_file($cache_file_path, JSON::encode_json($data));
}

sub get_cached_data {
    my ($cache_file_path, $args, $ttl_sec) = @_;

    my $data;
    if ($args->{use_cache} && -e $cache_file_path) {
        my $cache_timestamp = File::stat::stat($cache_file_path)->mtime;
        my $cur_timestamp   = time();
        my $sec_left        = $cur_timestamp - $cache_timestamp;
        if ($args->{force_cache} || $sec_left < $ttl_sec) {
            $data = JSON::decode_json(read_file($cache_file_path));
        }

    }
    return $data;
}

sub _get_args {
    my $args = {
        use_cache    => 1,
        only_update  => $ENV{'ONLY_DOWNLOAD'},
        force_update => 0,
        force_cache  => 0,
    };

    my $result = Getopt::Long::GetOptions(
        'cache!'       => \$args->{use_cache},
        'only_update'  => \$args->{only_update},
        'force_update' => \$args->{force_update},
        'force_cache'  => \$args->{force_cache},
        #---
        'help|?|h' => \$args->{'help'},
    );

    if (!$result || $args->{'help'}) {
        pod2usage(-verbose => 2, -noperldoc => 1);
    }
    return $args;
}

my $args = _get_args();

fatal('File ".tanker" does not exists.') unless -f '.tanker';

my $config = Config::IniFiles->new(-file => '.tanker', -fallback => 'GENERAL', -nocase => TRUE)
  // fatal("File \".tanker\" is invalid:\n" . join("\n", @Config::IniFiles::errors));

my $tanker_url;
if (defined($config->val('GENERAL', 'server')) && $config->val('GENERAL', 'server') eq 'production') {
    $tanker_url = $Yandex::Tanker::PROD_URL;
} elsif (defined($config->val('GENERAL', 'server')) && $config->val('GENERAL', 'server') eq 'test') {
    $tanker_url = $Yandex::Tanker::TEST_URL;
} else {
    fatal('No "server" option in config file ".tanker". Expected it to be "production" or "test".');
}

my $tanker = Yandex::Tanker->new(
    {
        project => scalar($config->val('GENERAL', 'project-id')),
        url     => $tanker_url,
        ($args->{'only_update'} ? () : (token => get_secret('tanker-pi2-token'))),
    }
);

$cui->status('Getting project info') if $cui;
my $project_info = $tanker->get_project_info();
$cui->nostatus() if $cui;

my (%new_keysets, %exists_keysets);
if ($args->{'only_update'}) {
    print "Downloading translations from tanker 😻 \n";
} else {

    my $ticket_link = get_ticket_link();
    unless (defined $ticket_link) {
        fatal('Your branch name has incorrect format');
    } else {
        $ticket_link = $startrek_base_url . $ticket_link;
    }

    $cui = Curses::UI->new(
        -clear_on_exit => FALSE,
        -color_support => TRUE,
    );

    $cui->set_binding(sub {exit 0}, "\cQ", "\cD");

    my @file_ignore_re = $config->val('GENERAL', 'file_ignore_re');

    my @invalid_dirs = grep {!-d $_} $config->val('GENERAL', 'dir');
    fatal(
        ngettext(
            'Directory "%s" defined, but does not exists.',
            "Directories defined, but do not exists:\n%s",
            scalar(@invalid_dirs),
            join("\n", @invalid_dirs)
        )
    ) if @invalid_dirs;

    my @invalid_files = grep {!-f $_} $config->val('GENERAL', 'file');
    fatal(
        ngettext(
            'File "%s" defined, but does not exists.',
            "Files defined, but do not exists:\n%s",
            scalar(@invalid_files),
            join("\n", @invalid_files)
        )
    ) if @invalid_files;

    my $keysets = get_cached_data($CACHE_KEYSET_PATH, $args, 60 * 30);
    unless ($keysets) {
        $cui->status('Downloading keysets from Tanker');
        $keysets = JSON::decode_json($tanker->get_project_tjson(status => 'unapproved'))->{'keysets'};
        $cui->nostatus();
        update_cache_data($CACHE_KEYSET_PATH, $keysets);
    }

    my $to_translate = get_cached_data($CACHE_NEW_KEYS_PATH, $args, 60);
    unless ($to_translate) {
        $to_translate = [];

        my @files;
        find(
            {
                wanted => sub {
                    if (-f $File::Find::name) {
                        foreach my $ignore_re (@file_ignore_re) {
                            if ($File::Find::name =~ /$ignore_re/) {
                                return;
                            }
                        }
                        push(@files, $File::Find::name);
                    }
                },
                no_chdir => TRUE
            },
            $config->val('GENERAL', 'dir'),
            $config->val('GENERAL', 'file')
        );

        my $extractor = QBit::Gettext::Extract->new();

        $cui->progress(
            -max   => scalar(@files),
            -title => 'Extracting messages from files',
        );

        my $file_no = 0;
        foreach my $file (@files) {
            $cui->setprogress(++$file_no, $file);
            $extractor->extract_from_file($file);
        }
        $cui->noprogress();

        $cui->status('Compare messages');
        my %tanker_messages;
        foreach my $keyset (keys(%$keysets)) {
            foreach my $key (keys(%{$keysets->{$keyset}{'keys'}})) {
                my ($message, $context) = $key =~ /^(.+?)(?:\Q$TANKER_SEPARATOR\E(.+))?$/;

                my $msg = $tanker_messages{$message, $context // ''} //= {};

                if (keys(%$msg)) {
                    $msg->{'references'} .= "\n$keysets->{$keyset}{'keys'}{$key}{'info'}{'references'}";
                    $msg->{'is_plural'} = TRUE if $keysets->{$keyset}{'keys'}{$key}{'info'}{'is_plural'};
                } else {
                    push_hs(
                        $msg,
                        {
                            message => $message,
                            (defined($context) ? (context => $context) : ()),
                            is_plural  => !!$keysets->{$keyset}{'keys'}{$key}{'info'}{'is_plural'},
                            references => $keysets->{$keyset}{'keys'}{$key}{'info'}{'references'},
                            (
                                $keysets->{$keyset}{'keys'}{$key}{'info'}{'is_plural'}
                                ? ()
                                : (form => $keysets->{$keyset}{'keys'}{$key}{'translations'}{'en'}{'form'})
                            ),
                        }
                    );
                }
            }
        }

        foreach my $message (values(%{$extractor->po->{'__MESSAGES__'}})) {
            if (
                exists(
                    $tanker_messages{tanker_quote($message->{'message'}), tanker_quote($message->{'context'} // '')}
                )
               )
            {
                # Если был перевод без множественного числа, а стал с множественным
                if (defined($message->{'plural'})
                    && !$tanker_messages{tanker_quote($message->{'message'}), tanker_quote($message->{'context'} // '')}
                    ->{'is_plural'})
                {
                    push(
                        @$to_translate,
                        {
                            %$message,
                            tanker_form =>
                              $tanker_messages{$message->{'message'}, $message->{'context'} // ''}->{'form'}
                        }
                    );
                }
            } else {
                push(@$to_translate, $message);
            }
        }

        update_cache_data($CACHE_NEW_KEYS_PATH, $to_translate);

        $cui->nostatus();
    }

    $cui->dialog(
        ngettext(
            '%i message to translation', '%i messages to translation',
            scalar(@$to_translate),      scalar(@$to_translate)
        )
    ) if @$to_translate;

    my $i = 0;
    foreach my $message (@$to_translate) {
        ++$i;
        my $translate = $cui->tempdialog(
            'Dialog::YandexTankerMessage',
            (defined($message->{'context'}) ? (-message_context => $message->{'context'}) : ()),
            -message => $message->{'message'},
            (defined($message->{'plural'}) ? (-message_plural => $message->{'plural'}) : ()),
            -total   => scalar(@$to_translate),
            -current => $i,
        ) // next;

        my $keyset = $message->{'lines'}[0];
        $keyset =~ s/\//__/g;
        $keyset =~ s/\:(?:\d+)$//g;

        my $cur_keysets = exists($keysets->{$keyset}) ? \%exists_keysets : \%new_keysets;

        $cur_keysets->{$keyset}{$message->{'message'}} = {
            (defined($message->{'context'}) ? (context => $message->{'context'}) : ()),
            (
                defined($message->{'plural'})
                ? (
                    plural => $message->{'plural'},
                    form1  => $translate->[0],
                    form2  => $translate->[1],
                    form3  => $translate->[2],
                  )
                : (form => $translate->[0])
            ),
        };
    }

    if (keys(%new_keysets)) {
        $cui->status('Creating keysets');
        $tanker->create_keyset(
            file   => to_tjson_file(\%new_keysets, $ticket_link),
            keyset => [map                         {$_} keys(%new_keysets)],
            format => 'tjson'
        );
        $cui->nostatus();
    }

    if (keys(%exists_keysets)) {
        $cui->status('Merging keysets');
        $tanker->_action_on_keyset(
            file   => to_tjson_file(\%exists_keysets, $ticket_link),
            keyset => [map                            {$_} keys(%exists_keysets)],
            format => 'tjson',
            action => 'onlyadd'
        );
        $cui->nostatus();
    }
}

if ($args->{'only_update'} || $args->{'force_update'} || %new_keysets || %exists_keysets) {

    $cui->progress(
        -max   => scalar(@{$project_info->{'languages'}}),
        -title => 'Downloading and compiling traslations',
    ) if $cui;

    my $i = 0;
    foreach my $lang (@{$project_info->{'languages'}}) {
        $cui->setprogress(++$i, sprintf('%s/%s Processing "%s"', $i, scalar(@{$project_info->{'languages'}}), $lang))
          if $cui;

        my $po_content = $tanker->get_po_translation(language => $lang, status => 'approved', fix_po_content => TRUE,);

        utf8::decode($po_content);

        my ($tmp_fh, $tmp_filename) = tempfile("${lang}XXXXXX", SUFFIX => '.po', TMPDIR => TRUE, UNLINK => TRUE);

        binmode($tmp_fh, ":utf8");
        print $tmp_fh $po_content;
        close($tmp_fh);

        my $lang_dir = "./locale/$LANG_CODES{$lang}";
        system('mkdir', '-p', "$lang_dir/LC_MESSAGES") == 0 or fatal("Cannot create dir $lang_dir/LC_MESSAGES: $?");

        my $command =
            "msgfmt $tmp_filename -o $lang_dir/LC_MESSAGES/"
          . ($config->val('GENERAL', 'locale_domain') // 'project')
          . '.mo 2>&1';

        my $result = `$command`;

        if ($? != 0) {
            fatal("Cannot compile .mo for $lang: $?\nError: $result");
        }
    }
    $cui->noprogress if $cui;
}
