#!/usr/bin/perl -w

=encoding utf8

=head1 DESCRIPTION

Cкрипт запускает обновление статистики для определенных дат и продуктов

=head1 PARAMS

 from     - дата начала
 to       - дата конца
 ttl      - не обновлять если последний забор был не позднее TTL
 storage  - [all;mysql;clickhouse]. По дефолту 'all'

 product     - список продуктов (названия аксессоров через запятую, префикс 'statistics_' можно не указывать)
 cron        - список Сron крон задач для составления списка продуктов
 substr      - список подстрок для фильтрации продуктов
 logins      - список логинов
 pageids     - список пейджей


=head1 USAGE

 update_statistics  --from=startofmonth  --to=today      --ttl=1d     --cron=all
 update_statistics  --from=yesterday     --to=today      --ttl=2h     --product=advnet_context_on_site_direct,mobile_app_rtb
 update_statistics  --from=2017-01-01    --to=2017-01-02 --ttl=30m    --cron=stat_direct,adv_net_video
 update_statistics  --from=2017-01-01    --to=2017-01-02 --ttl=30m    --logins=gomailru-ads,avito
 update_statistics  --from=2017-01-01    --to=2017-01-02 --ttl=30m    --pageids=678234,1123423
 update_statistics  --from=yesterday     --to=today      --ttl=1h30m  --substr=direct,rtb,mobile

=cut

use strict;

use lib::abs qw( ../lib );

use Utils::Moment;

use qbit;
use Cron;
use Utils::Logger qw( INFOF  INFO  ERROR );

use Pod::Usage;
use Getopt::Long qw();

use feature 'say';
use Carp;

#####
_run();
#####

sub _run {

    my $app = _get_app();

    my $stat_accessors      = _get_updating_stats_accessors($app);
    my $cron_stat_accessors = _get_cron_stats_accessors($app);

    my ($dates, $ttl, $storages, $res_products, $logins, $page_ids) = _get_args($stat_accessors, $cron_stat_accessors);

    INFO '#START';

    _validate_logins_pages($app, $logins, $page_ids);

    foreach my $date (@$dates) {
        INFOF 'start %s', $date;
        foreach my $group_name (sort keys %$res_products) {
            INFOF '  start %s', $group_name;
            foreach my $product (@{$res_products->{$group_name}}) {

                my $accessor = 'statistics_' . $product;
                INFOF '    start %s', $accessor;

                foreach my $storage (@$storages) {
                    INFOF '      storage %s', $storage;

                    my ($is_to_update, $last_update) = _is_to_start_update($app, $accessor, $storage, $ttl);

                    if ($is_to_update) {

                        my $exception = undef;
                        try {
                            $app->$accessor->update_statistics(
                                from    => $date,
                                to      => $date,
                                storage => $storage,
                                @$logins   ? ('logins'   => $logins)   : (),
                                @$page_ids ? ('page_ids' => $page_ids) : (),
                            );
                        }
                        catch {
                            ($exception) = @_;
                            if (ref($exception)) {
                                ERROR $exception->as_string(1);
                            } else {
                                ERROR $@;
                            }
                        };

                        $app->$accessor->__save_regular_update_run_date('for_month', storage => $storage)
                          unless $exception;

                        INFOF '      end %s', $storage;
                    } else {
                        INFO sprintf('    ... skipped (last update %s )', $last_update);
                    }
                }
            }
            INFOF '  end %s', $group_name;
        }
        INFOF 'end %s', $date;
    }

    INFO "#END";
}

sub _is_to_start_update {
    my ($app, $accessor, $storage, $ttl) = @_;

    my $is_to_update  = 1;
    my $last_run_date = 'undef';
    if ($ttl) {
        my $key = $app->$accessor->__get_kv_store_key_for_month('storage' => $storage);
        $last_run_date = $app->$accessor->kv_store->get_last_change($key);

        if ($last_run_date) {
            my $now_ts = Utils::Moment->now()->get_timestamp();
            my $last_run_ts = Utils::Moment->new(dt => $last_run_date)->get_timestamp();
            $is_to_update = 0 if $now_ts - $last_run_ts < $ttl;
        }
    }

    return ($is_to_update, $last_run_date);
}

sub _get_cron_stats_accessors {
    my $app = shift;

    my $methods = $app->get_cron_methods();

    my $cron_stat_accessors = {};
    foreach my $path (sort keys %$methods) {
        foreach my $method (sort keys %{$methods->{$path}}) {
            my $package = $methods->{$path}->{$method}->{package};

            my @stat_accessors = eval '@' . $package . "::STATISTICS_UPDATE_PRODUCT_LIST";
            if (@stat_accessors) {
                $cron_stat_accessors->{$path} = [map {s/^statistics_//; $_} @stat_accessors];
            }
        }
    }

    return $cron_stat_accessors;
}

sub _get_updating_stats_accessors {
    my $app = shift;
    return {map {$_ => 1} $app->statistics->get_level_ids_for_reloading()};
}

sub _get_app {

    my $app = Cron->new();

    $app->pre_run();

    $app->set_cur_user({id => 0});

    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {TRUE};

    return $app;
}

sub _validate_logins_pages {
    my ($app, $logins, $page_ids) = @_;

    my $page_models = {};

    my $found_page_ids = {};
    if (@$page_ids) {
        my $data = $app->all_pages->get_all(
            fields => [qw(page_id  login  model)],
            filter => {page_id => $page_ids}
        );

        $page_models = {map {$_->{'model'} => 1} @$data};

        $found_page_ids = {map {$_->{'page_id'} => $_->{'login'}} @$data};

        my @not_found_pages = grep {!$found_page_ids->{$_}} @$page_ids;
        die sprintf 'Unknown page ids found "%s"', join(', ', @not_found_pages) if @not_found_pages;
    }

    my $found_logins = {};
    if ($logins) {
        my $data = $app->partner_db->query->select(
            table  => $app->all_pages->partner_db_table(),
            fields => {login => 'login',},
            filter => {'login' => $logins},
        )->group_by('login')->get_all();

        $found_logins = {map {$_->{login} => 1} @$data};

        my @not_found_logins = grep {!$found_logins->{$_}} @$logins;
        die sprintf 'Unknown logins found "%s"', join(', ', @not_found_logins) if @not_found_logins;
    }

    return 1;
}

sub _get_args {
    my ($stat_accessors, $cron_stat_accessors) = @_;

    my $from    = '';
    my $to      = '';
    my $ttl     = '';
    my $storage = 'all';

    my $product   = '';
    my $cron      = '';
    my $substr    = '';
    my $loginstr  = '';
    my $pageidstr = '';

    my $help = 0;
    Getopt::Long::GetOptions(
        #--- Obligatory
        'from=s'    => \$from,
        'to=s'      => \$to,
        'ttl=s'     => \$ttl,
        'storage:s' => \$storage,
        #-- Optional
        'product:s' => \$product,
        'cron:s'    => \$cron,
        'substr:s'  => \$substr,
        'logins:s'  => \$loginstr,
        'pageids:s' => \$pageidstr,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    # today, yesterday, startofmonth
    {
        my $moment = Utils::Moment->now();
        $from = $moment->get_d() if $from eq 'today';
        $to   = $moment->get_d() if $to   eq 'today';

        $from = $moment->minus(day => 1)->get_d() if $from eq 'yesterday';
        $to   = $moment->minus(day => 1)->get_d() if $to   eq 'yesterday';

        $from = $moment->get_month_start()->get_d() if $from eq 'startofmonth';
    }

    #-- Проверяем зн-ия входных параметров
    my $errors = [];
    push(@$errors, '"to" must be equal or greater than "from"')  unless $to ge $from;
    push(@$errors, 'wrong "from" format, it must be YYYY-MM-DD') unless $from =~ /^\d{4}-\d{2}-\d{2}$/;
    push(@$errors, 'wrong "to" format, it must be YYYY-MM-DD')   unless $to =~ /^\d{4}-\d{2}-\d{2}$/;

    push(@$errors, 'wrong "ttl" format') unless !$ttl || $ttl =~ /^(\d+(d|h|m))+$/;

    push(@$errors, 'wrong "storage" format') unless grep {$storage eq $_} qw( all  mysql  clickhouse );

    unless ($product || $cron || $substr) {
        push @$errors, 'you must to specify one of "product", "cron", "substr"';
    }

    my @products = map {s/^statistics_//; $_} split /,/, $product;
    my @extra_products = grep {$_ ne 'all' && !$stat_accessors->{$_}} @products;
    if (@extra_products) {
        push @$errors, sprintf('unknown products "%s"', join(', ', sort @extra_products));
    }

    my @crons = split /,/, $cron;
    my @extra_cron = grep {$_ ne 'all' && !$cron_stat_accessors->{$_}} @crons;
    if (@extra_cron) {
        push @$errors,
          sprintf(
            'unknown crons "%s". It must be one of: %s',
            join(', ', sort @extra_cron),
            join(', ', sort keys %$cron_stat_accessors)
          );
    }

    my $res_products = {};
    my $logins       = [];
    my $page_ids     = [];
    my $dates        = [];
    my $storages     = [qw( mysql clickhouse )];
    if (@$errors) {
        print join("\n", @$errors), "\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(1);
    } else {

        # dates
        {
            my $moment  = Utils::Moment->new(dt => $from . ' 00:00:00');
            my $curr    = $from;
            my $counter = 0;
            while ($curr le $to) {
                push @$dates, $curr;
                $curr = $moment->plus(day => ++$counter)->get_d();
            }
        }

        # ttl
        {
            my $ttl_sec = 0;

            my ($days) = ($ttl =~ /(\d+)+d/);
            $ttl_sec += $days * 24 * 60 * 60 if $days;
            my ($hours) = ($ttl =~ /(\d+)+h/);
            $ttl_sec += $hours * 60 * 60 if $hours;
            my ($mins) = ($ttl =~ /(\d+)+m/);
            $ttl_sec += $mins * 60 if $mins;

            $ttl = $ttl_sec;
        }

        # products
        {
            @products = keys %$stat_accessors if grep {$_ eq 'all'} @products;
            map {$res_products->{$_} = 1} @products;

            @crons = keys %$cron_stat_accessors if grep {$_ eq 'all'} @crons;
            map {$res_products->{$_} = 1} map {@{$cron_stat_accessors->{$_}}} @crons;

            if ($substr) {
                (my $regex = $substr) =~ s/,/|/g;
                $res_products = {
                    map {$_ => 1}
                    grep {$_ =~ /$regex/} keys %$res_products
                };
            }

            $res_products = {all => $res_products};
            if (@crons) {
                my $new_products = {};
                foreach my $cron (@crons) {
                    $new_products->{$cron} = [
                        map {delete $res_products->{all}->{$_}; $_}
                        grep {$res_products->{all}->{$_}} @{$cron_stat_accessors->{$cron}}
                    ];
                }
                $res_products = {%$res_products, %$new_products};
            }
            $res_products->{all} = [sort keys %{$res_products->{all}}];
            delete $res_products->{all} unless @{$res_products->{all}};
        }

        # logins
        $logins = [split /\s*,\s*/, $loginstr] if $loginstr;

        # pageids
        $page_ids = [split /\s*,\s*/, $pageidstr] if $pageidstr;

        $storages = ['mysql']      if $storage eq 'mysql';
        $storages = ['clickhouse'] if $storage eq 'clickhouse';
    }

    return ($dates, $ttl, $storages, $res_products, $logins, $page_ids);
}
