(function () {
    'use strict';

    /**
     * Преобразует дату из строки в экземпляр Date
     *
     * @param {string} d Например, '2014-12-31'
     * @return {number}
     **/
    function getDateFromISOFormat(d) {
        if (typeof d !== 'string' || d.length === 0) {
            return;
        }
        var ds = d.split('-');

        var date = parseInt(ds[2], 10);
        var month = parseInt(ds[1], 10) - 1;
        var year = parseInt(ds[0], 10);

        return (new Date(year, month, date));
    }

    /**
     * Преобразует строку года к числу года
     *
     * @param {string} year +10|-10|1900
     * @return {number}
     **/
    function parseYear(year) {
        var result = parseInt(year, 10);
        var currentYear = (new Date()).getFullYear();

        if (year[0] === '+' || year[0] === '-') {
            result = currentYear + result;
        } else if (isNaN(result)) {
            result = currentYear;
        }

        return result;
    }

    /**
     * Возвращает 1 января первого года периода и 31 декабря второго года периода
     *
     * @param {string} range '1900:+00'|'-10:+10'|'c-10:c+10'|'1900:c+1'
     * @return {object}
     **/
    function getRangeDates(range) {
        var years = range.replace(/[^-+:0-9]+/g, '').split(':');
        return {
            minDate: parseYear(years[0]) + '-01-01',
            maxDate: parseYear(years[1]) + '-12-31'
        };
    }

    BEM.DOM.decl('b-calendar', {
        onSetMod: {
            js: function () {
                var params = {
                    firstDay: 1,
                    dateFormat: 'yy-mm-dd',
                    changeMonth: true,
                    changeYear: true,
                    minDate: getDateFromISOFormat(this.params.minDate),
                    maxDate: getDateFromISOFormat(this.params.maxDate),
                    defaultDate: getDateFromISOFormat(this.params.value),
                    onSelect: this._onSelect.bind(this)
                };

                var rangeDates;
                if (this.params.yearRange) {
                    params.yearRange = this.params.yearRange;
                    // Уснатандиваем minDate и maxDate,
                    // чтобы нельзя было выбрать дату за пределами пиапазона годов
                    // т.к. год будет отображаться только из диапазона
                    if (!params.minDate) {
                        rangeDates = getRangeDates(params.yearRange);
                        params.minDate = getDateFromISOFormat(rangeDates.minDate);
                    }
                    if (!params.maxDate) {
                        rangeDates = rangeDates || getRangeDates(params.yearRange);
                        params.maxDate = getDateFromISOFormat(rangeDates.maxDate);
                    }
                }

                this.elem('container').datepicker(params);

                this.on(this.elem('container'), 'click', this._onContainerClick, this);
            }
        },

        /**
         * При выборе даты тригерит событие change
         **/
        _onSelect: function (selectedDate, e) {
            e.data = selectedDate;
            this.trigger('change', e);
        },

        /**
         * При клике на контейнер, если были нажаты стрелки
         * вперед или назад (навигация по месяцам), - тригерит
         * событие clickOnMonthNav
         **/
        _onContainerClick: function (e) {
            var _this = this;
            $(e.target).parents().andSelf().each(function (index, el) {
                if ($(el).is('.ui-datepicker-prev, .ui-datepicker-next')) {
                    _this.trigger('clickOnMonthNav');
                    return false;
                }
            });
        },

        /**
         * Устанавливает дату или возвращает выбранную дату
         *
         * @param {string} value Дата в формате 2014-12-31
         **/
        val: function (value) {
            if (typeof value !== 'undefined' && value !== null) {
                this.elem('container').datepicker('setDate', value);
                return;
            }
            return this.elem('container').datepicker({dateFormat: 'yy-mm-dd'}).val();
        }
    });
})();
