/*global Highcharts*/
(function () {
    'use strict';

    var chart = require('utils/chart');
    var colors = require('utils/colors');
    var lodash = require('lodash');

    /**
     * Формирует html цветной точки с учетом указанного цвета.
     *
     * @param  {string} color
     * @return {string}
     */
    function buildBullet(color) {
        return '<span style="color:' + color + ';font-size:18px;line-height:12px;vertical-align:top">\u25CF</span>';
    }

    /**
     * Формирует html для выделенного элемента тултипа.
     *
     * @param  {String} color
     * @param  {String} name
     * @param  {String} val
     * @param  {String} currency
     * @return {String}
     */
    function buildHoveredHTML(color, name, val, currency) {
        var bgcolor = '#deebef';

        return [
            '<tr>',
            '<td style="background:' + bgcolor + '">',
                name + ':',
            '</td>',
            '<td style="' + [
                'background:' + bgcolor,
                'padding-left:1em',
                'text-align:right'
            ].join(';') + '">',
                val,
            '</td>',
            '<td style="background:' + bgcolor + ';">',
                currency,
            '</td>',
            '</tr>'
        ];
    }

    /**
     * Формирует html для невыделенного элемента тултипа.
     *
     * @param  {String} color
     * @param  {String} name
     * @param  {String} val
     * @param  {String} currency
     * @return {String}
     */
    function buildUnhoveredHTML(color, name, val, currency) {
        return [
            '<tr>',
            '<td>',
                name + ':',
            '</td>',
            '<td style="' + [
                'padding-left:1em',
                'text-align:right'
            ].join(';') + '">',
                val,
            '</td>',
            '<td>',
                currency,
            '</td>',
            '</tr>'
        ];
    }

    /**
     * Формирует отформатированную html строку для заголовка.
     *
     * @param  {Object} params
     * @return {String}
     */
    function buildHTML(params) {
        var html;

        if (params.hovered) {
            html = buildHoveredHTML(params.color, params.name, params.val, params.currency);
        } else {
            html = buildUnhoveredHTML(params.color, params.name, params.val, params.currency);
        }

        return html.join('');
    }

    BEM.DOM.decl('b-chart', {
        onSetMod: {
            js: function () {
                this._initBlock();
            }
        },

        /**
         * Инициализация блока
         *
         */
        _initBlock: function () {
            /*
             * this.params.series structure:
             *      [{
             *          name: name,
             *          data: [{x:dateTimestamp, y:value}],
             *          axisId: axisId,
             *          axisName: dimensionAxisName
             *     }]
             *
             * this.params.yAxis structure:
             *      [{
             *          name: axisName
             *          id: axisId
             *     }]
             */
            var type = 'Chart';

            if (this.getMod('type') === 'stock') {
                type = 'StockChart';
            }

            this.params = this.params || {};

            this.params.series && this.params.series.forEach(function (graph, i) {
                graph.color = graph.color || colors.getColorByIndex(i);
                graph.name = buildBullet(graph.color) + ' ' + graph.name;
            });

            Highcharts.setOptions(this._globalOptions());
            this.chart = new Highcharts[type](this._getChartParams());
        },

        /**
         * Глобальные настройки для библиотеки HighCharts
         *
         * Применяются ко всем графикам
         */
        _globalOptions: function () {
            var monthList = [
                'jan',
                'feb',
                'mar',
                'apr',
                'may',
                'jun',
                'jul',
                'aug',
                'sep',
                'oct',
                'nov',
                'dec'
            ];

            var weightList = [
                'kilo',
                'mega',
                'giga',
                'tera',
                'peta',
                'exa'
            ];

            var hsh = {
                global: {
                    useUTC: true
                },
                lang: {
                    rangeSelectorZoom: BEM.I18N('b-chart', 'range-selector-zoom'),

                    loading: BEM.I18N('b-chart', 'loading'),

                    months: monthList.map(function (month) {
                        return BEM.I18N('b-chart', 'month-' + month);
                    }),

                    shortMonths: monthList.map(function (month) {
                        return BEM.I18N('b-chart', 'month-' + month + '-short');
                    }),

                    numericSymbols: weightList.map(function (prefix) {
                        return ' ' + BEM.I18N('b-chart', 'numeric-symbol-' + prefix);
                    })
                }
            };

            return hsh;
        },

        /**
         * Локальные настройки
         *
         * Пересчитываются для каждого chart-a
         */
        _plotOptions: function (params) {
            var _this = this;
            var plotOptions = {
                series: {
                    marker: {
                        enabled: false,
                        states: {
                            hover: {
                                enabled: true,
                                lineColor: '#ffffff',
                                lineWidth: 4,
                                radius: 6
                            }
                        },
                        symbol: 'circle'
                    },
                    states: {
                        hover: {
                            halo: false,
                            lineWidth: params.multiple ? 4 : 2
                        }
                    },
                    turboThreshold: 0
                }
            };

            if (params.multiple) {
                plotOptions.series.events = {
                    mouseOver: function () {
                        var chart = _this.chart;
                        var points = chart.hoverPoints || _this._pointsCache;

                        _this._pointIndex = this._i;

                        if (points) {
                            if (typeof points[0].setState !== 'function') {
                                _this._pointsCache = null;

                                return;
                            }

                            _this._pointsCache = points;
                            chart.tooltip.refresh(points);
                        }
                    },
                    mouseOut: function () {
                        _this._pointIndex = null;
                    }
                };
            }

            return plotOptions;
        },

        /**
         * Функция (!!!), используемая для формирования содержимого тултипа
         * (попап, который следует за курсором и выводит инфу о графиках)
         *
         *
         *
         * this: ссылается на некоторые данные, подробнее
         *
         * this.percentage (not shared) / this.points[i].percentage (shared)
         *      Stacked series and pies only. The point's percentage of the total.
         * this.point (not shared) / this.points[i].point (shared)
         *      The point object. The point name, if defined, is available through this.point.name.
         * this.points
         *      In a shared tooltip, this is an array containing all other properties for each point.
         * this.series (not shared) / this.points[i].series (shared)
         *      The series object. The series name is available through this.series.name.
         * this.total (not shared) / this.points[i].total (shared)
         *      Stacked series only. The total value at this point's x value.
         * this.x
         *      The x value. This property is the same regardless of the tooltip being shared or not.
         * this.y (not shared) / this.points[i].y (shared)
         *      The y value.
         */
        _tooltipFormatter: function (bl) {
            var params = bl.params;
            var points = bl._points = this.points || [this.point];
            var html = '';

            var currency;
            var type;
            var yAxis;

            // Дата
            if (bl.getMod('type') === 'stock') {
                html += BEMHTML.apply({
                    block: 'b-formatter',
                    tag: 'span',
                    mods: {type: 'date'},
                    value: {date: params.datesMap[this.x], type: params.interval},
                    separator: ' '
                });
            }

            html += '<table>';

            points.forEach(function (point, i) {
                yAxis = point.series.options.yAxis;
                type = params.yAxis && params.yAxis[yAxis].type;
                currency = type === 'money' ?
                    ' ' + params.yAxis[yAxis].extraValues.entity_field.currency :
                    '';

                var value = point.y;

                switch (type) {
                case 'percent':
                    value = parseFloat(value);

                    if (isNaN(value)) {
                        value = '-';
                    } else {
                        value = Math.round(value);
                        currency = ' %';
                    }

                    break;
                default:
                    value = chart.formatNumber(value, Boolean(currency));
                }

                html += buildHTML({
                    name: point.series.name,
                    val: value,
                    currency: currency,
                    hovered: bl._pointIndex === i
                });
            });

            html += '</table>';

            return html;
        },

        /**
         * @return {array}
         **/
        _formatSeries: function () {
            var series = (this.params.series || []).map(function (seriesItem, i) {
                var ctx = {
                    color: seriesItem.color || colors.getColorByIndex(i),
                    data: seriesItem.data,
                    name: seriesItem.name,
                    type: seriesItem.type,
                    yAxis: seriesItem.yAxis != null ? seriesItem.yAxis : i
                };

                seriesItem.dashStyle && (ctx.dashStyle = seriesItem.dashStyle);

                return ctx;
            });

            return series;
        },

        /**
         * @return {object}
         **/
        _formatYAxis: function () {
            var series = this.params.series;

            return this.params.yAxis.map(function (axis, i) {
                var isOpposite = i % 2 !== 0;
                var dots = lodash
                    .chain(series)
                    .filter({yAxis: i})
                    .pluck('color')
                    .map(buildBullet)
                    .value()
                    .join('');

                return {
                    gridLineColor: '#e5e5e5',
                    labels: {
                        align: 'left',
                        enabled: true,
                        style: {
                            fontSize: '11px'
                        },
                        x: 10
                    },
                    lineColor: '#666666',
                    lineWidth: 1,
                    min: 0,
                    minRange: 0.1, // Чтобы не рисовал нулевую линию по центру графика.
                    offset: (isOpposite ? -40 : 65) + Math.ceil(i / 2) * 85,
                    opposite: isOpposite,
                    startOnTick: true,
                    tickColor: '#666666',
                    tickLength: 7,
                    title: {
                        align: 'low',
                        offset: isOpposite ? -10 : null,
                        margin: isOpposite ? -35 : 10,
                        rotation: 270,
                        text: dots + ' ' + (axis.label || ''),
                        style: {
                            color: '#666666',
                            fontSize: '13px',
                            fontWeight: 'normal',
                            letterSpacing: '0.3px'
                        }
                    }
                };
            }, this);
        },

        /**
         * @return xAxis
         **/
        _formatXAxis: function () {
            var xAxis = {
                dateTimeLabelFormats: {
                    day: '%e %b',
                    week: '%e %b'
                },
                lineColor: '#666666',
                tickColor: '#666666',
                tickLength: 7
            };

            if (this.getMod('type') === 'stock') {
                xAxis.ordinal = false;
            } else {
                xAxis.categories = this.params.categories;
                xAxis.labels = {rotation: -60, align: 'right'};
            }

            return xAxis;
        },

        /**
         * Формирует общий объект параметров для библиотеки Highcharts
         *
         */
        _getChartParams: function () {
            var _this = this;
            var isColumn = Array.isArray(this.params.categories);
            var seriesData = this._formatSeries();

            var options = {
                chart: {
                    renderTo: this.elem('container').get(0),
                    style: {
                        fontFamily: 'Arial, Helvetica, sans-serif'
                    }
                },

                // Highchart by default puts a credits label in the lower right corner of the chart.
                // This can be changed using these options.
                credits: {
                    enabled: false
                },

                legend: {
                    align: 'center',
                    enabled: true,
                    itemMarginBottom: 10,
                    itemHoverStyle: {
                        color: '#f00'
                    },
                    itemStyle: {
                        color: '#666',
                        fontWeight: 'normal'
                    },
                    labelFormatter: function () {
                        return this.name.replace(/^(<span[^>]+?>.<\/span>\s)(.*?)$/,
                            '$1 <span class="legend-text">$2</span>');
                    },
                    symbolWidth: 0,
                    useHTML: false,
                    verticalAlign: 'top'
                },

                // The navigator is a small series below the main series, displaying a view of the entire data set.
                // It provides tools to zoom in and out on parts of the data as well as panning across the dataset.
                navigator: {
                    enabled: false
                },

                // The range selector is a tool for selecting ranges to display within the chart.
                // It provides buttons to select preconfigured ranges in the chart,
                // like 1 day, 1 week, 1 month etc.
                // It also provides input boxes where min and max dates can be manually input.
                rangeSelector: {
                    inputEnabled: false,
                    enabled: false
                },

                plotOptions: this._plotOptions({
                    multiple: !isColumn && seriesData.length > 1
                }),

                scrollbar: {
                    enabled: false
                },

                series: seriesData,

                title: {
                    text: null
                },

                tooltip: {
                    backgroundColor: 'rgba(255, 255, 255, 0.95)',
                    formatter: function () {
                        // В библиотеке this ссылается на некоторые данные,
                        // поэтому пробрасываем ссылочку на наш блок
                        return _this._tooltipFormatter.call(this, _this);
                    },
                    useHTML: true
                },

                xAxis: this._formatXAxis(),

                yAxis: this._formatYAxis()
            };

            var axisNum = this.params.yAxis.length;

            if (axisNum > 1) {
                options.chart.marginRight = Math.floor(axisNum / 2) * 85 + 30;
            }

            return options;
        }
    });
})();
