/* global alert: false, confirm: false*/

(function() {
    'use strict';
    BEM.DOM.decl('b-dashboard', {
        onSetMod: {
            js: function() {
                this._initBlock();
                this._addI18N();
                this._toggleActions();
                this._addWidgets();
                this._renderWidgets();
                this._addListeners();
            },
            'drag-and-drop': {
                yes: function() {
                    this._toggleActions(true);
                },
                '': function() {
                    this._toggleActions();
                }
            },
            state: {
                saving: function() {
                    this.bProgressPopup.setMod('progress', 'on');
                },
                '': function() {
                    this.bProgressPopup.delMod('progress');
                }
            }
        },

        _initBlock: function() {
            // {number}
            this.top = this.domElem.offset().top;
            // {jQuery}
            this.$window = $(window);
            // {number}
            this.windowHeight = this.$window.height();

            // {array:number} - array of widgets ids
            this.widgetIdList = [];
            // {array:number} - for cancel buton
            this.widgetIdListBackup = [];
            // {number:b-widget}
            this.widgets = {};
            // {number:bemjson} - id:opts
            this.widgetsOpts = {};
            // {number:b-widget}
            this.widgetsOptsBackup = {};
            // params of the active widget for drag and drop
            this.movingWidget = {
                // {number}
                index: -1,
                // {b-widget}
                block: null,
                // {jQuery}
                $el: null,
                // {jQuery}
                $placeholder: null,
                margin: {top: 0, left: 0},
                // {string:number} - { clientX, clientY, pageX, pageY } cursor position
                position: null
            };
            // {string:function}
            this.functions = {
                onDrop: null,
                onMove: null,
                onResize: null
            };
            // {b-widget-wetup}
            this.bWidgetSetup = null;
            // {b-widget-library}
            this.bWidgetLibrary = this.findBlockInside('b-widget-library');
            // {b-progress-popup}
            this.bProgressPopup = this.findBlockInside('b-progress-popup');
            // {number}
            this._widgetNewId = 0;
            // {boolean}
            this.isMoving = false;
            // {boolean} - delay for moving after position change to prevent blocks from jumping
            this.shouldAddDelay = false;
            // {number} - timestamp
            this.scrollStart = 0;
            // {number}
            this.scrollStep = 0;
            // {boolean}
            this._hasChanges = false;

            this.delMod('drag-and-drop');
        },

        _addListeners: function() {
            this.bindTo('setup', 'click', this._onSetupClick);
            this.bindTo('save', 'click', this._onSaveClick);
            this.bindTo('cancel', 'click', this._onCancelClick);

            BEM.channel('widget').on('setup', $.proxy(this._onWidgetSetup, this));
            BEM.channel('widget').on('close', $.proxy(this._onWidgetClose, this));
            BEM.channel('widget').on('resize', $.proxy(this._onWidgetResize, this));

            this.bWidgetLibrary.on('select', $.proxy(this._onAddWidget, this));
            this.$window.on('beforeunload', $.proxy(this._onBeforeUnload, this));
        },

        _onBeforeUnload: function() {
            if (this.hasMod('drag-and-drop') && this._hasChanges) {
                return BEM.I18N('b-dashboard',
                   'Changes haven\'t been saved. Are you sure you want to leave this page?');
            }
        },

        /**
         * Does I18N decl for complicated
         **/
        _addI18N: function() {
            var block,
                i18n = this.params.addToI18N,
                name,
                result;

            if (!i18n) {
                return;
            }

            for (block in i18n) {
                result = {};
                for (name in i18n[block]) {
                    result[name] = i18n[block][name];
                }
                BEM.I18N.decl(block, result);
            }
        },

        _toggleActions: function(editMode) {
            if (editMode) {
                this.elem('actions').show();
                this.elem('setup').hide();
            } else {
                this.elem('actions').hide();
                this.elem('setup').show();
            }
        },

        /** Render methods **/

        /**
         * Generates widgets bemjson
         *
         * @return {bemjson}
         **/
        _addWidgets: function() {
            var _this = this,
                optList = this.widgetIdList;

            if (!optList || !optList.length) {
                optList = this.params.widgets || [];
            }

            if (typeof optList === 'string') {
                optList = $.parseJSON(optList);
            }

            optList.forEach(function(opt) {
                _this._addWidget(opt, true);
            });
        },

        /**
         * Add widget to list,
         * add widgetOpts if needed
         *
         * @param {object} opts
         * @param {number} widgetId
         * @param {boolean} [addToEnd] - false if we want to prepend widget
         **/
        _addWidget: function(opts, widgetId, addToEnd) {

            if (typeof widgetId === 'boolean') {
                addToEnd = widgetId;
                widgetId = undefined;
            }

            if (!this.widgetsOpts[widgetId]) {
                widgetId = this._addWidgetOpts(opts);
            } else {
                $.extend(this.widgetsOpts[widgetId], opts);
            }

            if (addToEnd) {
                this.widgetIdList.push(widgetId);
            } else {
                this.widgetIdList.unshift(widgetId);
            }
        },

        /**
         * Generates widget bemjson with this.params.widgets item
         * @param {number} widgetId
         * @return {bemjson}
         **/
        _genWidget: function(widgetId) {
            var bemjson = {
                    block: 'b-widget',
                    mods: {},
                    mix: [{block: 'b-dashboard', elem: 'widget'}],
                    js: {id: widgetId}
                },
                i18n = this.params.i18n.widget,
                text,
                type;

            bemjson = $.extend(true, bemjson, this.widgetsOpts[widgetId]);

            text = i18n[bemjson.mods.type] && i18n[bemjson.mods.type][bemjson.mods.subtype];
            if (text) {
                bemjson.text = text;
            }

            type = bemjson.mods.type;
            if (this.params.signs && this.params.signs[type]) {
                bemjson.js.sign = this.params.signs[type];
            }

            return bemjson;
        },

        /**
         * Generate widgets params from elem('widget')
         *
         * @return {array:bemjson}
         **/
        _getWidgetsParams: function() {
            var _this = this,
                params = [];

            params = this.widgetIdList.map(function(id) {
                return _this.widgetsOpts[id];
            });

            return params;
        },

        _refreshWidgetsSizes: function() {
            var _this = this;

            this.widgetIdList.forEach(function(id) {
                _this.widgets[id].refreshSize();
            });
        },

        _setWidgetsSizes: function() {
            var _this = this,
                w = this.movingWidget;

            this.widgetIdList.forEach(function(widgetId) {
                var $el,
                    bWidget = _this.widgets[widgetId],
                    offset;

                $el = bWidget.domElem;
                // is active
                if (bWidget === w.block) {
                    offset = w.$placeholder.offset();
                }

                bWidget.refreshSize({offset: offset});
            });

        },

        /** Save methods **/

        _onSetupClick: function() {
            this.widgetIdListBackup = this.widgetIdList.slice(0);
            this.widgetsOptsBackup = $.extend(true, {}, this.widgetsOpts);
            this._addDragAndDrop();
        },

        _onCancelClick: function() {
            this._removeDragAndDrop();
            this._removeWidgets();

            this.widgetsOpts = $.extend(true, {}, this.widgetsOptsBackup);
            this.widgetIdList = this.widgetIdListBackup.slice(0);

            this._renderWidgets();
            this.dropElemCache();
            this._hasChanges = false;
        },

        _onSaveClick: function() {
            this._save(true);
            this._hasChanges = false;
        },

        _onAddWidget: function(e, type) {
            this._setupWidget(this._addWidgetOpts(type));
        },

        _save: function(removeDND) {
            this.setMod('state', 'saving');
            $.ajax({
                type: 'POST',
                url: this.params.saveURL,
                dataType: 'json',
                data: {
                    json: JSON.stringify(this._getWidgetsParams()),
                    sign: this.params.saveSign
                },
                success: $.proxy(this._onSaveSuccess, this, removeDND),
                error: $.proxy(this._onSaveError, this)
            });
        },

        _onSaveSuccess: function(removeDND) {
            this.delMod('state');

            if (removeDND) {
                this._removeDragAndDrop(true);
                this.setMod(this.elem('tools'), 'mode', 'view');
            }
        },

        _onSaveError: function(data) {
            this.delMod('state');
            alert(data.statusText);
        },

        /**
         * Show widget setup
         *
         * @param {event} e
         * @param {number} widgetId
         **/
        _onWidgetSetup: function(e, widgetId) {
            this._setupWidget(widgetId);
        },

        _setupWidget: function(widgetId) {
            var widget = this.widgetsOpts[widgetId];

            BEM.DOM.update(this.elem('widget-setup'), BEMHTML.apply({
                block: 'b-widget-setup',
                mods: {type: widget.mods.type}
            }));
            this.bWidgetSetup = this.findBlockInside(this.elem('widget-setup'), 'b-widget-setup');
            this.bWidgetSetup.on('apply', $.proxy(this._onWidgetSetupApply, this));
            // I don't want entities_tree to be in html, so I pass it through this method
            this.bWidgetSetup.show({
                widget: $.extend(true, {}, widget),
                widgetId: widgetId,
                entities_tree: this.params.entities_tree,
                // Есть виджеты, которые нужно отображать с другим деревом.
                // В статистике для уровней и для графиков используется одно и то же дерево.
                // У пользователя нет прав просматривать статистику, но виджет с графиком мы ему рисуем.
                entities_tree_with_internal: this.params.entities_tree_with_internal
            });
        },

        _onWidgetClose: function(e, widgetId) {
            if (confirm(BEM.I18N('b-dashboard', 'Are you sure you want to delete this widget?'))) {
                this._removeWidget(widgetId);
                this._hasChanges = true;
                //this._save();
            }
        },

        _onWidgetResize: function() {
            this._setWidgetsSizes();
        },

        _onWidgetSetupApply: function(e, opts) {
            var bWidget = this.widgets[opts.widgetId],
                oldWidgetId;

            if (bWidget) {
                oldWidgetId = opts.widgetId;
            }

            if (oldWidgetId === undefined) {
                // add new widget
                this._addWidget(opts.widget, opts.widgetId);
                this._renderWidget(opts.widgetId);
            } else {
                // edit widget
                bWidget.refresh(opts.widget);
                this.widgetsOpts[oldWidgetId] = opts.widget;
            }

            this._hasChanges = true;
            this._refreshWidgetsSizes();
            //this._save();
            this.bWidgetSetup.destruct();
            this.bWidgetSetup = null;
        },

        /**
         *
         **/
        _removeWidgets: function() {
            var _this = this;

            this.widgetIdList.forEach(function(widgetId) {
                _this._removeWidget(widgetId, true);
            });

            this.widgetIdList = [];
        },

        /**
         * Remove widget
         *
         * @param {number} widgetId
         * @param {boolean} doNotRemoveFromList - needed for this.widgetIdList.forEach
         **/
        _removeWidget: function(widgetId, doNotRemoveFromList) {
            var bWidget = this.widgets[widgetId];

            if (bWidget === undefined || bWidget === null) {
                return;
            }

            if (!doNotRemoveFromList) {
                this.widgetIdList.splice(this.widgetIdList.indexOf(widgetId), 1);
            }
            delete this.widgetsOpts[widgetId];
            delete this.widgets[widgetId];
            bWidget.destruct();
        },

        _renderWidgets: function() {
            var i;

            for (i = this.widgetIdList.length - 1; i >= 0; i--) {
                this._renderWidget(this.widgetIdList[i]);
            }

            this._refreshWidgetsSizes();
        },

        /**
         * Render new widget after passed one or at the end,
         * init rendered widget
         *
         * @param {number} widgetNewId
         * @return {b-widget}
         **/
        _renderWidget: function(widgetNewId) {
            var $widgetNew,
                bWidget;

            $widgetNew = this._addWidgetToDom(widgetNewId);
            bWidget = this._initWidget(widgetNewId, $widgetNew);

            return bWidget;
        },

        /**
         * Generate widget opts and bemjson,
         * add opts to this.widgetOpts
         *
         * @param {bemjson} opts
         * @return {number}
         **/
        _addWidgetOpts: function(opts) {
            var defOpts = {
                    mods: {},
                    js: {}
                },
                widgetNewId = this._widgetNewId++;

            if (typeof opts === 'string') {
                defOpts.mods.type = opts;
                opts = defOpts;
            } else {
                opts = $.extend(true, defOpts, opts);
            }

            this.widgetsOpts[widgetNewId] = opts;

            return widgetNewId;
        },

        /**
         * Renders widget,
         * add widgetId to this.widgetIdList
         *
         * @param {number} widgetNewId
         * @return {jQuery}
         **/
        _addWidgetToDom: function(widgetNewId) {
            var bemhtml = BEMHTML.apply(this._genWidget(widgetNewId));
            BEM.DOM.prepend(this.elem('widgets'), bemhtml);

            return this.findElem('widget').first();
        },

        /**
         * Init b-widget,
         * add b-widget to this.widgets
         *
         * @param {number} widgetNewId
         * @param {jQuery} $widgetNew
         * @return {b-widget}
         **/
        _initWidget: function(widgetNewId, $widgetNew) {
            var bWidget = this.findBlockOn($widgetNew, 'b-widget');

            bWidget.refresh({
                entities_tree: this.params.entities_tree,
                // Есть виджеты, которые нужно отображать с другим деревом.
                // В статистике для уровней и для графиков используется одно и то же дерево.
                // У пользователя нет прав просматривать статистику, но виджет с графиком мы ему рисуем.
                entities_tree_with_internal: this.params.entities_tree_with_internal
            });
            this.widgets[widgetNewId] = bWidget;

            if (this.hasMod('drag-and-drop', 'yes')) {
                bWidget.setMod('locked', 'yes');
            }
            return bWidget;
        },

        /** Move Methods **/

        /**
         * Add elements and events for drag and drop
         **/
        _addDragAndDrop: function() {
            this.setMod('drag-and-drop', 'yes');
            this._setWidgetsSizes();
            this._lockWidgets();

            this.functions = {
                onResize: $.throttle(this._onResize, 1000, this),
                onMove: $.throttle(this._onMove, 50, this),
                onDrop: $.proxy(this._onDrop, this)
            };

            this.$window.on('resize', this.functions.onResize)
                .on('mousemove', this.functions.onMove)
                .on('mouseup', this.functions.onDrop);

            this.domElem.on('mousedown', this.buildSelector('widget'), $.proxy(this._onDrag, this));
        },

        _removeDragAndDrop: function(removeLocks) {
            this.delMod('drag-and-drop');
            if (removeLocks) {
                this._removeWidgetsLock();
            }

            this.$window.off('resize', this.functions.onResize)
                .off('mousemove', this.functions.onMove)
                .off('mouseup', this.functions.onDrop);
            this.domElem.off('mousedown', this.buildSelector('widget'));
        },

        /**
         * Refresh board size and offset,
         * render loks on widget elem
         **/
        _onResize: function() {
            this.windowHeight = this.$window.height();
            this._lockWidgets();
        },

        /**
         * Sets active widget and its margin
         *
         * @param {event} e
         **/
        _onDrag: function(e) {
            var $el = $(e.currentTarget),
                bWidget = this.findBlockOn($el, 'b-widget'),
                offset = bWidget.size.block.offset,
                w = this.movingWidget;
            e.preventDefault();

            w.margin = {
                left: offset.left - e.pageX,
                top: offset.top - e.pageY
            };
            w.$el = $el;
            w.block = bWidget;
            w.index = this.widgetIdList.indexOf(bWidget.getId());
            w.position = {
                clientX: e.clientX,
                clientY: e.clientY,
                pageX: e.pageX,
                pageY: e.pageY
            };

            if (!w.$placeholder) {
                this._renderPlaceholder();
            }
        },

        /**
         * Reset active widget and its style,
         * hide placeholder
         **/
        _onDrop: function() {
            var w = this.movingWidget;

            if (w.$el) {
                w.$el.removeAttr('style');
                w.$el = null;
            }

            if (w.$placeholder) {
                w.$placeholder.hide();
            }

            w.block = null;
            this.scrollStart = 0;
        },

        _onMove: function(e) {
            this._move({
                clientX: e.clientX,
                clientY: e.clientY,
                pageX: e.pageX,
                pageY: e.pageY
            });
        },

        /**
         * Move active widget and placeholder,
         * scroll page if needed
         *
         * @param {event|Object} coord
         * @param {Number} coord.clientX
         * @param {Number} coord.clientY
         * @param {Number} coord.pageX
         * @param {Number} coord.pageY
         **/
        _move: function(coord) {
            var basePosition; // position of the element (placeholder atm)

            if (this.movingWidget.$el !== null && !this.isMoving) {
                this.isMoving = true;
                basePosition = this.movingWidget.position;

                if (!this.shouldAddDelay ||
                    (Math.abs(coord.pageX - basePosition.pageX) > 5 || Math.abs(coord.pageY - basePosition.pageY) > 5))
                {
                    this.shouldAddDelay = false;
                    this._scroll(coord);
                    this.movingWidget.$el.css(this._genCss(coord));
                    this._moveActiveWidget(coord);
                }

                this.isMoving = false;
            }
        },

        /**
         * Scroll page if needed
         *
         * @param {Object} coord
         * @param {Number} coord.clientX
         * @param {Number} coord.clientY
         * @param {Number} coord.pageX
         * @param {Number} coord.pageY
         **/
        _scroll: function(coord) {
            var scrollStepPrev = this.scrollStep,
                scrollTop = this.$window.scrollTop(),
                w = this.movingWidget,
                widget = {
                    bottom: coord.clientY + w.block.size.block.height + w.margin.top,
                    top: coord.clientY + w.margin.top
                },
                win = {
                    bottom: this.windowHeight,
                    top: 0
                };

            if (widget.bottom >= win.bottom) {
                this.scrollStep = 1;
            } else if (widget.top <= win.top) {
                this.scrollStep = -1;
            } else {
                this.scrollStep = 0;
            }

            if (this.scrollStep !== 0 && this.scrollStep === scrollStepPrev) {
                if (!this.scrollStart) {
                    // start scroll, one at a time
                    this.scrollStart = (new Date()).getTime();
                    this._scrollWindow(scrollTop);
                }
            } else {
                // stop scroll
                this.scrollStart = 0;
            }
        },

        /**
         * Scroll window
         *
         * @param {number} scrollTop - window.scrollTop of the previous step
         **/
        _scrollWindow: function(scrollTop) {
            var delay = 0.5, // sec
                sec, // seconds passed from this.scrollStart
                stepCoef,
                stepCount = 5; // 1..10

            if (this.scrollStart === 0) {
                return;
            }

            sec = ((new Date()).getTime() - this.scrollStart) / 1000;

            if (sec > delay) {
                stepCoef = Math.ceil(10 / stepCount);
                delay = Math.max(1 / sec * stepCoef, 0.1);
                scrollTop += this.scrollStep;
                this.$window.scrollTop(scrollTop);
            }

            setTimeout($.proxy(this._scrollWindow, this, scrollTop), 1);
        },

        _lockWidgets: function() {
            $.each(this.widgets, function(i, bWidget) {
                bWidget.setMod('locked', 'yes');
            });

        },

        _removeWidgetsLock: function() {
            $.each(this.widgets, function(i, bWidget) {
                bWidget.delMod('locked');
            });
        },

        /**
         * Return css depending on the cursor position
         *
         * @param {Object} coord
         * @param {Number} coord.clientX
         * @param {Number} coord.clientY
         * @param {Number} coord.pageX
         * @param {Number} coord.pageY
         * @return {object}
         **/
        _genCss: function(coord) {
            var bWidget = this.movingWidget.block,
                css = {},
                margin = this.movingWidget.margin,
                size = this.movingWidget.block.size.block;

            size = bWidget.size.block;

            css.left = coord.clientX;
            css.top = coord.clientY;

            css.width = size.width;
            css.height = size.height;

            css.position = 'fixed';
            css.zIndex = '2';

            css.marginLeft = margin.left;
            css.marginTop = margin.top;

            return css;
        },

        /**
         * Render placeholder and active widget to new place depending on the cursor position
         *
         * @param {Object} coord
         * @param {Number} coord.clientX
         * @param {Number} coord.clientY
         * @param {Number} coord.pageX
         * @param {Number} coord.pageY
         **/
        _moveActiveWidget: function(coord) {
            var list = this.widgetIdList,
                newIndex = this._getPosition(coord.pageX, coord.pageY),
                oldIndex,
                w = this.movingWidget;

            oldIndex = w.index;

            this._setPlaceholderParams();
            this._movePlaceholder(newIndex);

            if (oldIndex < newIndex) {
                newIndex = newIndex - 1;
            }

            // widget position changed
            if (newIndex != oldIndex) {
                list.splice(oldIndex, 1);
                list.splice(newIndex, 0, w.block.getId());
                this.movingWidget.index = newIndex;
                this.movingWidget.position = $.extend({}, coord);

                BEM.DOM.after(w.$placeholder, w.$el);
                this._setWidgetsSizes();
                if (this.scrollStart === 0) {
                    this.shouldAddDelay = true;
                }
                this._hasChanges = true;
            }

            w.$placeholder.show();
        },

        _renderPlaceholder: function() {
            this.movingWidget.$placeholder = $(BEMHTML.apply({
                block: 'b-dashboard',
                elem: 'widget-placeholder',
                content: {
                    elem: 'widget-placeholder-content'
                }
            }));
        },

        _movePlaceholder: function(i) {
            var $el,
                length,
                idList = this.widgetIdList,
                w = this.movingWidget;

            length = idList.length;

            if (i < length) {
                $el = this.widgets[idList[i]].domElem;
                BEM.DOM.before($el, w.$placeholder);
            } else {
                BEM.DOM.append(this.elem('widgets'), w.$placeholder);
            }
        },

        _setPlaceholderParams: function() {
            var lockSize,
                size,
                w = this.movingWidget;

            lockSize = w.block.size.lock;
            size = w.block.size.block;

            w.$placeholder.css({
                height: size.height,
                width: size.width
            });

            w.$placeholder.find(this.buildSelector('widget-placeholder-content')).css({
                height: lockSize.height,
                width: lockSize.width
            });
        },

        /**
         * Returns position active widget and placeholder to be placed to
         *
         * @param {number} pageX - pageX of the event
         * @param {number} pageY - pageY of the event
         **/
        _getPosition: function(pageX, pageY) {
            var _this = this,
                bWidget,
                length = this.widgetIdList.length,
                offset,
                size,
                w = this.movingWidget,
                widgetNext,
                widgetList = [];

            // looking for a line and its widgets
            this.widgetIdList.forEach(function(widgetId, i) {
                bWidget = _this.widgets[widgetId];
                size = bWidget.size.block;
                offset = size.offset;
                if ((pageY > offset.top && pageY <= offset.top + size.height) || // inside line
                    (pageY <= offset.top && i === 0) || // above first line
                    (pageY >= offset.top + size.height && i === length - 1) // we are under last line
                ) {
                    widgetList.push({
                        ind: i,
                        offset: offset,
                        size: size,
                        block: bWidget
                    });
                }
            });

            if (widgetList.length === 1) {
                // if line has only one widget
                widgetNext = widgetList[0];
            } else if (widgetList.length) {
                // if there are more than one widget in line
                widgetList.forEach(function(el) {
                    if (!widgetNext && pageX < el.offset.left + el.size.width) {
                        widgetNext = el;
                    }
                });
            }

            // in case cursor was on the right to the last widget of the line
            if (!widgetNext) {
                widgetNext = this.widgets[this.widgetIdList[length - 1]];
                widgetNext = {
                    ind: length - 1,
                    offset: widgetNext.size.block.offset,
                    size: widgetNext.size.block,
                    block: widgetNext
                };
            }

            // if we need to paste active widget before or after line widget
            if (widgetNext.block !== w.block &&
                widgetNext.size.height - (pageY - widgetNext.offset.top) <
                widgetNext.size.height * (pageX - widgetNext.offset.left) / widgetNext.size.width)
            {
                widgetNext.ind = widgetNext.ind + 1;
            }
            return widgetNext.ind;
        }
    });
})();
