/*global BEM*/
(function (BEM) {
    'use strict';

    var promises = BEM.blocks['i-promises'];
    var ajax = promises.ajax;
    var lodash = require('lodash');
    var intersection = lodash.intersection;
    var keys = lodash.keys;

    function genBlockId(siteId, impId) {
        return 'R-' + siteId + '-' + impId;
    }

    function parsePreset(preset) {
        var groups = /(\d+)-(\d+)/.exec(preset);
        return {
            siteId: groups ? groups[1] : '',
            impId: groups ? groups[2] : ''
        };
    }

    function showError(block, text) {
        if (block.findElem('notification').length === 0) {
            block.domElem.append(BEMHTML.apply({
                block: 'b-form-input',
                elem: 'notification'
            }));
        }
        block.showError(text);
    }

    BEM.DOM.decl('b-dsp-testform', {
        onSetMod: {
            js: function () {
                this.form = this.elem('form');
                this.form.submit(this.onSubmit.bind(this));
                this.presetSelect = this.findBlockOn('preset-select', 'b-form-select');
                this.dspSelector = this.findBlockInside('b-dsp-selector');
                this.impIdInput = this.findBlockInside('input-imp-id', 'b-form-input');
                this.siteIdInput = this.findBlockInside('input-site-id', 'b-form-input');
                this.refererInput = this.findBlockInside('input-site-referer', 'b-form-input');
                this.clearResults();

                this.presetSelect.on('change', this.onPresetSelectChanged, this);
                this.dspSelector.on('change', this.onDspSelectorChanged, this);
            }
        },

        onSubmit: function (e) {
            e.preventDefault();
            var data = this.form.serialize();

            this.clearResults();

            ajax.postJSON(this.params.action, data)
                .then(this.updateErrors.bind(this))
                .then(this.updatePreview.bind(this))
                .then(this.updateResponse.bind(this))
                .always(this.showResults.bind(this));
        },

        onPresetSelectChanged: function () {
            var values = parsePreset(this.presetSelect.val());
            if (values && values.siteId && values.impId) {
                this.siteIdInput.val(values.siteId);
                this.impIdInput.val(values.impId);
                this.refererInput.val('http://partner2.yandex.ru/');
            } else {
                this.impIdInput.val('');
                this.siteIdInput.val('');
                this.refererInput.val('');
            }

        },

        onDspSelectorChanged: function (e, dspData) {
            this.elem('dsp-id').val(dspData.id);
            this.elem('dsp-tag').val(dspData.tag);
        },

        clearResults: function () {
            ['errors', 'request', 'response', 'debug'].forEach(function (elem) {
                this.findBlockInside(elem, 'b-form-input').val('');
                this.elem(elem).hide();
            }, this);

            this.findBlockOn('preview', 'b-pcode-preview').updateData(null);
            this.elem('preview').hide();

            this.findBlocksInside({block: 'b-form-input', modName: 'notification', modVal: 'error'})
                .forEach(function (input) {
                    input.hideNotification();
                });
        },

        showResults: function (promise) {
            var val = promise.valueOf();
            if (val.rtb_host_response_json && val.can_view_rtb_host_response === 1) {
                this.elem('debug').show();
            }

            if ((val.error && typeof val.error === 'string') ||
                (val.error && val.error.form) ||
                (Array.isArray(val.error) && val.error.length > 0)) {
                this.elem('errors').show();
            } else {
                this.elem('preview').show();
            }

            if (val.request) {
                this.elem('request').show();
            }

            if (val.response) {
                this.elem('response').show();
            }
        },

        updatePreview: function (response) {
            var preview = this.findBlockOn('preview', 'b-pcode-preview');
            if (response.rtb_host_response_json) {
                var previewData = JSON.parse(response.rtb_host_response_json);
                preview.blockId = genBlockId(this.siteIdInput.val(), this.impIdInput.val());
                preview.updateData(previewData);
            }
            return response;
        },

        updateErrors: function (response) {
            var error = response.error;
            var errorHtml = '';

            if (typeof error === 'string') {
                errorHtml = error;
            } else if (Array.isArray(error)) {
                errorHtml += error.join('\n');
            } else if (error.form) {
                errorHtml = error.form;
            } else if (error.fields) {
                var fields = this.findBlocksInside('b-form-input').reduce(function (result, block) {
                    var input = block.domElem.find('input');
                    result[input.prop('name')] = block;
                    return result;
                }, {});

                var fieldNames = intersection(keys(fields), keys(error.fields));

                fieldNames.forEach(function (fieldName, i) {
                    var block = fields[fieldName];
                    showError(block, error.fields[fieldName]);

                    if (i === 0) {
                        block.setMod('focused', 'yes');
                    }
                }, this);
            }

            if (errorHtml) {
                this.findBlockInside('errors', 'b-form-input').val(errorHtml);
            }
            return response;
        },

        updateResponse: function (response) {
            if (response.can_view_rtb_host_response === 1) {
                this.findBlockInside('debug', 'b-form-input').val(response.rtb_host_response_json);
            }
            this.findBlockInside('request', 'b-form-input').val(response.request);
            this.findBlockInside('response', 'b-form-input').val(response.response);
            return response;
        }
    });

})(BEM);
