/**
*
* @author Ilya Rezvov <irezvov@yandex-team.ru>
* @date 2012
* @description Module describes block for editing values of entity fields of dictionary type.
*
*/

(function (BEM) {
    'use strict';

    BEM.DOM.decl({block: 'b-field-editor', modName: 'type', modVal: 'dictionary'}, {
        onSetMod: {
            disabled: function (modName, modVal) {
                this.value.setMod(modName, modVal);
                this.operation.setMod(modName, modVal);
                this._disabled = modVal === 'yes';
                if (this._disabled) {
                    this.popup.hide();
                }
            }
        },

        /**
         * @public
         * Get or set value of field editor
         *
         * @param {array|string} v [key: string] Value, if exist, then set value for val
         * @return Value [key: string]
         * @trigger change
         */
        val: function(v) {
            var _this = this;

            if (typeof v !== 'undefined') {
                if (!$.isArray(v)) {
                    v = [v];
                }
                v && v.forEach(function (key) {
                    var target = _this.getOptions().find('input[value="' + key + '"]');
                    if (target.length) {
                        var $checkbox = target.parents('.b-form-checkbox');
                        _this.findBlockOn($checkbox, 'b-form-checkbox').setMod('checked', 'yes');
                    }
                });
                this.updateValue();
                this.trigger('change', this);
                return v;
            } else {
                var res = [];
                this.getOptions().filter('.b-form-checkbox_checked_yes').each(function (i, e) {
                    var $checkbox = $(e);
                    var checkbox = _this.findBlockOn($checkbox, 'b-form-checkbox');
                    checkbox.isChecked() && res.push(checkbox.val());
                });
                return res.length ? res : '';
            }
        },

        /**
         * @protected
         * Lookup value label by key in fields values
         *
         * @param {string} v Key of value
         * @return {string} Label of value
         */
        getLabelByVal: function(v) {
            var l = null;
            $.each(this.f('values'), function() {
                if (this.id == v) {
                    l = this.label;
                    return false;
                }
            });
            return l;
        },

        /**
         * @protected
         * Update values in read-only input
         *
         */
        updateValue: function() {
            var v = this.val();
            var that = this;
            var ls = $.map(v, function(v) {return that.getLabelByVal(v);});
            this.value.val(ls.join(', '));
        },

        /**
        * @public
        * Build controls elements for field
        *
        * @param {object} f Field description
        */
        build: function (f) {
            var elems = this.builder(f),
                bemjson,
                html = '';

            if (this.params.table) {
                var hint = this.params.label;
                if (this.params.hint) {
                    hint = [
                        this.params.label,
                        '&nbsp;',
                        {
                            block: 'b-help-box',
                            content: {
                                elem: 'content-wrapper',
                                mix: [{block: 'b-help-box', elem: 'hint-wrapper'}],
                                content: this.params.hint
                            }
                        }
                    ];
                }

                bemjson = [
                    {
                        block: 'b-field-editor',
                        elem: 'lable-cell',
                        tag: 'td',
                        content: hint
                    },
                    elems.op ? {
                        block: 'b-field-editor',
                        elem: 'op-cell',
                        tag: 'td',
                        content: elems.op
                    } : '',
                    {
                        block: 'b-field-editor',
                        elem: 'value-cell',
                        tag: 'td',
                        attrs: {colspan: (elems.op ? '' : '2')},
                        content: elems.val
                    }
                ];
            } else {
                bemjson = [
                    {
                        block: 'b-field-editor',
                        elem: 'label',
                        content: this.params.label
                    },
                    elems.op,
                    elems.val
                ];
            }

            html = BEMHTML.apply(bemjson);

            this.domElem.html(html);
            BEM.DOM.init(this.domElem, function () {
                this.domElem.find('.b-popupa__content').html(this.buildValues(f.values));
            }, this);

            this.value = this.findBlockOn('value', 'b-form-input');
            this.operation = this.findBlockOn('operation', 'b-form-select');

            /* Код данного мода */

            this.popup = this.findBlockOn('values-list', 'b-popupa');
            this.popup.elem('content').css({
                'max-width': this.elem('value').width() * 2,
                'min-width': this.elem('value').width()
            });
            this.popup.on('outside-click', this.updateValue, this);

            var _this = this;
            this.popup.domElem.on('click', '.b-form-checkbox__checkbox', function () {
                var $this = $(this).parent().parent();
                if (!$this.hasClass('b-form-checkbox_js_inited')) {
                    var checkbox = _this.findBlockOn($this, 'b-form-checkbox');
                    if (checkbox.isChecked()) {
                        checkbox.setMod('checked', 'yes');
                    } else {
                        checkbox.delMod('checked');
                    }
                }
            });

            var that = this;
            this.value.elem('input').attr('readonly', true);
            this.value.bindTo('click', function() {
                if (!that._disabled) {
                    that.popup.show(that.value.elem('input'));
                }
            }, this);
        },

        /**
        * @protected
        * (overload) Build BEMJSON for represent b-field-editor
        * It would be overrided by b-filed-editors of other types
        *
        * @param {object} f Field object
        * @return {BEMJSON}
        */
        builder: function(f) {
            var b = this.__base(f);

            b.val = [b.val, {
                block: 'b-popupa',
                mix: [{block: 'b-field-editor', elem: 'values-list'}],
                mods: {theme: 'ffffff', direction: 'down'},
                content: [
                    {
                        elem: 'content',
                        content: '[[content]]'
                    }
                ]
            }];

            return b;
        },

        /**
         * @protected
         * Build BEMJSON for option of dictionary
         *
         * @param {array} values Array of values in form [{value: string, label: string }]
         * @return {BEMJSON}
         */
        buildValues: function(values) {
            if (values.length === 0) {
                return [BEM.I18N('b-field-editor', 'No available values')];
            }

            var template = BEMHTML.apply({
                block: 'b-form-checkbox',
                mods: {theme: 'grey-m', size: 'm'},
                mix: [{block: 'b-field-editor', elem: 'dict-value'}],
                checkboxAttrs: {id: '[[bid]]', value: '[[id]]'},
                content: {elem: 'label', content: '[[label]]'},
                js: false
            });

            var isRoleDictionary = this.params.field.name === 'role_id';

            if (isRoleDictionary) {
                values = values.sort(function (a, b) {
                    return a.label > b.label ? 1 : -1;
                });
            }

            return values.reduce(function (result, value) {
                return result + template
                    .replace('[[id]]', value.id)
                    .replace('[[label]]', value.label)
                    .replace(/\[\[bid\]\]/g, $.identify());
            }, '');
        },

        getOptions: function () {
            return this.options ||
                (this.options = this.popup.domElem.find('.b-field-editor__dict-value'));
        },

        destruct: function () {
            this.__base.apply(this, arguments);
            this.popup.destruct();
        }
    });
})(BEM);
