/**
*
* @author Ilya Rezvov <irezvov@yandex-team.ru>
* @date 2012
* @description Module describes block for editing values of entity fields.
*
*/

(function () {
    'use strict';
    /**
    * @block b-filter-editor
    *
    * @param {object} field Field for building on initialization
    * @param {array} operations The list of avaible operations for field
    */
    BEM.DOM.decl('b-field-editor', {
        onSetMod: {
            js: function () {

                this.field = this.params.field;
                this.operations = this.params.operations;

                if (this.field) {
                    this.build(this.f());
                }
            }
        },

        // public
        /**
        * @public
        * Get or set value of field editor
        *
        * @param {string} v Value, if exist, then set value for val
        * @return Value
        * @trigger change
        */
        val: function (v) {
            if (typeof v != 'undefined') {
                // set value
                this.value.val(v);
                this.trigger('change', this);
                return v;
            } else {
                // get value
                return this.value.val().replace(/^\s+|\s+$/g, '');
            }
        },

        /**
        * @public
        * Get or set operation
        *
        * @param {string} o If it is getted, set operation
        * @trigger change
        */
        op: function (o) {
            if (typeof o != 'undefined') {
                this.operation.val(o);
                this.trigger('change', this);
                return o;
            } else {
                return this.operation.val();
            }
        },

        /**
        * @public
        * Get description of field condition
        *
        * @return {array} [fieldName, op, value]
        */
        getDescription: function () {
            return [this.f('name'), this.op(), this.val()];
        },

        /**
        * @public
        * Build controls elements for field
        *
        * @param {object} f Field description
        */
        build: function (f) {
            var elems = this.builder(f),
                bemjson,
                html = '';

            if (this.params.table) {
                var hint = this.params.label;
                if (this.params.hint) {
                    hint = [
                        this.params.label,
                        '&nbsp;',
                        {
                            block: 'b-help-box',
                            content: {
                                elem: 'content-wrapper',
                                mix: [{block: 'b-help-box', elem: 'hint-wrapper'}],
                                content: this.params.hint
                            }
                        }
                    ];
                }

                bemjson = [
                    {
                        block: 'b-field-editor',
                        elem: 'lable-cell',
                        tag: 'td',
                        content: hint
                    },
                    elems.op ? {
                        block: 'b-field-editor',
                        elem: 'op-cell',
                        tag: 'td',
                        content: elems.op
                    } : '',
                    {
                        block: 'b-field-editor',
                        elem: 'value-cell',
                        tag: 'td',
                        attrs: {colspan: (elems.op ? '' : '2')},
                        content: elems.val
                    }
                ];
            } else {
                bemjson = [
                    {
                        block: 'b-field-editor',
                        elem: 'label',
                        content: this.params.label
                    },
                    elems.op,
                    elems.val
                ];
            }

            html = BEMHTML.apply(bemjson);
            this.domElem.html(html);
            BEM.DOM.init(this.domElem);

            this.value = this.findBlockOn('value', 'b-form-input');
            this.operation = this.findBlockOn('operation', 'b-form-select');
        },

        // protected

        /**
        * @protected
        * Shortcut for access to field or it property
        *
        * @param {string} k Name of field's property
        */
        f: function (k) {
            return k ? this.field[k] : this.field;
        },

        /**
        * @protected
        * Build BEMJSON for represent b-field-editor
        * It would be overrided by b-filed-editors of other types
        *
        * @param {object} f Field object
        * @return {op: BEMJSON, val: BEMJSON}
        */
        builder: function (f) {
            var ops = this.makeSelectOptions(this.operations, function (k, v) {
                    if (v.type == f.type) {
                        return [v.code, v.name];
                    }
                }),
                op = this.makeSelect(ops, 'operation'),
                value = this.makeInput('value');
            return {op: op, val: value};
        },

        // BEMJson builders
        /**
        * @protected
        * Build options list for b-form-select
        *
        * @param {object} val Source object
        * @param {function} it Transfrom source value to [k, v] pair
        * @param {object} ctx Context for iterator
        * @return {BEMJSON} options
        */
        makeSelectOptions: function (val, it, ctx) {
            var ops = [];

            if (!it) {
                it = function (k, v) {
                    return [k, v.toString()];
                };
            }

            $.each(val, function (k) {
                var args = it.call(ctx || {}, k, this);
                if (args) {
                    ops.push({
                        elem: 'option',
                        item: 'option',
                        attrs: {value: args[0]},
                        value: args[0],
                        content: args[1]
                    });
                }
            });

            return ops;
        },

        /**
        * @protected
        * Build BEMJSON with block mix for b-form-select
        *
        * @param {array} options List of BEMJSON options
        * @param {string} elem Name of block element
        * @return {BEMJSON} b-form-select
        */
        makeSelect: function (options, elem) {
            var select = {
                block: 'b-form-select',
                mods: {size: 's', theme: 'grey', layout: 'fixed'},
                content: [
                    {
                        block: 'b-form-button',
                        type: 'button',
                        mods: {size: 's', theme: 'grey-s', valign: 'middle'},
                        content: ''
                    },
                    {
                        elem: 'select',
                        attrs: {id: Math.random().toString()},
                        content: options
                    }
                ]
            };

            if (elem) {
                select.mix = [{block: 'b-field-editor', elem: elem}];
            }

            return select;
        },

        /**
        * @protected
        * Build BEMJSON with block mix for b-form-input
        *
        * @param {string} elem Name of block element
        * @return {BEMJSON} b-form-input
        */
        makeInput: function (elem) {
            var inpt = {
                block: 'b-form-input',
                mods: {theme: 'grey', size: 's'},
                content: {elem: 'input', attrs: {id: Math.random().toString()}}
            };

            if (elem) {
                inpt.mix = [{block: 'b-field-editor', elem: elem, mods: {type: 'input'}}];
            }

            return inpt;
        }
    });
})();
